---@class TestUtil
local M = {}

---@diagnostic disable: undefined-field
---@diagnostic disable: need-check-nil

---@param x number
---@param y number
---@param z number
---@return Vector
local function pos(x, y, z)
    return {x = x, y = y, z = z}
end


---Return a safe offset test region near the player.
---@param player ObjectRef
---@return Vector
function M.get_test_region(player)
    local p = assert(player:get_pos(), "player position required")
    local p_round = assert(vector.round(p), "player position required")
    local base = octchunk.snap_to_center(p_round)
    -- offset far enough to avoid touching the player's immediate area
    return vector.add(base, pos(0, 256, 0))
end




---Run a function on a voxel region and restore it after.
---@param pmin Vector
---@param pmax Vector
---@param fn fun(manip: VoxelManip, area: VoxelArea, data: integer[], param2_data: integer[])
function M.with_voxel_region(pmin, pmax, fn)
    core.load_area(pmin, pmax)
    local manip = core.get_voxel_manip()
    local e1, e2 = manip:read_from_map(pmin, pmax)
    local area = VoxelArea(e1, e2)
    local data = {}
    local param2_data = {}
    manip:get_data(data)
    manip:get_param2_data(param2_data)

    -- snapshot for restore
    local original = {}
    for i = 1, #data do
        original[i] = data[i]
    end

    local original_param2 = {}
    for i = 1, #param2_data do
        original_param2[i] = param2_data[i]
    end


    local ok, err = pcall(fn, manip, area, data, param2_data)

    -- restore
    manip:set_data(original)
    manip:set_param2_data(original_param2)
    manip:write_to_map(false)

    if not ok then
        error(err)
    end
end



---Run a function on a voxel region and provide manual restore for async tests.
---@param pmin Vector
---@param pmax Vector
---@param fn fun(manip: VoxelManip, area: VoxelArea, data: integer[], param2_data: integer[], restore: function)
function M.with_voxel_region_async(pmin, pmax, fn)
    core.load_area(pmin, pmax)
    local manip = core.get_voxel_manip()
    local e1, e2 = manip:read_from_map(pmin, pmax)
    local area = VoxelArea(e1, e2)
    local data = {}
    local param2_data = {}
    manip:get_data(data)
    manip:get_param2_data(param2_data)

    local original = {}
    for i = 1, #data do
        original[i] = data[i]
    end

    local original_param2 = {}
    for i = 1, #param2_data do
        original_param2[i] = param2_data[i]
    end

    local function restore()
        manip:set_data(original)
        manip:set_param2_data(original_param2)
        manip:write_to_map(false)
    end

    fn(manip, area, data, param2_data, restore)
end




---Fill a voxel area buffer with content IDs from a generator.
---@param area VoxelArea
---@param data integer[]
---@param pmin Vector
---@param pmax Vector
---@param content_id_fn fun(x: number, y: number, z: number): integer
function M.fill_region(area, data, pmin, pmax, content_id_fn)
    for x = pmin.x, pmax.x do
        for y = pmin.y, pmax.y do
            for z = pmin.z, pmax.z do
                local idx = area:index(x, y, z)
                data[idx] = content_id_fn(x, y, z)
            end
        end
    end
end



---Count a content_id within a snapshot.
---@param pmin Vector
---@param pmax Vector
---@param target_cid integer
---@param size integer
---@return integer count
function M.count_cid_in_snapshot(pmin, pmax, target_cid, size)
    local map = octmap.new(pmin, pmax, {
        store_chunk_blobs = true,
        force_batches = true,
        max_voxelmanip_volume = (size + 1) ^ 3 * 2,
    })

    local count = 0
    map:for_each_node_cid(pmin, pmax, function(_, _, _, cid)
        if cid == target_cid then
            count = count + 1
        end
    end)
    return count
end



return M
