# Architecture

## The Core Concept: Octchunks and Trees

The fundamental challenge this library solves is handling massive world regions without exhausting server memory. It achieves this by abandoning giant 3D arrays in favor of a **grid of octrees**.

The world is divided into **octchunks**: cubic regions of 16x16x16 nodes. Each octchunk is represented as an **octree**—a hierarchical structure that recursively subdivides space. If a large contiguous area consists of the same material (like a massive volume of underground stone or empty air), the octree collapses that entire branch into a single leaf. Instead of storing 4,096 individual nodes for a uniform chunk, the system stores just one.

A "snapshot" is simply a 3D grid of these trees. This allows the system to be **spatially adaptive**: regions with complex building details expand into deep, precise hierarchies, while empty or solid regions remain computationally "flat".

## Data Flow: Smart Capturing

Capturing a region into a snapshot requires moving data from Luanti's map to our octree structure. To prevent the "RAM spike" common when reading large areas, the library uses a **batching planner**. 

Instead of reading the entire selection at once, the system calculates a series of smaller sub-regions (batches) that fit within a configurable memory budget (`max_voxelmanip_volume`). 
1. The main thread loads and reads a batch from the map.
2. The data is handed to octree builders (either synchronously or via **async workers**).
3. Workers compress each 16x16x16 tree into a **zstd blob**.
4. Only the compressed blobs are stored in the final snapshot matrix.

By serializing and compressing trees immediately, the library can capture world regions far larger than the server's available RAM.

## Working with Snapshots: Lazy Loading and Deltas

Snapshots are designed for high-performance reads and sparse, efficient writes.

### Efficient Reading
When you query a node, the system finds the corresponding tree blob in the grid. To avoid decompressing everything, it uses **lazy decompression**: it only expands a tree when a node inside it is actually requested. These expanded trees are then kept in an **LRU cache**; when the cache fills up, the least recently used trees are discarded, ensuring memory usage stays strictly bounded.

### Sparse Writing
Modifying a snapshot doesn't touch the live map immediately. Instead, changes are recorded as **deltas**—sparse tiny records of what changed where. 
- **`set_node_at`**: Marks a tree as "dirty" and stores the new node data in a local delta table.
- **`write_to_map`**: This is the commit phase. To avoid "MapEditEvent spam" (which causes client lag), the system groups dirty trees by spatial proximity into 80x80x80 batches. Each batch is written to the live world in a single VoxelManip operation.

## Engine Integration and Memory Recovery

Beyond Lua's garbage collector, the library actively manages Luanti's internal C++ memory. 
- **Immediate Disposal**: It uses `manip:close()` (available in Luanti 5.13+) to explicitly signal the engine to free large VoxelManip buffers immediately after use.
- **Aggressive Unloading**: If enabled in the settings, during heavy operations like saving a large snapshot, the mod temporarily forces the engine's mapblock cache to zero (`server_unload_unused_data_timeout = 0`). This pushes the engine to evict mapblocks from RAM as soon as they are no longer needed, preventing the "invisible" memory growth that often crashes servers during large map operations. It causes mapblocks to freeze as long as the player doesn't interact with them during the operation, that's why it's disabled by default. Only use it on very low-memory servers.

## Restore Tracking: Surgical Efficiency

Replacing a 500x500x500 region every time a single block changes is inefficient. Restore tracking solves this by detecting exactly which parts of the world differ from the snapshot.

### Detection Pipeline
1. **Event Hook**: The mod registers `on_mapblocks_changed`. When any block is modified, it identifies which 16x16x16 trees in our grid are affected.
2. **Deduplication and Queueing**: Affected tree keys are added to a **deduplication set** (to avoid redundant work if the same block is hit many times) and a **FIFO verify queue**.
3. **Budgeted Background Scan**: A globalstep loop processes the queue. To avoid lagging the server, it uses a micro-budget (default 5ms per step). It compares the snapshot tree against the current world state.
4. **Surgical Restore**: Only trees found to be actually different (dirty) are stored. When `map:schedule_restore()` is called, only those specific 16x16x16 volumes are written back to the world.
