local ID = octchunk.ID
local CHILDREN = octchunk.CHILDREN

--#region types
--#endregion



---Check if a tree is uniform (single leaf, no children).
---@param tree OctNode
---@return boolean
local function is_uniform_tree(tree)
	return tree[CHILDREN] == nil
end

---Get a uniform tree's content_id (root ID).
---@param tree OctNode
---@return integer
local function get_uniform_content_id(tree)
	return tree[ID]
end


---Sparsify a map by detecting uniform trees and replacing them with nil.
---@param map OctMap The map to sparsify (modified in place)
---@return number sparse_count Number of trees replaced with nil
function octmap.sparsify(map)
	-- Count uniform trees by content_id
	local uniform_counts = {}
	matrix3d.iterate(map.trees, function(x, y, z)
		local cell = matrix3d.get(map.trees, x, y, z)
		if cell == nil then return end -- already sparse
		local tree = type(cell) == "string" and octchunk.deserialize(cell, {use_cache = false}) or cell
		if is_uniform_tree(tree) then
			local cid = get_uniform_content_id(tree)
			uniform_counts[cid] = (uniform_counts[cid] or 0) + 1
		end
	end)

	-- Find most common uniform content_id
	local best_cid, best_count = nil, 0
	for cid, count in pairs(uniform_counts) do
		if count > best_count then
			best_cid, best_count = cid, count
		end
	end

	if not best_cid then
		return 0
	end

	map.default_node = map.content_id_map[best_cid]
	map.default_content_id = best_cid

	-- Replace uniform trees matching default with nil
	local sparse_count = 0
	matrix3d.iterate(map.trees, function(x, y, z)
		local cell = matrix3d.get(map.trees, x, y, z)
		if cell == nil then return end
		local tree = type(cell) == "string" and octchunk.deserialize(cell, {use_cache = false}) or cell
		if is_uniform_tree(tree) and get_uniform_content_id(tree) == best_cid then
			if type(cell) ~= "string" then
				octcache.release(cell)
			end
			matrix3d.set(map.trees, x, y, z, nil)
			sparse_count = sparse_count + 1
		end
	end)

	local total = map.trees.size.x * map.trees.size.y * map.trees.size.z
	map_octree.debug(string.format("Sparsified: %d/%d trees (default=%s)", sparse_count, total, map.default_node))
	return sparse_count
end



---Sparsify using pre-collected uniform counts and positions.
---Called by store_map after async workers have already identified uniform trees.
---@param map OctMap
---@param uniform_counts table<integer, integer> Content ID to count
---@param uniform_positions table<integer, { [1]: integer, [2]: integer, [3]: integer, [4]: integer }> Packed positions {gx, gy, gz, content_id}
---@return integer sparse_count
function octmap.sparsify_with_positions(map, uniform_counts, uniform_positions)
	-- Find most common uniform content_id
	local best_cid, best_count = nil, 0
	for cid, count in pairs(uniform_counts) do
		if count > best_count then
			best_cid, best_count = cid, count
		end
	end

	if not best_cid then
		return 0
	end

	map.default_content_id = best_cid
	map.default_node = (map.content_id_map and map.content_id_map[best_cid]) or map.default_node or "air"

	-- Replace uniform trees matching default with nil
	local sparse_count = 0
	for _, pos in ipairs(uniform_positions) do
		if pos[4] == best_cid then
			local cell = matrix3d.get(map.trees, pos[1], pos[2], pos[3])
			if cell ~= nil and type(cell) ~= "string" then
				octcache.release(cell)
			end
			matrix3d.set(map.trees, pos[1], pos[2], pos[3], nil)
			sparse_count = sparse_count + 1
		end
	end

	local total = map.trees.size.x * map.trees.size.y * map.trees.size.z
	map_octree.debug(string.format("Sparsified: %d/%d trees (default=%s)", sparse_count, total, map.default_node))
	return sparse_count
end
