local modpath = core.get_modpath("map_octree")

map_octree.tests = map_octree.tests or {}

map_octree.tests._async_pending = map_octree.tests._async_pending or 0

local registry = {}

local function register(name, fn)
    assert(type(name) == "string" and name ~= "", "test name must be a non-empty string")
    assert(type(fn) == "function", "test must be a function")
    if registry[name] then
        error("Duplicate test name: " .. name)
    end
    registry[name] = fn
end

local function load_cases()
    if map_octree.tests._loaded then
        return
    end
    map_octree.tests._loaded = true

    local cases_path = modpath .. "/src/tests/cases"
    local files = core.get_dir_list(cases_path, false)
    table.sort(files)

    for _, fname in ipairs(files) do
        if fname:sub(-4) == ".lua" and not fname:match("^%d+_") then
            dofile(cases_path .. "/" .. fname)
        end
    end
end

local function run_all(ctx)
    load_cases()

    local names = {}
    for name in pairs(registry) do
        names[#names + 1] = name
    end
    table.sort(names)

    local passed = 0
    local failed = 0
    local failures = {}

    local t0 = core.get_us_time()
    for _, name in ipairs(names) do
        local fn = registry[name]
        local ok, err = pcall(fn, ctx)
        if ok then
            passed = passed + 1
        else
            failed = failed + 1
            failures[#failures + 1] = string.format("- %s: %s", name, tostring(err))
        end
    end
    local t1 = core.get_us_time()

    return {
        passed = passed,
        failed = failed,
        total = #names,
        ms = (t1 - t0) / 1000,
        failures = failures,
    }
end

map_octree.tests.register = register
map_octree.tests.run_all = run_all


map_octree.tests.async_pending = function()
    return map_octree.tests._async_pending or 0
end


map_octree.tests.async_should_skip = function(label)
    local pending = map_octree.tests._async_pending or 0
    if pending > 0 then
        core.log("warning", string.format("[test] skipping %s: async pending=%d", tostring(label), pending))
        return true
    end
    return false
end


map_octree.tests.async_begin = function(label)
    map_octree.tests._async_pending = (map_octree.tests._async_pending or 0) + 1
    core.log("action", string.format("[test] async begin: %s (pending=%d)", tostring(label), map_octree.tests._async_pending))
end


map_octree.tests.async_end = function(label)
    local pending = map_octree.tests._async_pending or 0
    if pending > 0 then
        pending = pending - 1
    end
    map_octree.tests._async_pending = pending
    core.log("action", string.format("[test] async end: %s (pending=%d)", tostring(label), pending))
    if pending == 0 then
        core.log("action", "[test] all async tests completed")
    end
end


map_octree.tests.async_start = function(label)
    map_octree.tests.async_begin(label)
    local done = false
    return function()
        if done then
            return
        end
        done = true
        map_octree.tests.async_end(label)
    end
end
