-- Async tests cannot block - results are logged via core.after polling
local async_test_results = {}

map_octree.tests.register("async save: basic correctness", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("async save: basic") then
		return
	end


	local pos = vector.round(ctx.player:get_pos())
	local center = octchunk.snap_to_center(pos)
	local test_id = "async_basic_" .. os.time()

	async_test_results[test_id] = {status = "pending", start = core.get_us_time()}
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	map_octree.save_to_file_async(center, center, {
		file_name = test_id,
		subdir = "tests",
		max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		async_inflight = 1,
	}, function(ok, result)
		local res = async_test_results[test_id]
		if not ok then
			res.status = "FAIL"
			res.error = "async save failed: " .. tostring(result)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local map = result

		if not map or not map.trees then
			res.status = "FAIL"
			res.error = "map or map.trees is nil"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local first_tree = matrix3d.get(map.trees, 1, 1, 1)
		-- After sparsify, uniform trees become nil (sparse)
		if first_tree ~= nil and type(first_tree) ~= "string" then
			res.status = "FAIL"
			res.error = "expected blob (string or nil), got " .. type(first_tree)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local test_pos = center
		local got_name = octmap.get_node_name(map, test_pos)
		if not got_name then
			res.status = "FAIL"
			res.error = string.format("query failed at %s", core.pos_to_string(test_pos))
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		-- Compare with direct VoxelManip read
		local half = octchunk.SIZE / 2
		local min_truth = vector.subtract(center, {x = half, y = half, z = half})
		local max_truth = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})
		core.load_area(min_truth, max_truth)
		local manip = core.get_voxel_manip()
		local e1, e2 = manip:read_from_map(min_truth, max_truth)
		local area = VoxelArea(e1, e2)
		local data = {}
		local param2_data = {}
		manip:get_data(data)
		manip:get_param2_data(param2_data)

		local checked = 0
		for _ = 1, 100 do
			local rx = min_truth.x + math.random(0, octchunk.SIZE - 1)
			local ry = min_truth.y + math.random(0, octchunk.SIZE - 1)
			local rz = min_truth.z + math.random(0, octchunk.SIZE - 1)
			local idx = area:index(rx, ry, rz)
			local expected_name = core.get_name_from_content_id(data[idx])
			local expected_p2 = param2_data[idx] or 0

			local got_name2, got_p2 = map:get_node_at(rx, ry, rz)
			if not got_name2 then
				res.status = "FAIL"
				res.error = string.format("missing node at %d,%d,%d", rx, ry, rz)
				core.log("error", "[test] " .. test_id .. ": " .. res.error)
				if end_async then end_async() end
				return
			end

			if expected_name ~= got_name2 then
				res.status = "FAIL"
				res.error = string.format(
					"mismatch at %d,%d,%d: expected '%s', got '%s'",
					rx, ry, rz, expected_name, got_name2
				)
				core.log("error", "[test] " .. test_id .. ": " .. res.error)
				if end_async then end_async() end
				return
			end
			if expected_p2 ~= got_p2 then
				res.status = "FAIL"
				res.error = string.format(
					"param2 mismatch at %d,%d,%d: expected %d, got %d",
					rx, ry, rz, expected_p2, tonumber(got_p2) or -1
				)
				core.log("error", "[test] " .. test_id .. ": " .. res.error)
				if end_async then end_async() end
				return
			end
			checked = checked + 1
		end

		res.status = "PASS"
		res.checked = checked
		local elapsed_ms = (core.get_us_time() - res.start) / 1000
		core.log("action", string.format("[test] %s: PASS - verified %d positions in %.1fms",
			test_id, checked, elapsed_ms))
		if end_async then end_async() end
	end)

	core.log("action", "[test] " .. test_id .. ": async operation started (result will be logged)")
end)


map_octree.tests.register("async save: multi-batch with inflight=2", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("async save: multi") then
		return
	end


	local pos = vector.round(ctx.player:get_pos())
	local base_center = octchunk.snap_to_center(pos)
	local S = octchunk.SIZE
	local half = S / 2
	local max_center = vector.add(base_center, {x = S, y = S, z = 0})
	local test_id = "async_multi_" .. os.time()

	async_test_results[test_id] = {status = "pending", start = core.get_us_time()}
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	local limit_one_tree = (S + 1) ^ 3 + 1024

	map_octree.save_to_file_async(base_center, max_center, {
		file_name = test_id,
		subdir = "tests",
		max_voxelmanip_volume = limit_one_tree,
		async_inflight = 2,
	}, function(ok, result)
		local res = async_test_results[test_id]
		if not ok then
			res.status = "FAIL"
			res.error = "async save failed: " .. tostring(result)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local map = result

		if map.trees.size.x ~= 2 or map.trees.size.y ~= 2 or map.trees.size.z ~= 1 then
			res.status = "FAIL"
			res.error = string.format("expected 2x2x1 trees, got %dx%dx%d",
				map.trees.size.x, map.trees.size.y, map.trees.size.z)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		for x = 1, 2 do
			for y = 1, 2 do
				local tree = matrix3d.get(map.trees, x, y, 1)
				-- After sparsify, uniform trees become nil (sparse)
				if tree ~= nil and type(tree) ~= "string" then
					res.status = "FAIL"
					res.error = string.format("tree at %d,%d,1 is %s, expected string or nil", x, y, type(tree))
					core.log("error", "[test] " .. test_id .. ": " .. res.error)
					if end_async then end_async() end
					return
				end
			end
		end

		local min_truth = vector.subtract(base_center, {x = half, y = half, z = half})
		local max_truth = vector.add(max_center, {x = half - 1, y = half - 1, z = half - 1})
		core.load_area(min_truth, max_truth)
		local manip = core.get_voxel_manip()
		local e1, e2 = manip:read_from_map(min_truth, max_truth)
		local area = VoxelArea(e1, e2)
		local data = {}
		local param2_data = {}
		manip:get_data(data)
		manip:get_param2_data(param2_data)

		local checked = 0
		for _ = 1, 200 do
			local rx = min_truth.x + math.random(0, max_truth.x - min_truth.x)
			local ry = min_truth.y + math.random(0, max_truth.y - min_truth.y)
			local rz = min_truth.z + math.random(0, max_truth.z - min_truth.z)
			local idx = area:index(rx, ry, rz)
			local expected_name = core.get_name_from_content_id(data[idx])
			local expected_p2 = param2_data[idx] or 0

			local got_name, got_p2 = map:get_node_at(rx, ry, rz)
			if got_name then
				if expected_name ~= got_name then
					res.status = "FAIL"
					res.error = string.format(
						"mismatch at %d,%d,%d: expected '%s', got '%s'",
						rx, ry, rz, expected_name, got_name
					)
					core.log("error", "[test] " .. test_id .. ": " .. res.error)
					if end_async then end_async() end
					return
				end
				if expected_p2 ~= got_p2 then
					res.status = "FAIL"
					res.error = string.format(
						"param2 mismatch at %d,%d,%d: expected %d, got %d",
						rx, ry, rz, expected_p2, tonumber(got_p2) or -1
					)
					core.log("error", "[test] " .. test_id .. ": " .. res.error)
					if end_async then end_async() end
					return
				end
				checked = checked + 1
			end
		end

		res.status = "PASS"
		res.checked = checked
		local elapsed_ms = (core.get_us_time() - res.start) / 1000
		core.log("action", string.format("[test] %s: PASS - verified %d positions in %.1fms",
			test_id, checked, elapsed_ms))
		if end_async then end_async() end
	end)

	core.log("action", "[test] " .. test_id .. ": async operation started (result will be logged)")
end)