local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

-- Async tests cannot block - results are logged via core.after polling
local async_results = {}


map_octree.tests.register("octree manip: write_to_map sync", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local pos0 = testutil.get_test_region(ctx.player)
	local base = vector.add(octchunk.snap_to_center(pos0), {x = 0, y = 0, z = 1024})
	local pos1 = base
	local pos2 = vector.add(base, {x = octchunk.SIZE, y = 0, z = 0})

	local node_a = "default:stone"
	local node_b = "default:dirt"
	assert(core.get_content_id(node_a), "missing node: " .. node_a)
	assert(core.get_content_id(node_b), "missing node: " .. node_b)

	local half = octchunk.SIZE / 2
	local pA = {x = pos1.x - (half - 1), y = pos1.y, z = pos1.z}
	local pB_center = vector.add(pos1, {x = octchunk.SIZE, y = 0, z = 0})
	local pB = {x = pB_center.x - (half - 1), y = pB_center.y, z = pB_center.z}
	local pmin = {x = pos1.x - half, y = pos1.y - half, z = pos1.z - half}
	local pmax = {x = pos2.x + (half - 1), y = pos2.y + (half - 1), z = pos2.z + (half - 1)}

	testutil.with_voxel_region(pmin, pmax, function()
		local m = map_octree.new_octree_manip()
		m:read_from_map(pos1, pos2)

		assert(m:set_node_at(pA.x, pA.y, pA.z, node_a, 3), "set_node_at A failed")
		assert(m:set_node_at(pB.x, pB.y, pB.z, node_b, 4), "set_node_at B failed")
		assert(m:write_to_map(), "write_to_map failed")

		local nA = core.get_node(pA)
		assert(nA and nA.name == node_a, "live mismatch A")
		assert((nA.param2 or 0) == 3, "live param2 mismatch A")

		local nB = core.get_node(pB)
		assert(nB and nB.name == node_b, "live mismatch B")
		assert((nB.param2 or 0) == 4, "live param2 mismatch B")

		local sA, sp2A = m:get_node_at(pA.x, pA.y, pA.z)
		assert(sA == node_a, "snapshot mismatch A")
		assert((sp2A or 0) == 3, "snapshot param2 mismatch A")

		local sB, sp2B = m:get_node_at(pB.x, pB.y, pB.z)
		assert(sB == node_b, "snapshot mismatch B")
		assert((sp2B or 0) == 4, "snapshot param2 mismatch B")
	end)
end)


map_octree.tests.register("octree manip: write_to_map_async", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("octree manip async") then
		return
	end

	local pos0 = testutil.get_test_region(ctx.player)
	local base = vector.add(octchunk.snap_to_center(pos0), {x = 0, y = 0, z = 2048})
	local pos1 = base
	local pos2 = vector.add(base, {x = octchunk.SIZE, y = 0, z = 0})

	local node_a = "default:cobble"
	local node_b = "default:sand"
	assert(core.get_content_id(node_a), "missing node: " .. node_a)
	assert(core.get_content_id(node_b), "missing node: " .. node_b)

	local half = octchunk.SIZE / 2
	local pA = {x = pos1.x - (half - 2), y = pos1.y + 1, z = pos1.z}
	local pB_center = vector.add(pos1, {x = octchunk.SIZE, y = 0, z = 0})
	local pB = {x = pB_center.x - (half - 2), y = pB_center.y + 1, z = pB_center.z}
	local pmin = {x = pos1.x - half, y = pos1.y - half, z = pos1.z - half}
	local pmax = {x = pos2.x + (half - 1), y = pos2.y + (half - 1), z = pos2.z + (half - 1)}

	local test_id = "octree_manip_async_" .. os.time()
	async_results[test_id] = {status = "pending", start = core.get_us_time()}
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	local m = map_octree.new_octree_manip()
	m:read_from_map(pos1, pos2)

	assert(m:set_node_at(pA.x, pA.y, pA.z, node_a, 1), "set_node_at A failed")
	assert(m:set_node_at(pB.x, pB.y, pB.z, node_b, 2), "set_node_at B failed")

	m:write_to_map_async(function(ok, err)
		local res = async_results[test_id]
		if not ok then
			res.status = "FAIL"
			res.error = tostring(err)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		-- Verify live map via VoxelManip readback (more reliable than core.get_node for unloaded blocks)
		core.load_area(pmin, pmax)
		local manip = core.get_voxel_manip()
		local e1, e2 = manip:read_from_map(pmin, pmax)
		local area = VoxelArea(e1, e2)
		local data = {}
		local param2_data = {}
		manip:get_data(data)
		manip:get_param2_data(param2_data)
		manip:close()

		local cid_a = core.get_content_id(node_a)
		local cid_b = core.get_content_id(node_b)
		local idxA = area:index(pA.x, pA.y, pA.z)
		local idxB = area:index(pB.x, pB.y, pB.z)
		local got_cid_a = data[idxA]
		local got_p2_a = (param2_data[idxA] or 0)
		local got_cid_b = data[idxB]
		local got_p2_b = (param2_data[idxB] or 0)

		if not (got_cid_a == cid_a and got_p2_a == 1) then
			res.status = "FAIL"
			res.error = "live mismatch A"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end
		if not (got_cid_b == cid_b and got_p2_b == 2) then
			res.status = "FAIL"
			res.error = "live mismatch B"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local sA, sp2A = m:get_node_at(pA.x, pA.y, pA.z)
		local sB, sp2B = m:get_node_at(pB.x, pB.y, pB.z)
		if not (sA == node_a and (sp2A or 0) == 1) then
			res.status = "FAIL"
			res.error = "snapshot mismatch A"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end
		if not (sB == node_b and (sp2B or 0) == 2) then
			res.status = "FAIL"
			res.error = "snapshot mismatch B"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		res.status = "PASS"
		local elapsed_ms = (core.get_us_time() - res.start) / 1000
		core.log("action", string.format("[test] %s: PASS - %.1fms", test_id, elapsed_ms))
		if end_async then end_async() end
	end)

	core.log("action", "[test] " .. test_id .. ": async operation started (result will be logged)")
end)