# map_octree — API Documentation

**Note:** This codebase is fully LuaCATsizzato.

This library provides a high-performance, sparse 3D data structure (Octree) for capturing, storing, and modifying Luanti world data.

If you are looking for internal design details, see [ARCHITECTURE.md](ARCHITECTURE.md).

---

## Index

- [Core Concepts](#core-concepts)
- [Quick Start](#quick-start)
- [Global API (map_octree)](#global-api-map_octree)
- [The Map Object (Read-Only)](#the-map-object-read-only)
- [OctreeManip (Read/Write)](#octreemanip-readwrite)
- [Examples](#examples-all-api)
- [Advanced Options](#advanced-options)
- [Restore Tracking (Surgical Restore)](#restore-tracking-surgical-restore)
- [Performance Tips](#performance-tips)
- [Debugging](#debugging)

## Core Concepts

### The Grid
The world is partitioned into "Octchunks" of `16x16x16` nodes. A **Snapshot** is a 3D grid of these chunks, managed as a sparse structure. 

### Boundary Behavior & Non-Exact Perimeter
**Warning**: The library operates on full `16x16x16` chunks. 
- When you save or track an area, the mod **expands the selection** to cover every chunk touched by your coordinates. 
- This means the **perimeter is NOT EXACT**. It can extend up to 16 blocks beyond your requested bounds to stay aligned with the chunk grid.
- **Rule of Thumb**: Avoid saving two different map files for areas that are physically close to each other. Because of the chunk alignment, they will likely overlap and include nodes from the neighboring area.

### Coordinates
All **node-grid** API calls use **absolute world coordinates** (`x, y, z`). You don't need to worry about chunk offsets; the library handles node-to-chunk mapping automatically.

### Snapshots vs. Live Map
- **Map Object**: A read-only snapshot stored in memory or on disk. Great for schematics, area backups, or fast read-only analysis.
- **OctreeManip**: An interactive buffer. You read an area into it, record changes (deltas), and finally "apply" them back to the world.

---

## Quick Start

### 1. Capture and Query
```lua
local p1 = vector.new(0, 0, 0)
local p2 = vector.new(15, 15, 15)

-- Create a snapshot (saved to disk)
local map = map_octree.save_to_file(p1, p2, { file_name = "backup" })

-- Query a node (fast)
local name, param2, param1 = map:get_node_at(5, 2, 5)
print("Found: " .. (name or "nothing") .. " with param1=" .. (param1 or 0))
```

### 2. Stream an Area
To avoid creating massive Lua tables, use the streaming helper:
```lua
map:for_each_node(p1, p2, function(x, y, z, name, param2, param1)
    -- This runs for every node in the box
    if name == "default:dirt" then
        -- do something
    end
end)
```

---

## Global API (`map_octree`)

- **`map_octree.save_to_file(pos1, pos2, opts)`**: Captures a region of the world into a Map Object synchronously and saves it to disk.
  - **Returns**: `map` object on success, or `nil, err` on failure (serialize/compress errors).
  - **Note**: Can cause lag on very large regions (>80x80x80). Use the async version for big areas.
  - `opts`:
    - `file_name` (string, required): Base name of the snapshot file. The resulting file is `<file_name>.bin` under `worldpath/map_octree/<subdir>/`.
    - `subdir` (string, default: ""): Optional subfolder under `worldpath/map_octree/`.
    - `store_chunk_blobs` (boolean, default: true): Store each chunk as a serialized+zstd-compressed octree blob. Disabling this makes loading/saving faster but increases memory usage.
    - `cache_mb` (number, default: 5% of `map_octree_max_total_ram_budget_mb`): Cache budget in MiB, clamped by server limits.
    - `force_batches` (boolean, advanced, default: false): Force batch planning even if one VoxelManip would fit.
    - `max_voxelmanip_volume` (integer, advanced): Max VoxelManip volume per batch.
- **`map_octree.save_to_file_async(pos1, pos2, opts, callback)`**: The preferred way to capture large regions. It processes the world in background batches.
  - `opts`:
    - All options from `save_to_file` PLUS:
    - `async_inflight` (integer, advanced, default: max server capacity): Max async batch jobs in flight.
    - `flush_async_workers` (boolean, advanced, default: true): Flush async worker heaps after save.
    - `read_budget_ms` (number, advanced, default: 50): Time budget in milliseconds per server tick for reading world data.
  - **Callback**: `function(ok, result, info)`
    - `ok`: boolean, true if success.
    - `result`: the `map` object if success, or a `string` error message if failed.
    - `info`: table with technical stats (like `batch_count`).
- **`map_octree.load_map(file_name, opts)`**: Loads a previously saved snapshot from disk.
  - `file_name` (string): Base name of the snapshot file (without directory or `.bin` extension).
  - `opts`:
    - `subdir` (string): Optional subfolder under `worldpath/map_octree/`.
    - `key` (string, advanced): Registry key override (default: `<subdir>/<file_name>` or just `<file_name>`).
  - **Returns**: `map` object or `nil` if file not found or invalid format.
- **`map_octree.get_loaded_map(key)`**: Returns a loaded map from the registry.
  - `key` (string): Registry key.
  - **Returns**: `map` object or `nil` if missing.
- **`map_octree.list_loaded_maps()`**: Returns a sorted list of loaded map keys.
- **`map_octree.unload_map(key)`**: Removes a map from the registry and disables tracking (with flush) before unloading.
  - `key` (string): Registry key.
  - **Returns**: `true` on success, or `false, err` if the key was not loaded.
- **`map_octree.place(map, pos1, opts)`**: Pastes a snapshot back into the live world synchronously.
  - `map`: Optional. If nil, uses the last loaded map (`map_octree.current_map`).
  - `pos1`: Optional. The target coordinates (min corner). Defaults to original capture position.
  - `opts`:
    - `force_batches` (boolean, default: false): Force batch planning.
    - `max_voxelmanip_volume` (integer): Max VoxelManip volume per batch.
    - `cache_mb` (number): Cache budget used during batch planning.
- **`map_octree.place_async(map, pos1, opts, callback)`**: Same as `place`, but spreads the work over multiple server ticks.
  - `opts`:
    - All options from `place` PLUS:
    - `min_delay` (number, default: 0): Minimum delay in seconds between batches (server ticks).
  - **Callback**: `function(ok, err)`
    - `ok`: boolean, true if placement completed.
    - `err`: string error message if failed.
- **`map_octree.save_around_player(player, radius, height, opts)`**: Utility to capture a region around a player.
  - `player`: Player object.
  - `radius` (number, default: 100): Horizontal radius in nodes.
  - `height` (number, default: 128): Vertical extent (± from player Y coordinate).
  - `opts`:
    - `file_name` (string, required): Base name for the snapshot.
    - `subdir` (string): Optional subfolder.
    - `store_chunk_blobs` (boolean): Default: true.
    - `cache_mb` (number): Memory budget in MiB.
    - `force_batches` (boolean): Force batch planning.
  - **Returns**: `map` object on success, or `nil, err` on failure.
- **`map_octree.new_octree_manip(opts)`**: Creates a new `OctreeManip` instance (see [OctreeManip](#octreemanip-readwrite)).
  - `opts`:
    - `build_opts` (table): Options for building the internal snapshot memory structure.
      - `store_chunk_blobs` (boolean, default: true): Store compressed chunks.
      - `cache_mb` (number): Memory budget in MiB.
      - `force_batches` (boolean): Force batch planning.
      - `max_voxelmanip_volume` (integer): Max VoxelManip size per batch.
    - `frozen_snapshot` (boolean, default: false, internal): If true, writes do not refresh the internal snapshot. Avoid in normal flows.

---

## The Map Object (Read-Only)

Created via `save_to_file` or `load_map`.

- **`map:get_node_at(x, y, z)`** -> Returns `(name, param2, param1)` or `nil`.
- **`map:get_node_cid_at(x, y, z)`** -> Returns `(content_id, param2, param1)`. Faster for bulk logic as it avoids strings.
- **`map:size()`** -> Returns a vector of the dimensions.
- **`map:get_emerged_area()`** -> Returns `(minp, maxp)` world bounds.
- **`map:for_each_node(pos1, pos2, callback)`** -> Iterates over a region.
  - Callback: `function(x, y, z, name, param2, param1)`
    - `x, y, z`: world coordinates.
    - `name`: node name.
    - `param2`, `param1`: node parameters.
- **`map:for_each_node_cid(pos1, pos2, callback)`** -> Faster version of the above.
  - Callback: `function(x, y, z, cid, param2, param1)`
    - `x, y, z`: world coordinates.
    - `cid`: numeric content ID.
    - `param2`, `param1`: node parameters.
- **`map:read_ranges(ranges, callback, opts)`** -> Streams nodes for selected ranges.
  - Callback: `function(x, y, z, cid, param2, param1)`
  - `ranges`: list of ranges `{ pos1 = vector.new(...), pos2 = vector.new(...) }`.
  - `opts`:
    - `grid` (string, default: "node"): Whether ranges are in node coordinates or chunk indices ("chunk").
    - `dedup` (boolean): If true, ensures each chunk is processed only once (default: true for chunk grid).
- **`map:place(pos1, opts)`** -> Places this snapshot into the live world synchronously.
  - `pos1`: Optional. The target coordinates (min corner). Defaults to original capture position.
  - `opts`:
    - `force_batches` (boolean, default: false): Force batch planning.
    - `max_voxelmanip_volume` (integer): Max VoxelManip volume per batch.
    - `cache_mb` (number): Cache budget used during batch planning.
- **`map:place_async(pos1, opts, callback)`** -> Async version.
  - `opts`:
    - `force_batches` (boolean): Force batch planning.
    - `max_voxelmanip_volume` (integer): Max VoxelManip volume per batch.
    - `min_delay` (number, default: 0): Minimum delay in seconds between batches.
  - **Callback**: `function(ok, err)`

---

## OctreeManip (Read/Write)

The `OctreeManip` is used for "Edit" workflows: read from map -> modify in Lua -> write back.

### Initialization
```lua
local m = map_octree.new_octree_manip()
```

### Loading Data
- **`m:read_from_map(p1, p2)`**: Synchronous read into an in-memory snapshot. Handy for small areas, but does **not** ensure mapgen/emerge for never-generated regions.
- **`m:read_from_map_async(p1, p2, callback)`**: Triggers map generation first and doesn't block the main thread.
  - Callback: `function(ok, err)`
    - `ok`: boolean, true if read completed.
    - `err`: string error message if failed.

### Querying Data
These APIs mirror the Map Object querying functions. Reads come from the loaded snapshot, with any pending deltas overlaid (so staged edits are visible).
- **`m:get_emerged_area()`**: Returns `(minp, maxp)` bounds of the loaded snapshot.
- **`m:size()`**: Returns a vector of snapshot dimensions.
- **`m:get_node_at(x, y, z)`**: Returns `(name, param2, param1)` from the snapshot.
- **`m:for_each_node(pos1, pos2, callback)`**: Streams nodes from the snapshot.
  - Callback: `function(x, y, z, name, param2, param1)`
    - `x, y, z`: world coordinates.
    - `name`: node name.
    - `param2`, `param1`: node parameters.
- **`m:read_ranges(ranges, callback, opts)`**: Streams nodes for selected ranges.
  - Callback: `function(x, y, z, cid, param2, param1)`
  - `ranges`: list of ranges `{ pos1 = vector.new(...), pos2 = vector.new(...) }`.
  - `opts`:
    - `grid` (string, default: "node"): coordinates mode ("node" or "chunk").
    - `dedup` (boolean): avoid double-processing chunks.

### Modifying Data
- **`m:set_node_at(x, y, z, name, param2, param1)`**: Records a change at coordinates.
  - `param2` is optional (defaults to `0`). Values are clamped to `0..255`.
  - `param1` is optional. If omitted, the effective `param1` is taken from the snapshot (and any pending `param1` override for that node is cleared).
  - **Returns** `false` if the position is outside the loaded snapshot bounds; otherwise `true`.
- **`m:set_node_cid_at(x, y, z, cid, param2, param1)`**: Same as above, but takes `content_id` directly (faster for bulk edits).
- **`m:apply(p1, p2, func)`**: Bulk transform helper. Scans the snapshot and records deltas.
  - Callback: `func(x, y, z, cid, param2, param1)` -> return `(new_cid, new_param2, new_param1)` to change, or `nil` to keep original.
  - You can omit `new_param2` and `new_param1` in the return (both default to their original values from the snapshot).
  - Does NOT commit; call `write_to_map()` or `write_to_map_async()` after.
  - **Returns**: count of changed nodes.
- **`m:apply_async(p1, p2, func, opts, callback)`**: Async version of `apply()`. Time-budgeted per step.
  - `opts`: Currently reserved for future use.
  - **Callback**: `function(ok, changed_count, err)`
    - `ok`: boolean, true if apply completed.
    - `changed_count`: number of changed nodes.
    - `err`: string error message if failed.
  - Does NOT commit; call `write_to_map()` or `write_to_map_async()` after.
  - Time budget is controlled by setting `map_octree_write_budget_ms` (default 50ms).
- **`m:apply_ranges(ranges, func, opts)`**: Bulk transform over selected ranges.
  - `ranges`: list of ranges `{ pos1 = vector.new(...), pos2 = vector.new(...) }`.
  - `opts`:
    - `grid` (string, default: "node"): coordinates mode ("node" or "chunk").
    - `dedup` (boolean): avoid double-processing chunks.
  - Does NOT commit; call `write_to_map()` or `write_to_map_async()` after.
  - **Returns**: count of changed nodes.
- **`m:apply_ranges_async(ranges, func, opts, callback)`**: Async version of `apply_ranges()`.
  - `opts`:
    - `grid` (string, default: "node"): coordinates mode ("node" or "chunk").
    - `dedup` (boolean): avoid double-processing chunks.
  - **Callback**: `function(ok, changed_count, err)`
    - `ok`: boolean, true if apply completed.
    - `changed_count`: number of changed nodes.
    - `err`: string error message if failed.
  - Does NOT commit; call `write_to_map()` or `write_to_map_async()` after.
- **`m:clear_deltas()`**: Discards all pending changes without writing to map.

### Writing Back
- **`m:write_to_map()`**: Commits all recorded changes to the world instantly. Returns `true` on success.
- **`m:write_to_map_async(callback)`**: Commits changes in batches over time.
  - Callback: `function(ok, err)`
    - `ok`: boolean, true if write completed.
    - `err`: string error message if failed.
  - Time budget is controlled by setting `map_octree_write_budget_ms` (default 50ms).

---

## Examples

### Save two maps on disk and enable tracking
```lua
local p1 = vector.new(0, 0, 0)
local p2 = vector.new(31, 31, 31)
local p3 = vector.new(64, 0, 0)
local p4 = vector.new(95, 31, 31)

local map_a = map_octree.save_to_file(p1, p2, {file_name = "zone_a", subdir = "zones"})
local map_b = map_octree.save_to_file(p3, p4, {file_name = "zone_b", subdir = "zones"})

map_a:enable_tracking()
map_b:enable_tracking()
```

### Load two maps from disk and auto-restore every 60s
```lua
local map_a = map_octree.load_map("zone_a", {subdir = "zones"})
local map_b = map_octree.load_map("zone_b", {subdir = "zones"})

map_a:enable_tracking()
map_b:enable_tracking()

local function restore_loop()
  map_a:schedule_restore(function(ok, err)
    if not ok then
      core.log("error", "restore map_a failed: " .. tostring(err))
    end
  end)
  map_b:schedule_restore(function(ok, err)
    if not ok then
      core.log("error", "restore map_b failed: " .. tostring(err))
    end
  end)
  core.after(60, restore_loop)
end

core.after(60, restore_loop)
```

<details>
<summary>OctreeManip</summary>

**read_ranges (counting solid nodes in a snapshot)**
```lua
-- Read map area
local p1 = vector.new(0, 0, 0)
local p2 = vector.new(47, 31, 47)
local ranges = { {pos1 = p1, pos2 = p2} }

local m = map_octree.new_octree_manip()
m:read_from_map(p1, p2)

-- Count solid nodes
local air_cid = core.get_content_id("air")
local solid = 0
m:read_ranges(ranges, function(x, y, z, cid, param2, param1)
  if cid ~= air_cid then
    solid = solid + 1
  end
end)
core.log("action", "solid nodes in snapshot ranges: " .. solid)
```


**apply_ranges (replacing stone with sand synchronously)**
```lua
-- Read map area
local p1 = vector.new(0, 0, 0)
local p2 = vector.new(31, 31, 31)
local ranges = { {pos1 = p1, pos2 = p2} }

local m = map_octree.new_octree_manip()
m:read_from_map(p1, p2)

-- Define transform function
local stone_cid = core.get_content_id("default:stone")
local sand_cid = core.get_content_id("default:sand")
local function fn(x, y, z, cid, param2, param1)
  if cid == stone_cid then
    return sand_cid, param2, param1
  end
end

-- Apply changes
local changed = m:apply_ranges(ranges, fn)
core.log("action", "apply_ranges changed: " .. changed)

if not m:write_to_map() then
  core.log("error", "write_to_map failed")
  return
end
```


**apply_ranges_async (replacing stone with sand across multiple areas over time)**
```lua
-- Read map area
local p1 = vector.new(0, 0, 0)
local p2 = vector.new(63, 31, 63)
local ranges = {
  {pos1 = vector.new(0, 0, 0), pos2 = vector.new(15, 31, 15)},
  {pos1 = vector.new(32, 0, 32), pos2 = vector.new(47, 31, 47)},
}

local m = map_octree.new_octree_manip()
m:read_from_map(p1, p2)

-- Define transform function
local stone_cid = core.get_content_id("default:stone")
local sand_cid = core.get_content_id("default:sand")
local function fn(x, y, z, cid, param2, param1)
  if cid == stone_cid then
    return sand_cid, param2, param1
  end
end

-- Apply changes asynchronously
m:apply_ranges_async(ranges, fn, nil, function(ok, changed, err)
  if not ok then
    core.log("error", "apply_ranges_async failed: " .. tostring(err))
    return
  end

  m:write_to_map_async(function(wok, werr)
    if not wok then
      core.log("error", "write_to_map_async failed: " .. tostring(werr))
      return
    end
    core.log("action", "apply_ranges_async changed: " .. changed)
  end)
end)
```

</details>

---

## Advanced Options

- `force_batches` (Map creation option): forces batch planning even when the total volume would fit in a single VoxelManip. Useful for testing or for keeping per-batch memory spikes predictable.
- `map_octree_write_budget_ms` (setting): per-step time budget for async apply/write operations.

---

## Restore Tracking (Surgical Restore)

Restore tracking monitors external changes to placed snapshots and enables restoring **only the modified chunks** instead of re-placing the entire snapshot.

### How It Works
1. After placing a snapshot, enable tracking with `map:enable_tracking()`.
2. The engine callback `core.register_on_mapblocks_changed` detects modified mapblocks.
3. A background scanner (budget-limited via `map_octree_restore_scan_budget_ms`, default 5ms) verifies which chunks actually differ from the snapshot.
4. Call `map:schedule_restore()` to revert only the changed chunks.

### API
- **`map:enable_tracking()`**: Enable restore tracking for this snapshot.
- **`map:disable_tracking(opts)`**: Stop tracking.
  - `opts.flush` (boolean, default false): if true, clears pending/dirty state after cancelling any inflight restore.
- **`map:get_tracking_status()`** -> status table or `nil` if tracking is disabled.
  - `pending`: number of chunks queued for verification.
  - `dirty`: number of chunks confirmed different from the snapshot.
  - `base_corner`: world position (min corner) used as the tracking grid origin.
  - `max_corner`: world position (max corner) of the tracked snapshot bounds.
  - `min_blockpos`, `max_blockpos`: mapblock bounds (in mapblock coords, i.e. node coords / 16).
  - `chunk_size`: snapshot chunk grid size `{x,y,z}` when available.
  - `last_change`: stats for the last `on_mapblocks_changed` event seen by the tracker (or `nil`).
    - `t_us`: timestamp from `core.get_us_time()`.
    - `modified_block_count`: count provided by the engine callback.
    - `blocks_seen`: how many block hashes were iterated.
    - `blocks_in_bounds`, `blocks_out_of_bounds`: split by whether the block touched the tracked region.
    - `unique_chunks_enqueued`: number of unique chunks queued for verification.
- **`map:flush_tracking(opts)`** -> info table or `nil` if tracking is disabled.
  - Clears internal tracking state (pending verification queue and dirty set).
  - By default it also cancels any inflight `schedule_restore`.
  - `opts.pending` (boolean, default true): clear pending queue.
  - `opts.dirty` (boolean, default true): clear dirty set.
  - `opts.cancel_restore` (boolean, default true): cancel inflight restore.
  - Returns: table with `{pending_cleared, dirty_cleared, restore_cancelled}` where the first two are counts (number of items cleared) and the last is a boolean.
- **`map:is_restoring()`** -> boolean.
  - True while a `schedule_restore()` is inflight (either waiting for pending drain or writing).
- **`map:get_restore_state()`** -> state table or `nil` if tracking is disabled.
  - `active` (boolean): whether a restore operation is currently inflight.
  - `phase` (string): one of `"idle"`, `"waiting"`, or `"writing"`.
  - `token` (number): unique restore request identifier.
  - `started_us` (number): timestamp in microseconds when the restore started (from `core.get_us_time()`).
  - `err` (string or nil): error message if the restore failed, otherwise nil.
  - `suppress` (boolean): whether restore output is suppressed.
  - `pending` (number): count of chunks still queued for verification.
  - `dirty` (number): count of chunks confirmed different from the snapshot.
- **`map:schedule_restore(callback)`**: Async restore (waits for pending queue, then restores dirty).
  - Callback: `function(ok, err)`
    - `ok=false` with `err="cancelled"` if cancelled via `flush_tracking()`/`disable_tracking()`.
    - `ok=false` with `err="tracking disabled"` if tracking is disabled during the inflight restore.

### In-Game Commands
- You can track multiple snapshots at once (each loaded map has its own tracker). Use `/octree_track_list` to see keys and `-key=...` to target a specific tracker.
- `/octree_load <name> [-subdir=DIR] [-no-current]`: Load a snapshot into memory and (optionally) don't set as current.
- `/octree_maps [-subdir=DIR]`: List saved snapshot files (optionally in a subdir).
- `/octree_loaded`: List loaded maps in memory.
- `/octree_unload <key>`: Unload a map from the registry.
- `/octree_track <name> [-subdir=DIR]`: Enable tracking for a saved snapshot.
- `/octree_track_list`: List tracked snapshots in memory.
- `/octree_track_status [-key=KEY]`: Show pending/dirty counts (key defaults to the last tracked map).
- `/octree_restore [-sync] [-key=KEY]`: Restore dirty chunks (key defaults to the last tracked map).
- `/octree_untrack [-key=KEY]`: Disable tracking (key defaults to the last tracked map).

---

## Performance Tips

1. **Avoid Strings**: When scanning millions of nodes, use `_cid` functions to work with numeric Content IDs.
2. **Prefer Streaming**: For large regions, it is significantly faster to use `read_ranges`, `apply_ranges`, or `for_each_node` than calling `get_node_at` or `set_node_at`. Streaming minimizes internal hash lookups and chunk traversals.
3. **Reuse Manipulators**: Creating an `OctreeManip` is relatively cheap, but don't do it inside a loop nor each step.
4. **RAM Budget**: You can cap memory usage in `minetest.conf` (or via Settings tab) using `map_octree_max_total_ram_budget_mb` (see [README](README.md)).
5. **Time Budgets are Estimates**: `map_octree_write_budget_ms` and other per-step budgets are not strict caps. They do not include GC time, and real step time varies with memory pressure and GC frequency.