local tutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

local function calc_expected_max_vm_volume(budget_mb, inflight)
	inflight = math.max(1, math.floor(tonumber(inflight) or 1))
	return math.floor((budget_mb / 6 * 1024 * 1024) / 8 / inflight)
end


map_octree.tests.register("map:for_each_node visits all in-bounds nodes", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local center = tutil.get_test_region(ctx.player)
	local S = octchunk.SIZE
	local half = S / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})

	tutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		-- Fill cube with stone
		local c_stone = core.get_content_id("default:stone")
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_stone
				end
			end
		end

		-- Sparse param2 samples
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		param2_data[area:index(pmin.x, pmin.y, pmin.z)] = 11
		param2_data[area:index(pmax.x, pmax.y, pmax.z)] = 22
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local map = octmap.new(center, center, {
			store_chunk_blobs = true,
			max_voxelmanip_volume = (S + 1) ^ 3 * 2,
			force_batches = true,
		})

		local expected_count = S * S * S
		local checked = 0
		local count = map:for_each_node(pmin, pmax, function(x, y, z, name, p2)
			checked = checked + 1
			local name2, p22 = map:get_node_at(x, y, z)
			assert(name == name2, "name mismatch")
			assert((p2 or 0) == (p22 or 0), "param2 mismatch")
		end)

		assert(count == expected_count, "for_each_node count mismatch")
		assert(checked == expected_count, "for_each_node callback mismatch")
	end)
end)


map_octree.tests.register("OctreeManip:for_each_node scans snapshot and sees writes", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = octchunk.snap_to_center(tutil.get_test_region(ctx.player))
	local pos1 = base
	local pos2 = vector.add(base, {x = octchunk.SIZE, y = 0, z = 0})

	local half = octchunk.SIZE / 2
	local pA = {x = pos1.x - (half - 1), y = pos1.y, z = pos1.z}
	local pB_center = vector.add(pos1, {x = octchunk.SIZE, y = 0, z = 0})
	local pB = {x = pB_center.x - (half - 1), y = pB_center.y, z = pB_center.z}

	local pmin = {x = pos1.x - half, y = pos1.y - half, z = pos1.z - half}
	local pmax = {x = pos2.x + (half - 1), y = pos2.y + (half - 1), z = pos2.z + (half - 1)}

	tutil.with_voxel_region(pmin, pmax, function()
		local node_a = "default:stone"
		local node_b = "default:dirt"
		assert(core.get_content_id(node_a), "missing node: " .. node_a)
		assert(core.get_content_id(node_b), "missing node: " .. node_b)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pos1, pos2)

		local emin, emax = m:get_emerged_area()
		assert(emin and emax, "expected emerged bounds")

		local count0 = m:for_each_node(emin, emax, function() end)
		assert(type(count0) == "number" and count0 > 0, "expected scan count > 0")

		assert(m:set_node_at(pA.x, pA.y, pA.z, node_a, 3), "set_node_at A failed")
		assert(m:set_node_at(pB.x, pB.y, pB.z, node_b, 4), "set_node_at B failed")
		assert(m:write_to_map(), "write_to_map failed")

		local seen_a = false
		local seen_b = false
		m:for_each_node(emin, emax, function(x, y, z, name, p2)
			if x == pA.x and y == pA.y and z == pA.z then
				seen_a = (name == node_a and (p2 or 0) == 3)
			end
			if x == pB.x and y == pB.y and z == pB.z then
				seen_b = (name == node_b and (p2 or 0) == 4)
			end
		end)

		assert(seen_a, "scan did not see write A")
		assert(seen_b, "scan did not see write B")
	end)
end)


map_octree.tests.register("limits: apply_server_limits clamps cache and voxelmanip volume", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local key = "map_octree_max_total_ram_budget_mb"
	local old = core.settings:get(key)
	local ok, err = pcall(function()
		core.settings:set(key, "600")
		local opts = {cache_mb = 9999, max_voxelmanip_volume = 999999999}
		octmap.apply_server_limits(opts, 4)
		local expected_cache = math.floor(600 * 0.05)
		local expected_vm = calc_expected_max_vm_volume(600, 4)
		assert(opts.cache_mb == expected_cache, "cache_mb clamp mismatch")
		assert(opts.max_voxelmanip_volume == expected_vm, "max_voxelmanip_volume clamp mismatch")

		-- smaller values should stay as-is
		local opts2 = {cache_mb = 1, max_voxelmanip_volume = 123}
		octmap.apply_server_limits(opts2, 4)
		assert(opts2.cache_mb == 1, "cache_mb should not increase")
		assert(opts2.max_voxelmanip_volume == 123, "max_voxelmanip_volume should not increase")
	end)
	if old ~= nil then
		core.settings:set(key, old)
	else
		core.settings:set(key, "")
	end
	assert(ok, err)
end)


map_octree.tests.register("async default inflight cap affects computed volume cap", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	-- Get actual inflight from engine capacity
	local actual_inflight = core.get_async_threading_capacity() or 4

	local key = "map_octree_max_total_ram_budget_mb"
	local key_vm = "map_octree_max_voxelmanip_volume"
	local old = core.settings:get(key)
	local old_vm = core.settings:get(key_vm)
	core.settings:set(key, "600")
	core.settings:set(key_vm, "")  -- clear override to test derived cap

	local pos = ctx.player:get_pos()
	assert(pos, "player position required")
	---@cast pos vector
	local rounded = vector.round(pos)
	---@cast rounded vector
	local center = octchunk.snap_to_center(rounded)
	local opts = {
		file_name = "async_default_inflight_" .. os.time(),
		subdir = "tests",
		-- intentionally omit async_inflight to test default
		-- also omit max_voxelmanip_volume to test derived cap
		store_chunk_blobs = true,
	}

	local expected_vm = calc_expected_max_vm_volume(600, actual_inflight)

	map_octree.save_to_file_async(center, center, opts, function(ok, result, info)
		-- restore settings immediately; caps already computed
		if old ~= nil then
			core.settings:set(key, old)
		else
			core.settings:set(key, "")
		end
		if old_vm ~= nil then
			core.settings:set(key_vm, old_vm)
		else
			core.settings:set(key_vm, "")
		end

		if not ok then
			core.log("warning", "[test] async default inflight test: async save failed: " .. tostring(result))
			return
		end

		-- opts table is mutated by save_to_file_async
		if opts.max_voxelmanip_volume ~= expected_vm then
			core.log("warning", string.format("[test] async default inflight cap mismatch: expected %d got %s", expected_vm, tostring(opts.max_voxelmanip_volume)))
		end

		if type(info) ~= "table" or type(info.batch_count) ~= "number" then
			core.log("warning", "[test] async default inflight test: missing info.batch_count")
		end
	end)

	core.log("action", "[test] async default inflight test: started")
end)