--- Octree node deduplication cache.
-- Uses structural hashing to identify identical subtrees and share them.

local CHILDREN = octchunk.CHILDREN

---@class octcache
octcache = {}

local cache_set

local chunk_hash = map_octree.chunk_hash
local generate_hash = chunk_hash.generate

local trees_cache = {}
local hashes_refs = {}

octcache.trees_cache = trees_cache

local cache_info = {
    counter = 0,
    min_refs = 2,
    trim_trigger_size = 200000,
    last_trim_log_us = 0,
}

setmetatable(trees_cache, {
    __index = cache_info,
})


---Register a tree and all its children in the cache.
---@param octnode OctNode
function octcache.create(octnode)
    local hash = generate_hash(octnode)
    if trees_cache[hash] then return end

    cache_set(hash, octnode)

    for _, child in pairs(octnode[CHILDREN] or {}) do
        octcache.create(child)
    end
end



---Replace tree children with cached instances (deduplication).
---@param octnode OctNode
function octcache.use(octnode)
    local children = octnode[CHILDREN] or {}

    for i, child in pairs(children) do
        local child_hash = generate_hash(child)

        if child[CHILDREN] then
            octcache.use(child)
        end

        local cached = trees_cache[child_hash]
        children[i] = cached
        hashes_refs[child_hash] = (hashes_refs[child_hash] or 0) + 1
    end
end



---Release cached references for a tree (call when a tree is discarded).
---@param octnode OctNode
function octcache.release(octnode)
    local children = octnode[CHILDREN] or {}
    for _, child in pairs(children) do
        if child then
            local child_hash = generate_hash(child)
            local refs = hashes_refs[child_hash]
            if refs then
                refs = refs - 1
                if refs <= 0 then
                    hashes_refs[child_hash] = nil
                else
                    hashes_refs[child_hash] = refs
                end
            end
            if child[CHILDREN] then
                octcache.release(child)
            end
        end
    end
end



---Clear all caches (call when discarding a map).
function octcache.delete()
    for k in pairs(trees_cache) do
        rawset(trees_cache, k, nil)
    end
    for k in pairs(hashes_refs) do
        hashes_refs[k] = nil
    end
    chunk_hash.clear_memoization()
    cache_info.counter = 0
    cache_info.last_trim_log_us = 0
end



---Get cache statistics for debugging.
---@return {trees: integer, hashes: integer, memoized: integer}
function octcache.get_stats()
    local hash_count = 0
    for _ in pairs(hashes_refs) do hash_count = hash_count + 1 end
    local memo_count = chunk_hash.get_memoized_count()
    return {
        trees = cache_info.counter,
        hashes = hash_count,
        memoized = memo_count,
    }
end



-------------------------------------------------------------------------------
-- Local function implementations
-------------------------------------------------------------------------------

---Store a node in the cache if absent.
---@param hash string
---@param node OctNode
---@return boolean
function cache_set(hash, node)
    if trees_cache[hash] ~= nil then
        return false
    end
    rawset(trees_cache, hash, node)
    cache_info.counter = cache_info.counter + 1
    if cache_info.counter % cache_info.trim_trigger_size == 0 then
        octcache.trim(true)
    end
    return true
end



---Trim cache entries with low reference counts.
---@param log? boolean
---@return integer
function octcache.trim(log)
    local counter = cache_info.counter
    for hash, _ in pairs(trees_cache) do
        local refs = hashes_refs[hash] or 0
        if refs < cache_info.min_refs then
            rawset(trees_cache, hash, nil)
            hashes_refs[hash] = nil
            counter = counter - 1
        end
    end

    cache_info.counter = counter
    if log then
        local now = core.get_us_time()
        if now - (cache_info.last_trim_log_us or 0) >= 1000000 then
            cache_info.last_trim_log_us = now
            core.log("verbose", "[octcache] Trimmed deduplication cache to " .. cache_info.counter .. " trees")
        end
    end

    return counter
end
