local MAPBLOCK_SIZE = core.MAP_BLOCKSIZE or 16
assert(MAPBLOCK_SIZE == octchunk.SIZE, "mapblock size must match octchunk.SIZE")
local SETTING_RESTORE_SCAN_BUDGET_MS = "map_octree_restore_scan_budget_ms"
local DEFAULT_RESTORE_SCAN_BUDGET_MS = 5

local rt = map_octree.restore_tracking
rt.MAPBLOCK_SIZE = MAPBLOCK_SIZE



---Get scan budget in milliseconds.
---@param opts? {scan_budget_ms?: number}
---@return number
function rt.get_scan_budget_ms(opts)
	opts = opts or {}
	local v = tonumber(opts.scan_budget_ms)
	if v and v > 0 then
		return v
	end
	v = tonumber(core.settings:get(SETTING_RESTORE_SCAN_BUDGET_MS) or "")
	return (v and v > 0) and v or DEFAULT_RESTORE_SCAN_BUDGET_MS
end



---Convert node coordinate to mapblock coordinate.
---@param n number
---@return integer
function rt.node_to_block_coord(n)
	return math.floor(n / MAPBLOCK_SIZE)
end



---Compute block bounds from node bounds.
---@param minp vector
---@param maxp vector
---@return {x: integer, y: integer, z: integer} minb
---@return {x: integer, y: integer, z: integer} maxb
function rt.compute_block_bounds(minp, maxp)
	return {
		x = rt.node_to_block_coord(minp.x),
		y = rt.node_to_block_coord(minp.y),
		z = rt.node_to_block_coord(minp.z),
	}, {
		x = rt.node_to_block_coord(maxp.x),
		y = rt.node_to_block_coord(maxp.y),
		z = rt.node_to_block_coord(maxp.z),
	}
end



---Check if a block position is within bounds.
---@param bp {x: integer, y: integer, z: integer}
---@param minb {x: integer, y: integer, z: integer}
---@param maxb {x: integer, y: integer, z: integer}
---@return boolean
function rt.block_in_bounds(bp, minb, maxb)
	return bp.x >= minb.x and bp.x <= maxb.x
		and bp.y >= minb.y and bp.y <= maxb.y
		and bp.z >= minb.z and bp.z <= maxb.z
end



---Compute a linear chunk key for grid coordinates.
---@param map OctMap
---@param gx integer
---@param gy integer
---@param gz integer
---@return integer
function rt.chunk_key(map, gx, gy, gz)
	local size = map.trees.size
	return ((gx - 1) * size.y + (gy - 1)) * size.z + gz
end



---Convert a linear chunk key to grid coordinates.
---@param map OctMap
---@param key integer
---@return integer gx
---@return integer gy
---@return integer gz
function rt.key_to_grid(map, key)
	local size = map.trees.size
	local gz = ((key - 1) % size.z) + 1
	local tmp = math.floor((key - 1) / size.z)
	local gy = (tmp % size.y) + 1
	local gx = math.floor(tmp / size.y) + 1
	return gx, gy, gz
end



---Get the world-space min corner for a chunk.
---@param tracker RestoreTracker
---@param gx integer
---@param gy integer
---@param gz integer
---@return {x: number, y: number, z: number}
function rt.grid_to_node_min(tracker, gx, gy, gz)
	local base = tracker.base_corner
	return {
		x = base.x + (gx - 1) * MAPBLOCK_SIZE,
		y = base.y + (gy - 1) * MAPBLOCK_SIZE,
		z = base.z + (gz - 1) * MAPBLOCK_SIZE,
	}
end



---Check if grid coords are within map bounds.
---@param map OctMap
---@param gx integer
---@param gy integer
---@param gz integer
---@return boolean
function rt.grid_in_range(map, gx, gy, gz)
	local s = map.trees.size
	return gx >= 1 and gx <= s.x and gy >= 1 and gy <= s.y and gz >= 1 and gz <= s.z
end



---Clamp a number to a range.
---@param v number
---@param lo number
---@param hi number
---@return number
function rt.clamp(v, lo, hi)
	if v < lo then return lo end
	if v > hi then return hi end
	return v
end



---Convert block position to clamped chunk grid range.
---@param tracker RestoreTracker
---@param bp {x: integer, y: integer, z: integer}
---@return integer gx1
---@return integer gy1
---@return integer gz1
---@return integer gx2
---@return integer gy2
---@return integer gz2
function rt.blockpos_to_grid_range(tracker, bp)
	local map = tracker.map
	local s = map.trees.size
	local block_min = {
		x = bp.x * MAPBLOCK_SIZE,
		y = bp.y * MAPBLOCK_SIZE,
		z = bp.z * MAPBLOCK_SIZE,
	}
	local block_max = {
		x = block_min.x + MAPBLOCK_SIZE - 1,
		y = block_min.y + MAPBLOCK_SIZE - 1,
		z = block_min.z + MAPBLOCK_SIZE - 1,
	}

	local base = tracker.base_corner
	local gx1 = math.floor((block_min.x - base.x) / MAPBLOCK_SIZE) + 1
	local gy1 = math.floor((block_min.y - base.y) / MAPBLOCK_SIZE) + 1
	local gz1 = math.floor((block_min.z - base.z) / MAPBLOCK_SIZE) + 1
	local gx2 = math.floor((block_max.x - base.x) / MAPBLOCK_SIZE) + 1
	local gy2 = math.floor((block_max.y - base.y) / MAPBLOCK_SIZE) + 1
	local gz2 = math.floor((block_max.z - base.z) / MAPBLOCK_SIZE) + 1

	gx1 = rt.clamp(gx1, 1, s.x)
	gy1 = rt.clamp(gy1, 1, s.y)
	gz1 = rt.clamp(gz1, 1, s.z)
	gx2 = rt.clamp(gx2, 1, s.x)
	gy2 = rt.clamp(gy2, 1, s.y)
	gz2 = rt.clamp(gz2, 1, s.z)

	return gx1, gy1, gz1, gx2, gy2, gz2
end



---Build expected chunk buffers for verification.
---@param map OctMap
---@param emerged_min vector
---@param emerged_max vector
---@param chunk_min vector
---@param gx integer
---@param gy integer
---@param gz integer
---@param out {data: integer[], param2: integer[], param1: integer[]}
---@return integer volume
function rt.build_expected_chunk(map, emerged_min, emerged_max, chunk_min, gx, gy, gz, out)
	local default_cid = core.get_content_id(map.default_node or "air")
	local area = VoxelArea(emerged_min, emerged_max)
	local volume = (emerged_max.x - emerged_min.x + 1) * (emerged_max.y - emerged_min.y + 1) * (emerged_max.z - emerged_min.z + 1)

	local data = out.data
	local p2 = out.param2
	local p1 = out.param1

	for i = 1, volume do
		data[i] = default_cid
		p2[i] = 0
		p1[i] = 0
	end

	local cell = matrix3d.get(map.trees, gx, gy, gz)
	if cell ~= nil then
		local tree = assert(octmap._get_tree_cached(map, gx, gy, gz))
		local half = octchunk.SIZE / 2
		local center = vector.new(chunk_min.x + half, chunk_min.y + half, chunk_min.z + half)
		local chunk_max = vector.new(chunk_min.x + MAPBLOCK_SIZE - 1, chunk_min.y + MAPBLOCK_SIZE - 1, chunk_min.z + MAPBLOCK_SIZE - 1)
		octchunk.fill_voxel_area(tree, chunk_min, chunk_max, area, data, p2, p1, nil, center, tree.size or octchunk.SIZE)
	end
	-- Se cell == nil, il chunk è sparse e contiene solo default_node (già riempito sopra)

	return volume
end



---Compare live chunk data against snapshot buffers.
---@param tracker RestoreTracker
---@param gx integer
---@param gy integer
---@param gz integer
function rt.verify_chunk_against_snapshot(tracker, gx, gy, gz)
	local map = tracker.map
	if not rt.grid_in_range(map, gx, gy, gz) then
		return
	end

	local node_min = rt.grid_to_node_min(tracker, gx, gy, gz)
	local node_max = {
		x = node_min.x + MAPBLOCK_SIZE - 1,
		y = node_min.y + MAPBLOCK_SIZE - 1,
		z = node_min.z + MAPBLOCK_SIZE - 1,
	}

	core.load_area(node_min, node_max)
	local manip = core.get_voxel_manip()
	local e1, e2 = manip:read_from_map(node_min, node_max)

	local live_data = manip:get_data()
	local live_p2 = manip:get_param2_data()

	rt.build_expected_chunk(map, e1, e2, node_min, gx, gy, gz, tracker._expected)

	local expected_data = tracker._expected.data
	local expected_p2 = tracker._expected.param2

	local area = VoxelArea(e1, e2)
	local equal = true
	local cmp_min_x = math.max(node_min.x, e1.x)
	local cmp_min_y = math.max(node_min.y, e1.y)
	local cmp_min_z = math.max(node_min.z, e1.z)
	local cmp_max_x = math.min(node_max.x, e2.x)
	local cmp_max_y = math.min(node_max.y, e2.y)
	local cmp_max_z = math.min(node_max.z, e2.z)
	-- NOTE: param1 (light) is intentionally ignored here to avoid marking trees dirty
	-- during frequent lighting updates. Structural changes are captured by content_id/param2.
	for x = cmp_min_x, cmp_max_x do
		for y = cmp_min_y, cmp_max_y do
			for z = cmp_min_z, cmp_max_z do
				local i = area:index(x, y, z)
				if live_data[i] ~= expected_data[i] then
					equal = false
					break
				end
				if live_p2[i] ~= expected_p2[i] then
					equal = false
					break
				end
			end
			if not equal then break end
		end
		if not equal then break end
	end

	manip:close()

	local key = rt.chunk_key(map, gx, gy, gz)
	if equal then
		tracker.dirty[key] = nil
		tracker.dirty_coords[key] = nil
	else
		tracker.dirty[key] = true
		tracker.dirty_coords[key] = {gx = gx, gy = gy, gz = gz}
	end
end
