local tutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field, invisible


map_octree.tests.register("map: content_id read APIs", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local center = tutil.get_test_region(ctx.player)
	local S = octchunk.SIZE
	local half = S / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})

	local c_stone = core.get_content_id("default:stone")
	local c_dirt = core.get_content_id("default:dirt")
	assert(type(c_stone) == "number", "missing node: default:stone")
	assert(type(c_dirt) == "number", "missing node: default:dirt")

	tutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		-- Part A: uniform tree (likely sparsified) -> default cell path
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_stone
				end
			end
		end
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local map1 = octmap.new(center, center, {
			store_chunk_blobs = true,
			max_voxelmanip_volume = (S + 1) ^ 3 * 2,
			force_batches = true,
		})

		local cid1, p21 = map1:get_node_cid_at(center.x, center.y, center.z)
		assert(cid1 == c_stone, "cid mismatch (uniform)")
		assert((p21 or 0) == 0, "param2 mismatch (uniform)")

		local name1, p2n1 = map1:get_node_at(center.x, center.y, center.z)
		assert(name1 == "default:stone", "name mismatch (uniform)")
		assert((p2n1 or 0) == 0, "param2 mismatch (uniform name)")

		local expected_count = S * S * S
		local visited = 0
		local count = map1:for_each_node_cid(pmin, pmax, function(x, y, z, cid, p2)
			visited = visited + 1
			local cid2, p22 = map1:get_node_cid_at(x, y, z)
			assert(cid == cid2, "cid mismatch in scan")
			assert((p2 or 0) == (p22 or 0), "param2 mismatch in scan")
		end)
		assert(count == expected_count, "for_each_node_cid count mismatch")
		assert(visited == expected_count, "for_each_node_cid callback mismatch")

		-- Part B: non-uniform tree + param2
		local dirt_pos = {x = pmin.x, y = pmin.y, z = pmin.z}
		data[area:index(dirt_pos.x, dirt_pos.y, dirt_pos.z)] = c_dirt
		param2_data[area:index(dirt_pos.x, dirt_pos.y, dirt_pos.z)] = 7
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local map2 = octmap.new(center, center, {
			store_chunk_blobs = true,
			max_voxelmanip_volume = (S + 1) ^ 3 * 2,
			force_batches = true,
		})

		local cid_d, p2_d = map2:get_node_cid_at(dirt_pos.x, dirt_pos.y, dirt_pos.z)
		assert(cid_d == c_dirt, "cid mismatch (non-uniform)")
		assert((p2_d or 0) == 7, "param2 mismatch (non-uniform)")

		local name_d, p2_name_d = map2:get_node_at(dirt_pos.x, dirt_pos.y, dirt_pos.z)
		assert(name_d == "default:dirt", "name mismatch (non-uniform)")
		assert((p2_name_d or 0) == (p2_d or 0), "param2 mismatch (non-uniform name)")
	end)
end)


map_octree.tests.register("OctreeManip: content_id write + apply", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local pos0 = tutil.get_test_region(ctx.player)
	local base = vector.add(octchunk.snap_to_center(pos0), {x = 0, y = 0, z = 4096})
	local pos1 = base
	local pos2 = vector.add(base, {x = octchunk.SIZE, y = 0, z = 0})

	local node_a = "default:stone"
	local node_b = "default:dirt"
	local c_a = core.get_content_id(node_a)
	local c_b = core.get_content_id(node_b)
	assert(type(c_a) == "number", "missing node: " .. node_a)
	assert(type(c_b) == "number", "missing node: " .. node_b)

	local half = octchunk.SIZE / 2
	local pmin = {x = pos1.x - half, y = pos1.y - half, z = pos1.z - half}
	local pmax = {x = pos2.x + (half - 1), y = pos2.y + (half - 1), z = pos2.z + (half - 1)}

	local sub1 = {x = pmin.x, y = pmin.y, z = pmin.z}
	local sub2 = {x = pmin.x + 3, y = pmin.y + 3, z = pmin.z + 3}
	local expected_changed = (sub2.x - sub1.x + 1) * (sub2.y - sub1.y + 1) * (sub2.z - sub1.z + 1)

	tutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		-- Fill entire region with stone
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_a
				end
			end
		end
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pos1, pos2)

		local changed = m:apply(sub1, sub2, function(_, _, _, cid, p2)
			if cid == c_a then
				return c_b, p2
			end
			return
		end)
		assert(changed == expected_changed, "apply changed_count mismatch")

		-- Also verify direct content_id write API (override one node's param2)
		local marker = {x = sub2.x, y = sub2.y, z = sub2.z}
		assert(m:set_node_cid_at(marker.x, marker.y, marker.z, c_b, 12), "set_node_cid_at failed")

		assert(m:write_to_map(), "write_to_map failed")

		local n = core.get_node(marker)
		assert(n and n.name == node_b, "live mismatch after apply")
		assert((n.param2 or 0) == 12, "live param2 mismatch after set_node_cid_at")

		local cid_s, p2_s = m._snapshot:get_node_cid_at(marker.x, marker.y, marker.z)
		assert(cid_s == c_b, "snapshot cid mismatch")
		assert((p2_s or 0) == 12, "snapshot param2 mismatch")
	end)
end)