local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field


map_octree.tests.register("OctreeManip: apply_ranges checkerboard trees", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	local oc = octchunk

	local origin = testutil.get_test_region(ctx.player)
	local base = vector.add(map_octree.get_chunk_pos(origin), {x = 0, y = 0, z = 9472})
	local size = oc.SIZE * 2
	local pmin = base
	local pmax = vector.add(base, {x = size - 1, y = size - 1, z = size - 1})

	local node_base = "default:dirt"
	local node_target = "default:stone"
	local c_base = core.get_content_id(node_base)
	local c_target = core.get_content_id(node_target)
	assert(type(c_base) == "number", "missing node: " .. node_base)
	assert(type(c_target) == "number", "missing node: " .. node_target)

	local tree_volume = oc.SIZE * oc.SIZE * oc.SIZE
	local selected = 0
	local ranges = {}
	for gx = 1, 2 do
		for gy = 1, 2 do
			for gz = 1, 2 do
				if (gx + gy + gz) % 2 == 0 then
					selected = selected + 1
					ranges[#ranges + 1] = {min = {x = gx, y = gy, z = gz}, max = {x = gx, y = gy, z = gz}}
				end
			end
		end
	end

	testutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_base
				end
			end
		end
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pmin, pmax)

		local changed = m:apply_ranges(ranges, function(_, _, _, _, p2)
			return c_target, p2
		end, {grid = "chunk"})
		assert(changed == selected * tree_volume, "checkerboard changed_count mismatch")
		assert(m:write_to_map(), "write_to_map failed")

		local target_count = testutil.count_cid_in_snapshot(pmin, pmax, c_target, size)
		local expected_target = selected * tree_volume
		assert(target_count == expected_target, "checkerboard stone count mismatch")

		local base_count = testutil.count_cid_in_snapshot(pmin, pmax, c_base, size)
		local total = size * size * size
		assert(base_count == (total - expected_target), "checkerboard base count mismatch")
	end)
end)


map_octree.tests.register("OctreeManip: apply_ranges hollow cube", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	local oc = octchunk

	local origin = testutil.get_test_region(ctx.player)
	local base = vector.add(map_octree.get_chunk_pos(origin), {x = 0, y = 0, z = 9536})
	local size = oc.SIZE * 2
	local pmin = base
	local pmax = vector.add(base, {x = size - 1, y = size - 1, z = size - 1})

	local node_base = "default:dirt"
	local node_target = "default:stone"
	local c_base = core.get_content_id(node_base)
	local c_target = core.get_content_id(node_target)
	assert(type(c_base) == "number", "missing node: " .. node_base)
	assert(type(c_target) == "number", "missing node: " .. node_target)

	local thickness = 1
	local inner = size - thickness * 2
	local expected_shell = size * size * size - inner * inner * inner

	local ranges = {
		{pos1 = {x = pmin.x, y = pmin.y, z = pmin.z}, pos2 = {x = pmax.x, y = pmin.y + thickness - 1, z = pmax.z}},
		{pos1 = {x = pmin.x, y = pmax.y - thickness + 1, z = pmin.z}, pos2 = {x = pmax.x, y = pmax.y, z = pmax.z}},
		{pos1 = {x = pmin.x, y = pmin.y, z = pmin.z}, pos2 = {x = pmin.x + thickness - 1, y = pmax.y, z = pmax.z}},
		{pos1 = {x = pmax.x - thickness + 1, y = pmin.y, z = pmin.z}, pos2 = {x = pmax.x, y = pmax.y, z = pmax.z}},
		{pos1 = {x = pmin.x, y = pmin.y, z = pmin.z}, pos2 = {x = pmax.x, y = pmax.y, z = pmin.z + thickness - 1}},
		{pos1 = {x = pmin.x, y = pmin.y, z = pmax.z - thickness + 1}, pos2 = {x = pmax.x, y = pmax.y, z = pmax.z}},
	}

	testutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_base
				end
			end
		end
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pmin, pmax)

		m:apply_ranges(ranges, function(_, _, _, _, p2)
			return c_target, p2
		end)
		assert(m:write_to_map(), "write_to_map failed")

		local target_count = testutil.count_cid_in_snapshot(pmin, pmax, c_target, size)
		assert(target_count == expected_shell, "hollow cube stone count mismatch")
	end)
end)


map_octree.tests.register("OctreeManip: apply_ranges_async checkerboard", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("apply_ranges_async") then
		return
	end
	local oc = octchunk

	local origin = testutil.get_test_region(ctx.player)
	local base = vector.add(map_octree.get_chunk_pos(origin), {x = 0, y = 0, z = 9600})
	local size = oc.SIZE * 2
	local pmin = base
	local pmax = vector.add(base, {x = size - 1, y = size - 1, z = size - 1})

	local node_base = "default:dirt"
	local node_target = "default:stone"
	local c_base = core.get_content_id(node_base)
	local c_target = core.get_content_id(node_target)
	assert(type(c_base) == "number", "missing node: " .. node_base)
	assert(type(c_target) == "number", "missing node: " .. node_target)

	local tree_volume = oc.SIZE * oc.SIZE * oc.SIZE
	local selected = 0
	local ranges = {}
	for gx = 1, 2 do
		for gy = 1, 2 do
			for gz = 1, 2 do
				if (gx + gy + gz) % 2 == 1 then
					selected = selected + 1
					ranges[#ranges + 1] = {min = {x = gx, y = gy, z = gz}, max = {x = gx, y = gy, z = gz}}
				end
			end
		end
	end

	local test_id = "apply_ranges_async_" .. os.time()
	local expected_sand = selected * tree_volume
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	testutil.with_voxel_region_async(pmin, pmax, function(manip, area, data, param2_data, restore)
		for x = pmin.x, pmax.x do
			for y = pmin.y, pmax.y do
				for z = pmin.z, pmax.z do
					data[area:index(x, y, z)] = c_base
				end
			end
		end
		for i = 1, #param2_data do
			param2_data[i] = 0
		end
		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pmin, pmax)

		local function fail(msg)
			core.log("error", "[test] " .. test_id .. ": FAIL - " .. tostring(msg))
			if end_async then end_async() end
			restore()
		end
		local function pass()
			core.log("action", "[test] " .. test_id .. ": PASS")
			if end_async then end_async() end
			restore()
		end

		m:apply_ranges_async(ranges, function(_, _, _, _, p2)
			return c_target, p2
		end, {grid = "chunk"}, function(ok, changed)
			core.after(0, function()
				if not ok then
					fail("apply_ranges_async failed")
					return
				end
				if changed ~= expected_sand then
					fail("apply_ranges_async changed_count mismatch")
					return
				end
				if not m:write_to_map() then
					fail("write_to_map failed")
					return
				end
				local target_count = testutil.count_cid_in_snapshot(pmin, pmax, c_target, size)
				if target_count ~= expected_sand then
					fail("apply_ranges_async stone count mismatch")
					return
				end
				pass()
			end)
		end)
	end)
end)
