local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

local function get_chunk_bounds(center)
    local half = octchunk.SIZE / 2
    local pmin = vector.subtract(center, {x = half, y = half, z = half})
    local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})
    return pmin, pmax
end


map_octree.tests.register("save_around_player uses radius/height", function(ctx)
    assert(ctx and ctx.player, "ctx.player required")

    local radius = 8
    local height = 12
    local stamp = os.time()
    local file_name = "save_around_player_" .. stamp
    local subdir = "tests"

    local map, err = map_octree.save_around_player(ctx.player, radius, height, {
        file_name = file_name,
        subdir = subdir,
        store_chunk_blobs = true,
        force_batches = true,
        max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
    })
    assert(map, err or "save_around_player failed")

    local pos = assert(ctx.player:get_pos(), "player position required")
    local p1 = vector.new(pos.x - radius, pos.y - height, pos.z - radius)
    local p2 = vector.new(pos.x + radius, pos.y + height, pos.z + radius)
    local exp_min, exp_max = vector.sort(p1, p2)
    exp_min = assert(vector.round(exp_min))
    exp_max = assert(vector.round(exp_max))

    local got_min, got_max = map:get_emerged_area()
    assert(got_min and got_max, "map:get_emerged_area failed")
    assert(got_min.x == exp_min.x and got_min.y == exp_min.y and got_min.z == exp_min.z,
        "save_around_player min mismatch")
    assert(got_max.x == exp_max.x and got_max.y == exp_max.y and got_max.z == exp_max.z,
        "save_around_player max mismatch")

    local dir = map_octree.get_storage_dir(subdir)
    os.remove(dir .. "/" .. file_name .. ".bin")
end)


map_octree.tests.register("save_to_file supports store_chunk_blobs=false", function(ctx)
    assert(ctx and ctx.player, "ctx.player required")

    local base = testutil.get_test_region(ctx.player)
    local center = octchunk.snap_to_center(vector.add(base, {x = 0, y = 0, z = 9792}))
    local pmin, pmax = get_chunk_bounds(center)

    local file_name = "save_no_blobs_" .. os.time()
    local subdir = "tests"

    testutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
        local cid_air = core.get_content_id("air")
        local cid_dirt = core.get_content_id("default:dirt")
        assert(type(cid_air) == "number", "missing node: air")
        assert(type(cid_dirt) == "number", "missing node: default:dirt")

        for z = pmin.z, pmax.z do
            for y = pmin.y, pmax.y do
                for x = pmin.x, pmax.x do
                    local idx = area:index(x, y, z)
                    data[idx] = cid_air
                    if param2_data then param2_data[idx] = 0 end
                end
            end
        end
        data[area:index(center.x, center.y, center.z)] = cid_dirt
        if param2_data then param2_data[area:index(center.x, center.y, center.z)] = 3 end

        manip:set_data(data)
        if manip.set_param2_data and param2_data then
            manip:set_param2_data(param2_data)
        end
        manip:write_to_map(false)

        local map, err = map_octree.save_to_file(pmin, pmax, {
            file_name = file_name,
            subdir = subdir,
            store_chunk_blobs = false,
            cache_mb = 2,
            force_batches = true,
            max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
        })
        assert(map, err or "save_to_file failed")

        local name, p2 = map:get_node_at(center.x, center.y, center.z)
        assert(name == "default:dirt", "save_to_file blobless get_node_at mismatch")
        assert((p2 or 0) == 3, "save_to_file blobless param2 mismatch")

        local loaded = map_octree.load_map(file_name, {subdir = subdir, set_current = false})
        assert(loaded, "expected loaded map")
        local lname, lp2 = loaded:get_node_at(center.x, center.y, center.z)
        assert(lname == "default:dirt", "load_map blobless get_node_at mismatch")
        assert((lp2 or 0) == 3, "load_map blobless param2 mismatch")
    end)

    local dir = map_octree.get_storage_dir(subdir)
    os.remove(dir .. "/" .. file_name .. ".bin")
end)


map_octree.tests.register("save_to_file_async supports flush/read_budget opts", function(ctx)
    assert(ctx and ctx.player, "ctx.player required")
    if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("save_to_file_async opts") then
        return
    end

    local pos = assert(ctx.player:get_pos(), "player position required")
    pos = assert(vector.round(pos))
    local center = octchunk.snap_to_center(pos)
    local file_name = "async_opts_" .. os.time()
    local subdir = "tests"

    local test_id = "save_async_opts_" .. os.time()
    local end_async = nil
    if map_octree.tests.async_start then
        end_async = map_octree.tests.async_start(test_id)
    end

    map_octree.save_to_file_async(center, center, {
        file_name = file_name,
        subdir = subdir,
        async_inflight = 1,
        flush_async_workers = false,
        read_budget_ms = 1,
        max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
    }, function(ok, result)
        if not ok then
            core.log("error", "[test] " .. test_id .. ": async save failed: " .. tostring(result))
            if end_async then end_async() end
            return
        end

        local map = result
        assert(map and map.trees, "async save map missing")
        if end_async then end_async() end

        local dir = map_octree.get_storage_dir(subdir)
        os.remove(dir .. "/" .. file_name .. ".bin")
    end)
end)
