--- Octree chunk management: 16x16x16 voxel trees with LOD compression.
-- Trees store content_id values directly; name mapping is handled at octmap level.

--#region types
---@class Octchunk
---@field center vector Center position of this tree
---@field size integer Side length (16 for standard chunks)

---@alias OctNode {[1]: integer, [2]?: table<integer, OctNode>, [3]?: integer, [4]?: integer}

---@class FillCtx
---@field req_min vector
---@field req_max vector
---@field area VoxelArea
---@field out_data? integer[]
---@field out_param2? integer[]
---@field out_param1? integer[]
---@field resolve_value fun(cid: integer): integer
---@field resolve_cid? fun(cid: integer): integer

---@class EmitCtx
---@field req_min vector
---@field req_max vector
---@field cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@field resolve_value fun(cid: integer): integer

---@class ScanCtx
---@field req_min vector
---@field req_max vector
---@field cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@field resolve_value fun(cid: integer): integer
--#endregion

---@class octchunk
octchunk = {}
octchunk.SIZE = 16

local compress = core.compress
local serialize = core.serialize
local decompress = core.decompress
local deserialize = core.deserialize
local floor = math.floor

local ZSTD_LEVEL_TREE = 3 -- zstd level 3: good speed/ratio balance for voxel trees

local ID = 1
local CHILDREN = 2
local PARAM2 = 3
local PARAM1 = 4
octchunk.ID = ID
octchunk.CHILDREN = CHILDREN
octchunk.PARAM2 = PARAM2
octchunk.PARAM1 = PARAM1

-- Precomputed child offsets: index -> {dx, dy, dz} multipliers (-1 or +1)
local CHILD_OFFSETS = {
	{-1, -1, -1},
	{1,  -1, -1},
	{-1, 1,  -1},
	{1,  1,  -1},
	{-1, -1, 1},
	{1,  -1, 1},
	{-1, 1,  1},
	{1,  1,  1},
}

local merge_nodes
local get_child_center_coords
local explore_area
local fill_intersection
local identity_cid
local fill_node_impl
local emit_box_impl
local scan_node_impl



---Create a new octree chunk.
---@param center_or_pos1 vector Center position, or min corner if pos2 provided
---@param pos2? vector Max corner (optional)
---@return Octchunk
function octchunk.new(center_or_pos1, pos2)
	local new_tree = {}
	local center

	if pos2 then
		center_or_pos1, pos2 = vector.sort(center_or_pos1, pos2)
		center = vector.add(center_or_pos1, vector.divide(vector.subtract(pos2, center_or_pos1), 2))
	else
		center = center_or_pos1
	end

	new_tree.center = octchunk.snap_to_center(center) -- Center of the root node
	new_tree.size = octchunk.SIZE -- Area size

	octchunk.populate_tree(new_tree)

	return new_tree
end



local data = {}
local param2_data = {}
local param1_data = {}

-- Populate tree from pre-loaded VoxelArea (for batch processing)
-- Trees store content_id directly; global content_id_map at octmap level handles name mapping
-- area_param2_data and area_param1_data are optional; when nil, params default to 0 everywhere.
---@param root Octchunk
---@param area VoxelArea
---@param area_data integer[]
---@param area_param2_data? integer[]
---@param area_param1_data? integer[]
---@return Octchunk
function octchunk.populate_tree_from_area(root, area, area_data, area_param2_data, area_param1_data)
	local c = root.center
	explore_area(root, c.x, c.y, c.z, root.size, area, area_data, area_param2_data, area_param1_data)
	return root
end



---Populate tree with its own VoxelManip (slower for batch)
---@param root Octchunk
---@return Octchunk
function octchunk.populate_tree(root)
	local min = vector.subtract(root.center, root.size / 2)
	local max = vector.add(root.center, root.size / 2)

	local manip = core.get_voxel_manip()
	local emerged_pos1, emerged_pos2 = manip:read_from_map(min, max)
	local area = VoxelArea(emerged_pos1, emerged_pos2)
	manip:get_data(data)
	manip:get_param2_data(param2_data)
	manip:get_light_data(param1_data)
	manip:close()

	return octchunk.populate_tree_from_area(root, area, data, param2_data, param1_data)
end



---Iterative get_node lookup (zero allocations in hot path).
---Returns content_id directly; caller uses content_id_map for name lookup.
---@param tree Octchunk
---@param searched_pos vector
---@param center? vector Override center
---@param size? integer Override size
---@return integer content_id
function octchunk.get_node_cid(tree, searched_pos, center, size)
	-- Init from tree root
	if tree.center then
		center = tree.center
		size = tree.size
	end

	-- Floor searched_pos once (avoid vector.floor allocation)
	local sx, sy, sz = floor(searched_pos.x), floor(searched_pos.y), floor(searched_pos.z)
	local cx, cy, cz = center.x, center.y, center.z

	local node = tree

	-- Iterative descent
	while true do
		-- Inline get_pos_idx
		local index = 1 +
			(sx >= cx and 1 or 0) +
			(sy >= cy and 2 or 0) +
			(sz >= cz and 4 or 0)

		local children = node[CHILDREN]
		if not children or not children[index] then
			return node[ID]
		end

		-- Descend: update center coords in place
		local offset = size / 4
		local off = CHILD_OFFSETS[index]
		cx = cx + off[1] * offset
		cy = cy + off[2] * offset
		cz = cz + off[3] * offset
		size = size / 2

		node = children[index]
	end
end



---Iterative get_node lookup (returns node name string).
---@param tree Octchunk
---@param searched_pos vector
---@param center? vector Override center
---@param size? integer Override size
---@param content_id_map? table<integer, string>
---@return string|nil name
function octchunk.get_node_name(tree, searched_pos, center, size, content_id_map)
	local content_id = octchunk.get_node_cid(tree, searched_pos, center, size)
	if content_id == nil then
		return nil
	end
	if content_id_map then
		return content_id_map[content_id] or core.get_name_from_content_id(content_id)
	end
	return core.get_name_from_content_id(content_id)
end



---Iterative get_node_value (content_id + param2 + param1).
---@param tree Octchunk
---@param x number
---@param y number
---@param z number
---@param center? vector Override center
---@param size? integer Override size
---@return integer content_id
---@return integer param2
---@return integer param1
function octchunk.get_node_value_xyz(tree, x, y, z, center, size)
	if tree.center then
		center = tree.center
		size = tree.size
	end

	local sx, sy, sz = floor(x), floor(y), floor(z)
	local cx, cy, cz = center.x, center.y, center.z

	local node = tree
	while true do
		local index = 1 +
			(sx >= cx and 1 or 0) +
			(sy >= cy and 2 or 0) +
			(sz >= cz and 4 or 0)

		local children = node[CHILDREN]
		if not children or not children[index] then
			return node[ID], (node[PARAM2] or 0), (node[PARAM1] or 0)
		end

		local offset = size / 4
		local off = CHILD_OFFSETS[index]
		cx = cx + off[1] * offset
		cy = cy + off[2] * offset
		cz = cz + off[3] * offset
		size = size / 2

		node = children[index]
	end
end



---Vector version of get_node_value_xyz.
---@param tree Octchunk
---@param searched_pos vector
---@param center? vector
---@param size? integer
---@return integer content_id
---@return integer param2
---@return integer param1
function octchunk.get_node_value(tree, searched_pos, center, size)
	return octchunk.get_node_value_xyz(tree, searched_pos.x, searched_pos.y, searched_pos.z, center, size)
end



---Fill a VoxelArea buffer from an octree.
---Writes to out_data, out_param2, and/or out_param1 (at least one must be non-nil).
---@param tree Octchunk
---@param req_min vector
---@param req_max vector
---@param area VoxelArea
---@param out_data? integer[] Content ID buffer
---@param out_param2? integer[] Param2 buffer
---@param out_param1? integer[] Param1 buffer
---@param resolve_cid? fun(cid: integer): integer Content ID remapper
---@param center_override? vector
---@param size_override? integer
function octchunk.fill_voxel_area(tree, req_min, req_max, area, out_data, out_param2, out_param1, resolve_cid, center_override, size_override)
	assert(out_data ~= nil or out_param2 ~= nil or out_param1 ~= nil, "out_data, out_param2, or out_param1 required")
	local resolve_value = resolve_cid or identity_cid
	local ctx = {
		req_min = req_min,
		req_max = req_max,
		area = area,
		out_data = out_data,
		out_param2 = out_param2,
		out_param1 = out_param1,
		resolve_cid = resolve_cid,
		resolve_value = resolve_value,
	}
	local c = center_override or tree.center
	local s = size_override or tree.size
	fill_node_impl(ctx, tree, c.x, c.y, c.z, s)
end



---@param tree Octchunk
---@param req_min vector
---@param req_max vector
---@param cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@param resolve_cid? fun(cid: integer): integer Content ID remapper
---@param center_override? vector
---@param size_override? integer
function octchunk.for_each_in_bounds(tree, req_min, req_max, cb, resolve_cid, center_override, size_override)
	assert(type(cb) == "function", "callback required")
	local resolve_value = resolve_cid or identity_cid
	local ctx = {
		req_min = req_min,
		req_max = req_max,
		cb = cb,
		resolve_value = resolve_value,
	}

	local center = center_override or tree.center
	local size = size_override or tree.size
	scan_node_impl(ctx, tree, center.x, center.y, center.z, size)
end



---Snap a position to the center of its containing chunk.
---@param pos vector
---@return vector
function octchunk.snap_to_center(pos)
	local size = octchunk.SIZE
	local half_size = size / 2
	return {
		x = math.floor(pos.x / size) * size + half_size,
		y = math.floor(pos.y / size) * size + half_size,
		z = math.floor(pos.z / size) * size + half_size
	}
end



---Serialize an octree to compressed binary string.
---@param octnode Octchunk
---@return string
function octchunk.serialize(octnode)
	return compress(serialize(octnode), "zstd", ZSTD_LEVEL_TREE)
end



---Deserialize a compressed octree string.
---@param string string Compressed binary data
---@param opts? {use_cache: boolean}
---@return Octchunk
function octchunk.deserialize(string, opts)
	local use_cache = true
	if opts ~= nil and opts.use_cache == false then
		use_cache = false
	end

	local tree = deserialize(decompress(string, "zstd"))

	if use_cache then
		octcache.create(tree)
		octcache.use(tree)
	end
	return tree
end



---Remap all content_id values in an octree in-place.
---Useful when a snapshot was serialized using old engine content_ids
---and needs to be made compatible with the current server session.
---@param tree Octchunk
---@param resolve_cid fun(old_cid: integer, ctx?: any): integer
---@param ctx? any
function octchunk.remap_tree_ids(tree, resolve_cid, ctx)
	local stack = {tree}
	while #stack > 0 do
		local node = stack[#stack]
		stack[#stack] = nil

		local cid = node[ID]
		if cid ~= nil then
			if ctx ~= nil then
				node[ID] = resolve_cid(cid, ctx)
			else
				node[ID] = resolve_cid(cid)
			end
		end

		local children = node[CHILDREN]
		if children then
			for i = 1, 8 do
				local child = children[i]
				if child then
					stack[#stack + 1] = child
				end
			end
		end
	end
end



---Merge identical children into parent (called during DFS backtrack).
---@param octnode OctNode
function merge_nodes(octnode)
	local children = octnode[CHILDREN]
	if not children then return end

	-- Count occurrences of each child's (ID, param1, param2)
	-- Use a numeric combined key for speed: cid*65536 + param1*256 + param2
	local pair_counts = {}
	local most_common_id, most_common_param1, most_common_param2, max_count
	for i = 1, 8 do
		local child = children[i]
		if child then
			local cid = child[ID]
			local p1 = child[PARAM1] or 0
			local p2 = child[PARAM2] or 0
			local key = cid * 65536 + p1 * 256 + p2
			local new_count = (pair_counts[key] or 0) + 1
			pair_counts[key] = new_count
			if not max_count or new_count > max_count then
				most_common_id = cid
				most_common_param1 = p1
				most_common_param2 = p2
				max_count = new_count
			end
		end
	end

	octnode[ID] = most_common_id
	if most_common_param1 and most_common_param1 ~= 0 then
		octnode[PARAM1] = most_common_param1
	else
		octnode[PARAM1] = nil
	end
	if most_common_param2 and most_common_param2 ~= 0 then
		octnode[PARAM2] = most_common_param2
	else
		octnode[PARAM2] = nil
	end

	-- Remove leaf children that match parent
	local has_remaining = false
	for i = 1, 8 do
		local child = children[i]
		if child then
			local child_p1 = child[PARAM1] or 0
			local child_p2 = child[PARAM2] or 0
			if not child[CHILDREN] and child[ID] == most_common_id and child_p1 == (most_common_param1 or 0) and child_p2 == (most_common_param2 or 0) then
				children[i] = nil
			else
				has_remaining = true
			end
		end
	end

	if not has_remaining then
		octnode[CHILDREN] = nil
	end
end



---Get the center of a subnode based on the index and center of the parent.
---Returns 3 numbers (cx, cy, cz) instead of a table for performance.
---@param px number
---@param py number
---@param pz number
---@param parent_size number
---@param index integer
---@return number cx
---@return number cy
---@return number cz
function get_child_center_coords(px, py, pz, parent_size, index)
	local offset = parent_size / 4
	local off = CHILD_OFFSETS[index]
	return px + off[1] * offset, py + off[2] * offset, pz + off[3] * offset
end



---Populate a subtree from voxel buffers.
---@param octnode OctNode
---@param cx number
---@param cy number
---@param cz number
---@param size number
---@param area VoxelArea
---@param data integer[]
---@param param2_data integer[]|nil
---@param param1_data integer[]|nil
function explore_area(octnode, cx, cy, cz, size, area, data, param2_data, param1_data)
	local area_index = area.index

	-- Base case: single voxel
	if size == 1 then
		local idx = area_index(area, cx, cy, cz)
		octnode[ID] = data[idx]
		local p1 = param1_data and (param1_data[idx] or 0) or 0
		if p1 ~= 0 then
			octnode[PARAM1] = p1
		end
		local p2 = param2_data and (param2_data[idx] or 0) or 0
		if p2 ~= 0 then
			octnode[PARAM2] = p2
		end
		return
	end

	-- Base case size==2: read 2x2x2 voxels directly (avoids 8 recursive calls with float centers)
	if size == 2 then
		local x0, y0, z0 = cx - 1, cy - 1, cz - 1
		local ids = {}
		local p1s = {}
		local p2s = {}
		do
			local i = 1
			for dz = 0, 1 do
				for dy = 0, 1 do
					for dx = 0, 1 do
						local idx = area_index(area, x0 + dx, y0 + dy, z0 + dz)
						ids[i] = data[idx]
						p1s[i] = param1_data and (param1_data[idx] or 0) or 0
						p2s[i] = param2_data and (param2_data[idx] or 0) or 0
						i = i + 1
					end
				end
			end
		end

		-- Check uniformity
		local first_id = ids[1]
		local first_p1 = p1s[1]
		local first_p2 = p2s[1]
		local uniform = true
		for i = 2, 8 do
			if ids[i] ~= first_id or p1s[i] ~= first_p1 or p2s[i] ~= first_p2 then
				uniform = false
				break
			end
		end

		if uniform then
			octnode[ID] = first_id
			if first_p1 ~= 0 then
				octnode[PARAM1] = first_p1
			end
			if first_p2 ~= 0 then
				octnode[PARAM2] = first_p2
			end
			return
		end

		-- Not uniform: create children as leaves, then merge
		octnode[CHILDREN] = {}
		for i = 1, 8 do
			local child = {[ID] = ids[i]}
			local p1 = p1s[i]
			if p1 ~= 0 then
				child[PARAM1] = p1
			end
			local p2 = p2s[i]
			if p2 ~= 0 then
				child[PARAM2] = p2
			end
			octnode[CHILDREN][i] = child
		end
		merge_nodes(octnode)
		return
	end

	-- Fast path size==8: check 8x8x8 uniformity with iterator
	if size == 8 then
		local half = 4
		local min_x, min_y, min_z = cx - half, cy - half, cz - half
		local max_x, max_y, max_z = cx + half - 1, cy + half - 1, cz + half - 1

		local first_idx = area_index(area, min_x, min_y, min_z)
		local first_id = data[first_idx]
		local first_p1 = param1_data and (param1_data[first_idx] or 0) or 0
		local first_p2 = param2_data and (param2_data[first_idx] or 0) or 0

		local uniform = true
		for z = min_z, max_z do
			for y = min_y, max_y do
				for x = min_x, max_x do
					local idx = area_index(area, x, y, z)
					local id = data[idx]
					local p1 = param1_data and (param1_data[idx] or 0) or 0
					local p2 = param2_data and (param2_data[idx] or 0) or 0
					if id ~= first_id or p1 ~= first_p1 or p2 ~= first_p2 then
						uniform = false
						break
					end
				end
				if not uniform then
					break
				end
			end
			if not uniform then
				break
			end
		end

		if uniform then
			octnode[ID] = first_id
			if first_p1 ~= 0 then
				octnode[PARAM1] = first_p1
			end
			if first_p2 ~= 0 then
				octnode[PARAM2] = first_p2
			end
			return
		end
	end

	-- Recursive case: subdivide and explore children
	local child_size = size / 2
	local has_children = false
	for i = 1, 8 do
		local child_cx, child_cy, child_cz = get_child_center_coords(cx, cy, cz, size, i)
		child_cx, child_cy, child_cz = floor(child_cx), floor(child_cy), floor(child_cz)
		local child = {}
		explore_area(child, child_cx, child_cy, child_cz, child_size, area, data, param2_data, param1_data)
		if child[ID] or child[CHILDREN] then
			octnode[CHILDREN] = octnode[CHILDREN] or {}
			octnode[CHILDREN][i] = child
			has_children = true
		end
	end

	if not has_children then
		return
	end

	-- Merge identical children to parent if possible
	merge_nodes(octnode)
end



---Fill intersection between requested bounds and a node box.
---@param area VoxelArea
---@param out_data integer[]|nil
---@param out_param2 integer[]|nil
---@param out_param1 integer[]|nil
---@param min_x number
---@param min_y number
---@param min_z number
---@param max_x number
---@param max_y number
---@param max_z number
---@param req_min vector
---@param req_max vector
---@param cid integer
---@param p2 integer
---@param p1 integer
---@param resolve_cid? fun(cid: integer): integer
function fill_intersection(area, out_data, out_param2, out_param1, min_x, min_y, min_z, max_x, max_y, max_z, req_min, req_max, cid, p2, p1, resolve_cid)
	local ix1 = min_x
	local iy1 = min_y
	local iz1 = min_z
	local ix2 = max_x
	local iy2 = max_y
	local iz2 = max_z

	if ix1 < req_min.x then ix1 = req_min.x end
	if iy1 < req_min.y then iy1 = req_min.y end
	if iz1 < req_min.z then iz1 = req_min.z end
	if ix2 > req_max.x then ix2 = req_max.x end
	if iy2 > req_max.y then iy2 = req_max.y end
	if iz2 > req_max.z then iz2 = req_max.z end

	if ix1 > ix2 or iy1 > iy2 or iz1 > iz2 then
		return
	end

	local value = cid
	if resolve_cid then
		value = resolve_cid(cid)
	end

	for z = iz1, iz2 do
		for y = iy1, iy2 do
			for x = ix1, ix2 do
				local idx = area:index(x, y, z)
				if out_data then
					out_data[idx] = value
				end
				if out_param2 then
					out_param2[idx] = p2
				end
				if out_param1 then
					out_param1[idx] = p1
				end
			end
		end
	end
end



---Return content id unchanged.
---@param cid integer
---@return integer
function identity_cid(cid)
	return cid
end



---Fill node values into output buffers within requested bounds.
---@param ctx FillCtx
---@param node OctNode
---@param cx number
---@param cy number
---@param cz number
---@param size number
function fill_node_impl(ctx, node, cx, cy, cz, size)
	local req_min = ctx.req_min
	local req_max = ctx.req_max
	local area = ctx.area
	local out_data = ctx.out_data
	local out_param2 = ctx.out_param2
	local out_param1 = ctx.out_param1
	local resolve_value = ctx.resolve_value
	local resolve_cid = ctx.resolve_cid

	-- size==1 would produce fractional bounds with the generic formula.
	-- Handle it explicitly so we only write integer coordinates.
	if size == 1 then
		if cx < req_min.x or cy < req_min.y or cz < req_min.z or cx > req_max.x or cy > req_max.y or cz > req_max.z then
			return
		end
		local cid = node[ID]
		local p2 = node[PARAM2] or 0
		local p1 = node[PARAM1] or 0
		local value = resolve_value(cid)
		local idx = area:index(cx, cy, cz)
		if out_data then
			out_data[idx] = value
		end
		if out_param2 then
			out_param2[idx] = p2
		end
		if out_param1 then
			out_param1[idx] = p1
		end
		return
	end

	local half = size / 2
	local min_x, min_y, min_z = cx - half, cy - half, cz - half
	local max_x, max_y, max_z = cx + half - 1, cy + half - 1, cz + half - 1
	if max_x < req_min.x or max_y < req_min.y or max_z < req_min.z then
		return
	end
	if min_x > req_max.x or min_y > req_max.y or min_z > req_max.z then
		return
	end

	local cid = node[ID]
	local p2 = node[PARAM2] or 0
	local p1 = node[PARAM1] or 0
	local children = node[CHILDREN]
	if not children then
		fill_intersection(area, out_data, out_param2, out_param1, min_x, min_y, min_z, max_x, max_y, max_z, req_min, req_max, cid, p2, p1, resolve_cid)
		return
	end

	local child_size = size / 2
	for i = 1, 8 do
		local child_cx, child_cy, child_cz = get_child_center_coords(cx, cy, cz, size, i)
		local child = children[i]
		if child then
			fill_node_impl(ctx, child, floor(child_cx), floor(child_cy), floor(child_cz), child_size)
		else
			local ccx, ccy, ccz = floor(child_cx), floor(child_cy), floor(child_cz)
			if child_size == 1 then
				fill_intersection(
					area,
					out_data,
					out_param2,
					out_param1,
					ccx,
					ccy,
					ccz,
					ccx,
					ccy,
					ccz,
					req_min,
					req_max,
					cid,
					p2,
					p1,
					resolve_cid
				)
			else
				local ch_half = child_size / 2
				fill_intersection(
					area,
					out_data,
					out_param2,
					out_param1,
					ccx - ch_half,
					ccy - ch_half,
					ccz - ch_half,
					ccx + ch_half - 1,
					ccy + ch_half - 1,
					ccz + ch_half - 1,
					req_min,
					req_max,
					cid,
					p2,
					p1,
					resolve_cid
				)
			end
		end
	end
end



---Emit all nodes inside a box via callback.
---@param ctx ScanCtx
---@param min_x number
---@param min_y number
---@param min_z number
---@param max_x number
---@param max_y number
---@param max_z number
---@param cid integer
---@param p2 integer
---@param p1 integer
function emit_box_impl(ctx, min_x, min_y, min_z, max_x, max_y, max_z, cid, p2, p1)
	local req_min = ctx.req_min
	local req_max = ctx.req_max
	if max_x < req_min.x or max_y < req_min.y or max_z < req_min.z then
		return
	end
	if min_x > req_max.x or min_y > req_max.y or min_z > req_max.z then
		return
	end

	local x1 = min_x < req_min.x and req_min.x or min_x
	local y1 = min_y < req_min.y and req_min.y or min_y
	local z1 = min_z < req_min.z and req_min.z or min_z
	local x2 = max_x > req_max.x and req_max.x or max_x
	local y2 = max_y > req_max.y and req_max.y or max_y
	local z2 = max_z > req_max.z and req_max.z or max_z
	if x1 > x2 or y1 > y2 or z1 > z2 then
		return
	end

	local value = ctx.resolve_value(cid)
	local cb = ctx.cb
	for z = z1, z2 do
		for y = y1, y2 do
			for x = x1, x2 do
				cb(x, y, z, value, p2, p1)
			end
		end
	end
end



---Scan nodes inside bounds and emit values via callback.
---@param ctx ScanCtx
---@param node OctNode
---@param cx number
---@param cy number
---@param cz number
---@param size number
function scan_node_impl(ctx, node, cx, cy, cz, size)
	local req_min = ctx.req_min
	local req_max = ctx.req_max
	if size == 1 then
		if cx < req_min.x or cy < req_min.y or cz < req_min.z or cx > req_max.x or cy > req_max.y or cz > req_max.z then
			return
		end
		local cid = node[ID]
		local p2 = node[PARAM2] or 0
		local p1 = node[PARAM1] or 0
		local value = ctx.resolve_value(cid)
		ctx.cb(cx, cy, cz, value, p2, p1)
		return
	end

	local half = size / 2
	local min_x, min_y, min_z = cx - half, cy - half, cz - half
	local max_x, max_y, max_z = cx + half - 1, cy + half - 1, cz + half - 1
	if max_x < req_min.x or max_y < req_min.y or max_z < req_min.z then
		return
	end
	if min_x > req_max.x or min_y > req_max.y or min_z > req_max.z then
		return
	end

	local cid = node[ID]
	local p2 = node[PARAM2] or 0
	local p1 = node[PARAM1] or 0
	local children = node[CHILDREN]
	if not children then
		emit_box_impl(ctx, min_x, min_y, min_z, max_x, max_y, max_z, cid, p2, p1)
		return
	end

	local child_size = size / 2
	for i = 1, 8 do
		local child_cx, child_cy, child_cz = get_child_center_coords(cx, cy, cz, size, i)
		local child = children[i]
		if child then
			scan_node_impl(ctx, child, floor(child_cx), floor(child_cy), floor(child_cz), child_size)
		else
			local ccx, ccy, ccz = floor(child_cx), floor(child_cy), floor(child_cz)
			if child_size == 1 then
				emit_box_impl(ctx, ccx, ccy, ccz, ccx, ccy, ccz, cid, p2, p1)
			else
				local ch_half = child_size / 2
				emit_box_impl(
					ctx,
					ccx - ch_half,
					ccy - ch_half,
					ccz - ch_half,
					ccx + ch_half - 1,
					ccy + ch_half - 1,
					ccz + ch_half - 1,
					cid,
					p2,
					p1
				)
			end
		end
	end
end
