local MapMethods = {}
octmap.MapMethods = MapMethods

local chunk_coords_from_world = map_octree.chunk_coords_from_world
local chunk_world_bounds = map_octree.chunk_world_bounds
local get_requested_bounds = map_octree.get_requested_bounds


---Get node name at a world position.
---@param map OctMap
---@param pos vector
---@return string|nil name Node name or nil if out of bounds
function octmap.get_node_name(map, pos)
	local req_min, req_max = get_requested_bounds(map)
	if pos.x < req_min.x or pos.y < req_min.y or pos.z < req_min.z
		or pos.x > req_max.x or pos.y > req_max.y or pos.z > req_max.z then
		return nil
	end
	local tree_snapped_pos = octchunk.snap_to_center(pos)

	local dist_from_min_tree = vector.floor(vector.new(
		(tree_snapped_pos.x - map.minp.x) / octchunk.SIZE,
		(tree_snapped_pos.y - map.minp.y) / octchunk.SIZE,
		(tree_snapped_pos.z - map.minp.z) / octchunk.SIZE
	))
	---@cast dist_from_min_tree vector

	local gx = 1 + dist_from_min_tree.x
	local gy = 1 + dist_from_min_tree.y
	local gz = 1 + dist_from_min_tree.z

	if gx < 1 or gx > map.trees.size.x or
		gy < 1 or gy > map.trees.size.y or
		gz < 1 or gz > map.trees.size.z then
		return nil
	end

	-- Check for sparse cell BEFORE cache lookup
	local cell = matrix3d.get(map.trees, gx, gy, gz)
	if cell == nil then
		return map.default_node
	end

	local tree = octmap._get_tree_cached(map, gx, gy, gz)
	---@cast tree Octchunk
	return octchunk.get_node_name(tree, pos, nil, nil, map.content_id_map)
end



---Get node content_id at a world position.
---@param map OctMap
---@param pos vector
---@return integer|nil content_id
function octmap.get_node_cid(map, pos)
	local req_min, req_max = get_requested_bounds(map)
	if pos.x < req_min.x or pos.y < req_min.y or pos.z < req_min.z
		or pos.x > req_max.x or pos.y > req_max.y or pos.z > req_max.z then
		return nil
	end
	local tree_snapped_pos = octchunk.snap_to_center(pos)

	local dist_from_min_tree = vector.floor(vector.new(
		(tree_snapped_pos.x - map.minp.x) / octchunk.SIZE,
		(tree_snapped_pos.y - map.minp.y) / octchunk.SIZE,
		(tree_snapped_pos.z - map.minp.z) / octchunk.SIZE
	))
	---@cast dist_from_min_tree vector

	local gx = 1 + dist_from_min_tree.x
	local gy = 1 + dist_from_min_tree.y
	local gz = 1 + dist_from_min_tree.z

	if gx < 1 or gx > map.trees.size.x or
		gy < 1 or gy > map.trees.size.y or
		gz < 1 or gz > map.trees.size.z then
		return nil
	end

	local cell = matrix3d.get(map.trees, gx, gy, gz)
	if cell == nil then
		if map.default_content_id ~= nil then
			return map.default_content_id
		end
		if map.default_node then
			local ok, cid = pcall(core.get_content_id, map.default_node)
			return ok and cid or nil
		end
		return nil
	end

	local tree = octmap._get_tree_cached(map, gx, gy, gz)
	---@cast tree Octchunk
	return octchunk.get_node_cid(tree, pos)
end



function MapMethods:get_emerged_area()
	return get_requested_bounds(self)
end



function MapMethods:size()
	local minp, maxp = self:get_emerged_area()
	return vector.new(
		maxp.x - minp.x + 1,
		maxp.y - minp.y + 1,
		maxp.z - minp.z + 1
	)
end



-- Convenience wrappers around map_octree.place()/place_async().
-- pos1 is the target emerged min corner; when omitted, places back to the original position.
function MapMethods:place(pos1, opts)
	return map_octree.place(self, pos1, opts)
end



-- callback(ok, err)
function MapMethods:place_async(pos1, opts, callback)
	if type(opts) == "function" and callback == nil then
		callback, opts = opts, nil
	end
	return map_octree.place_async(self, pos1, opts, callback)
end



-- Streaming scan over a box without allocating volume-sized arrays.
-- Calls cb(x, y, z, name, param2, param1) for each in-bounds node.
---@param pos1 vector
---@param pos2 vector
---@param cb fun(x: number, y: number, z: number, name: string, param2: integer, param1: integer)
---@return integer count
function MapMethods:for_each_node(pos1, pos2, cb)
	assert(pos1 and pos2, "pos1 and pos2 required")
	assert(type(cb) == "function", "callback required")
	pos1, pos2 = vector.sort(pos1, pos2)
	pos1 = assert(vector.round(pos1))
	pos2 = assert(vector.round(pos2))

	local e1, e2 = self:get_emerged_area()
	---@cast e1 vector
	---@cast e2 vector
	local minx = math.max(pos1.x, e1.x)
	local miny = math.max(pos1.y, e1.y)
	local minz = math.max(pos1.z, e1.z)
	local maxx = math.min(pos2.x, e2.x)
	local maxy = math.min(pos2.y, e2.y)
	local maxz = math.min(pos2.z, e2.z)

	if minx > maxx or miny > maxy or minz > maxz then
		return 0
	end

	local gx_min, gy_min, gz_min = chunk_coords_from_world(self.minp, minx, miny, minz)
	local gx_max, gy_max, gz_max = chunk_coords_from_world(self.minp, maxx, maxy, maxz)
	if gx_min < 1 then gx_min = 1 end
	if gy_min < 1 then gy_min = 1 end
	if gz_min < 1 then gz_min = 1 end
	if gx_max > self.trees.size.x then gx_max = self.trees.size.x end
	if gy_max > self.trees.size.y then gy_max = self.trees.size.y end
	if gz_max > self.trees.size.z then gz_max = self.trees.size.z end

	local count = 0
	local default_name = self.default_node
	local content_id_map = self.content_id_map
	for gz = gz_min, gz_max do
		for gy = gy_min, gy_max do
			for gx = gx_min, gx_max do
				local chunk_min, chunk_max = chunk_world_bounds(self.minp, gx, gy, gz)
				local req_min = {
					x = math.max(minx, chunk_min.x),
					y = math.max(miny, chunk_min.y),
					z = math.max(minz, chunk_min.z),
				}
				local req_max = {
					x = math.min(maxx, chunk_max.x),
					y = math.min(maxy, chunk_max.y),
					z = math.min(maxz, chunk_max.z),
				}

				local cell = matrix3d.get(self.trees, gx, gy, gz)
				if cell == nil then
					if default_name ~= nil then
						for z = req_min.z, req_max.z do
							for y = req_min.y, req_max.y do
								for x = req_min.x, req_max.x do
									count = count + 1
									cb(x, y, z, default_name, 0, 0)
								end
							end
						end
					end
				else
					local tree = octmap._get_tree_cached(self, gx, gy, gz)
					---@cast tree Octchunk
					octchunk.for_each_in_bounds(tree, req_min, req_max, function(x, y, z, cid, p2, p1)
						local name = (content_id_map and content_id_map[cid]) or core.get_name_from_content_id(cid)
						count = count + 1
						cb(x, y, z, name, p2, p1)
					end)
				end
			end
		end
	end
	return count
end



-- Streaming scan over a box without allocating volume-sized arrays.
-- Calls cb(x, y, z, content_id, param2, param1) for each in-bounds node.
---@param pos1 vector
---@param pos2 vector
---@param cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@return integer count
function MapMethods:for_each_node_cid(pos1, pos2, cb)
	assert(pos1 and pos2, "pos1 and pos2 required")
	assert(type(cb) == "function", "callback required")
	pos1, pos2 = vector.sort(pos1, pos2)
	pos1 = assert(vector.round(pos1))
	pos2 = assert(vector.round(pos2))

	local e1, e2 = self:get_emerged_area()
	assert(e1 and e2, "map has no emerged area")
	---@cast e1 vector
	---@cast e2 vector
	local minx = math.max(pos1.x, e1.x)
	local miny = math.max(pos1.y, e1.y)
	local minz = math.max(pos1.z, e1.z)
	local maxx = math.min(pos2.x, e2.x)
	local maxy = math.min(pos2.y, e2.y)
	local maxz = math.min(pos2.z, e2.z)

	if minx > maxx or miny > maxy or minz > maxz then
		return 0
	end

	local gx_min, gy_min, gz_min = chunk_coords_from_world(self.minp, minx, miny, minz)
	local gx_max, gy_max, gz_max = chunk_coords_from_world(self.minp, maxx, maxy, maxz)
	if gx_min < 1 then gx_min = 1 end
	if gy_min < 1 then gy_min = 1 end
	if gz_min < 1 then gz_min = 1 end
	if gx_max > self.trees.size.x then gx_max = self.trees.size.x end
	if gy_max > self.trees.size.y then gy_max = self.trees.size.y end
	if gz_max > self.trees.size.z then gz_max = self.trees.size.z end

	local count = 0
	local default_cid = assert(self.default_content_id, "default_content_id required")
	for gz = gz_min, gz_max do
		for gy = gy_min, gy_max do
			for gx = gx_min, gx_max do
				local chunk_min, chunk_max = chunk_world_bounds(self.minp, gx, gy, gz)
				local req_min = {
					x = math.max(minx, chunk_min.x),
					y = math.max(miny, chunk_min.y),
					z = math.max(minz, chunk_min.z),
				}
				local req_max = {
					x = math.min(maxx, chunk_max.x),
					y = math.min(maxy, chunk_max.y),
					z = math.min(maxz, chunk_max.z),
				}

				local cell = matrix3d.get(self.trees, gx, gy, gz)
				if cell == nil then
					for z = req_min.z, req_max.z do
						for y = req_min.y, req_max.y do
							for x = req_min.x, req_max.x do
								count = count + 1
								cb(x, y, z, default_cid, 0, 0)
							end
						end
					end
				else
					local tree = octmap._get_tree_cached(self, gx, gy, gz)
					---@cast tree Octchunk
					octchunk.for_each_in_bounds(tree, req_min, req_max, function(x, y, z, cid, p2, p1)
						count = count + 1
						cb(x, y, z, cid, p2, p1)
					end)
				end
			end
		end
	end
	return count
end



---Streaming scan over selected ranges (node-grid or chunk-grid).
---@param ranges OctreeRange[]
---@param cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@param opts? RangeIterateOpts
---@return integer count
function MapMethods:read_ranges(ranges, cb, opts)
	assert(type(ranges) == "table", "ranges required")
	assert(type(cb) == "function", "callback required")
	local count = 0
	local default_cid = assert(self.default_content_id, "default_content_id required")
	local map_req_min, map_req_max = get_requested_bounds(self)

	map_octree.iterate_chunk_ranges(self.minp, self.trees.size, ranges, opts, function(gx, gy, gz, _, _, req_min, req_max)
		local clip_min = {
			x = math.max(req_min.x, map_req_min.x),
			y = math.max(req_min.y, map_req_min.y),
			z = math.max(req_min.z, map_req_min.z),
		}
		local clip_max = {
			x = math.min(req_max.x, map_req_max.x),
			y = math.min(req_max.y, map_req_max.y),
			z = math.min(req_max.z, map_req_max.z),
		}
		if clip_min.x > clip_max.x or clip_min.y > clip_max.y or clip_min.z > clip_max.z then
			return
		end
		local cell = matrix3d.get(self.trees, gx, gy, gz)
		if cell == nil then
			for z = clip_min.z, clip_max.z do
				for y = clip_min.y, clip_max.y do
					for x = clip_min.x, clip_max.x do
						count = count + 1
						cb(x, y, z, default_cid, 0, 0)
					end
				end
			end
		else
			local tree = octmap._get_tree_cached(self, gx, gy, gz)
			---@cast tree Octchunk
			octchunk.for_each_in_bounds(tree, clip_min, clip_max, function(x, y, z, cid, p2, p1)
				count = count + 1
				cb(x, y, z, cid, p2, p1)
			end)
		end
	end)

	return count
end



---Iterate chunks by range (chunk-grid or node-grid selection).
---@param ranges OctreeRange[] List of ranges; default expects chunk-grid min/max vectors
---@param cb fun(chunk_min: vector, chunk_max: vector, gx: integer, gy: integer, gz: integer, tree: Octchunk|nil, is_sparse: boolean)
---@param opts? RangeIterateOpts
---@return integer count
function MapMethods:for_each_chunk_range(ranges, cb, opts)
	assert(type(ranges) == "table", "ranges required")
	assert(type(cb) == "function", "callback required")
	local grid = opts and opts.grid or "chunk"
	local count = 0
	local req_min, req_max = get_requested_bounds(self)

	map_octree.iterate_chunk_ranges(self.minp, self.trees.size, ranges, {grid = grid, dedup = opts and opts.dedup}, function(gx, gy, gz, chunk_min, chunk_max)
		local clip_min = {
			x = math.max(req_min.x, chunk_min.x),
			y = math.max(req_min.y, chunk_min.y),
			z = math.max(req_min.z, chunk_min.z),
		}
		local clip_max = {
			x = math.min(req_max.x, chunk_max.x),
			y = math.min(req_max.y, chunk_max.y),
			z = math.min(req_max.z, chunk_max.z),
		}
		if clip_min.x > clip_max.x or clip_min.y > clip_max.y or clip_min.z > clip_max.z then
			return
		end
		local cell = matrix3d.get(self.trees, gx, gy, gz)
		local tree = cell ~= nil and octmap._get_tree_cached(self, gx, gy, gz) or nil
		count = count + 1
		cb(chunk_min, chunk_max, gx, gy, gz, tree, cell == nil)
	end)

	return count
end



-- Fast single-node read that returns content_id instead of node name.
function MapMethods:get_node_cid_at(x, y, z)
	local size = octchunk.SIZE
	local half = size / 2
	x = math.floor(x)
	y = math.floor(y)
	z = math.floor(z)
	local req_min, req_max = get_requested_bounds(self)
	if x < req_min.x or y < req_min.y or z < req_min.z
		or x > req_max.x or y > req_max.y or z > req_max.z then
		return nil
	end

	local sx = math.floor(x / size) * size + half
	local sy = math.floor(y / size) * size + half
	local sz = math.floor(z / size) * size + half

	local gx = 1 + math.floor((sx - self.minp.x) / size)
	local gy = 1 + math.floor((sy - self.minp.y) / size)
	local gz = 1 + math.floor((sz - self.minp.z) / size)

	if gx < 1 or gx > self.trees.size.x or gy < 1 or gy > self.trees.size.y or gz < 1 or gz > self.trees.size.z then
		return nil
	end

	local cell = matrix3d.get(self.trees, gx, gy, gz)
	if cell == nil then
		return assert(self.default_content_id, "default_content_id required"), 0, 0
	end
	local tree = octmap._get_tree_cached(self, gx, gy, gz)
	---@cast tree Octchunk
	local cid, p2, p1 = octchunk.get_node_value_xyz(tree, x, y, z)
	return cid, p2, p1
end



function MapMethods:get_node_at(x, y, z)
	local size = octchunk.SIZE
	local half = size / 2
	x = math.floor(x)
	y = math.floor(y)
	z = math.floor(z)
	local req_min, req_max = get_requested_bounds(self)
	if x < req_min.x or y < req_min.y or z < req_min.z
		or x > req_max.x or y > req_max.y or z > req_max.z then
		return nil
	end

	local sx = math.floor(x / size) * size + half
	local sy = math.floor(y / size) * size + half
	local sz = math.floor(z / size) * size + half

	local gx = 1 + math.floor((sx - self.minp.x) / size)
	local gy = 1 + math.floor((sy - self.minp.y) / size)
	local gz = 1 + math.floor((sz - self.minp.z) / size)

	if gx < 1 or gx > self.trees.size.x or gy < 1 or gy > self.trees.size.y or gz < 1 or gz > self.trees.size.z then
		return nil
	end

	local cell = matrix3d.get(self.trees, gx, gy, gz)
	if cell == nil then
		return self.default_node, 0, 0
	end
	local tree = octmap._get_tree_cached(self, gx, gy, gz)
	---@cast tree Octchunk
	local cid, p2, p1 = octchunk.get_node_value_xyz(tree, x, y, z)
	local name = (self.content_id_map and self.content_id_map[cid]) or core.get_name_from_content_id(cid)
	return name, p2, p1
end



local MapMT = {__index = MapMethods}

---Attach OOP methods to a map table.
---@param map OctMap
---@return OctMap
function octmap.attach_methods(map)
	return setmetatable(map, MapMT)
end
