--- Debug logging and visualization for map_octree_lib.
-- Controlled by map_octree.log_enabled and map_octree.wireframe_enabled.

local core_debug = core.debug



---Log a debug message (only when log_enabled is true).
---@param msg string
function map_octree.debug(msg)
	if map_octree.log_enabled then
		core_debug("[map_octree] " .. msg)
	end
end



---Log a verbose message (only when verbose_enabled is true).
---@param msg string
function map_octree.verbose(msg)
	if map_octree.verbose_enabled then
		core.log("verbose", "[map_octree] " .. msg)
	end
end



--#region debug_only
core.register_entity("map_octree:wireframe", {
	initial_properties = {
		visual = "cube",
		textures = {
			"mapoct_wireframe.png", "mapoct_wireframe.png", "mapoct_wireframe.png",
			"mapoct_wireframe.png", "mapoct_wireframe.png", "mapoct_wireframe.png"
		},
		use_texture_alpha = true,
		static_save = false,
		visual_size = {x = 64, y = 64, z = 64}
	},
	timer = 0,
	on_step = function(self, dtime)
		self.timer = self.timer + dtime
		if self.timer > 60 then
			self.object:remove()
		end
	end
})

core.register_entity("map_octree:node_marker", {
	initial_properties = {
		visual = "cube",
		textures = {
			"mapoct_wireframe.png", "mapoct_wireframe.png", "mapoct_wireframe.png",
			"mapoct_wireframe.png", "mapoct_wireframe.png", "mapoct_wireframe.png"
		},
		use_texture_alpha = true,
		static_save = false,
		visual_size = {x = 1, y = 1, z = 1}
	},
	timer = 0,
	on_step = function(self, dtime)
		self.timer = self.timer + dtime
		if self.timer > 5 then
			self.object:remove()
		end
	end
})
--#endregion


---Add a wireframe visualization cube (debug only).
---@param pos vector
---@param size? number
---@return ObjectRef?
function map_octree.add_wireframe(pos, size)
	if not map_octree.wireframe_enabled then return end
	local e = core.add_entity(pos, "map_octree:wireframe")
	if e and size then
		---@diagnostic disable-next-line: missing-fields
		e:set_properties({visual_size = {x = size, y = size, z = size}})
	end
	return e
end



---Add a node marker visualization cube (debug only).
---@param pos vector
---@return ObjectRef?
function map_octree.add_node_marker(pos)
	if not map_octree.wireframe_enabled then return end
	return core.add_entity(pos, "map_octree:node_marker")
end



---Get current Lua memory usage in KB.
---@return number
function map_octree.get_lua_mem_kb()
	return collectgarbage("count")
end



---Estimate memory usage of a snapshot (for debugging).
---@param snapshot OctMap?
---@return integer chunk_count
---@return integer blob_bytes
---@return integer obj_count
function map_octree.estimate_snapshot_mem(snapshot)
	if not snapshot or not snapshot.trees then
		return 0, 0, 0
	end

	local chunk_count = 0
	local blob_bytes = 0
	local obj_count = 0

	matrix3d.iterate(snapshot.trees, function(x, y, z)
		local cell = matrix3d.get(snapshot.trees, x, y, z)
		if cell then
			chunk_count = chunk_count + 1
			if type(cell) == "string" then
				blob_bytes = blob_bytes + #cell
			else
				obj_count = obj_count + 1
			end
		end
	end)

	return chunk_count, blob_bytes, obj_count
end



--#region debug_only
---Log memory stats (forces GC collection first).
---@param label? string
function map_octree.log_mem_stats(label)
	collectgarbage("collect")
	local lua_kb = map_octree.get_lua_mem_kb()
	map_octree.debug(string.format("[MEM %s] Lua: %.1f MB", label or "", lua_kb / 1024))
end



--#endregion
