--- Serialization format migrations.

map_octree = map_octree or {}
map_octree.serialization = map_octree.serialization or {}

local M = map_octree.serialization

M.MAGIC_PREFIX = "MOTM"
M.LATEST_VERSION = 2

local migrations = {}



---Register a migration function from a specific version.
---@param from_version integer
---@param fn fun(map: OctMap): OctMap|nil
function M.register_migration(from_version, fn)
    assert(type(from_version) == "number", "from_version must be a number")
    assert(type(fn) == "function", "migration fn required")
    migrations[from_version] = fn
end



---Return magic header for a format version.
---@param version integer
---@return string
function M.make_header(version)
    return M.MAGIC_PREFIX .. string.char(version)
end



---Apply migrations from the given version to the latest.
---@param map OctMap
---@param from_version integer
---@return OctMap|nil
---@return string|nil
function M.migrate(map, from_version)
    local version = tonumber(from_version) or 0
    if version <= 0 then
        return nil, "invalid format version"
    end
    if version > M.LATEST_VERSION then
        return nil, "unsupported future format"
    end

    while version < M.LATEST_VERSION do
        local fn = migrations[version]
        if not fn then
            return nil, "missing migration for v" .. tostring(version)
        end
        map = fn(map) or map
        version = version + 1
    end
    return map
end



M.register_migration(1, function(map)
    local half = octchunk.SIZE / 2
    map.requested_pos1 = vector.subtract(map.minp, {x = half, y = half, z = half})
    map.requested_pos2 = vector.add(map.maxp, {x = half - 1, y = half - 1, z = half - 1})
    return map
end)
