local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

local function get_chunk_bounds(center)
	local half = octchunk.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})
	return pmin, pmax
end


map_octree.tests.register("map api: place wrapper supports opts", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = vector.round(ctx.player:get_pos())
	local src_center = octchunk.snap_to_center(vector.add(base, {x = 64, y = 0, z = 9728}))
	local dst_center = vector.add(src_center, {x = octchunk.SIZE * 2, y = 0, z = 0})

	local src_min, src_max = get_chunk_bounds(src_center)
	local dst_min, dst_max = get_chunk_bounds(dst_center)

	local all_min = vector.new(
		math.min(src_min.x, dst_min.x),
		math.min(src_min.y, dst_min.y),
		math.min(src_min.z, dst_min.z)
	)
	local all_max = vector.new(
		math.max(src_max.x, dst_max.x),
		math.max(src_max.y, dst_max.y),
		math.max(src_max.z, dst_max.z)
	)

	local node_a = "default:dirt"
	local cid_a = core.get_content_id(node_a)
	local cid_air = core.get_content_id("air")
	assert(type(cid_a) == "number", "missing node: " .. node_a)
	assert(type(cid_air) == "number", "missing node: air")

	local marker = vector.add(src_min, {x = 2, y = 3, z = 4})
	local dst_marker = vector.add(marker, vector.subtract(dst_min, src_min))

	testutil.with_voxel_region(all_min, all_max, function(manip, area, data, param2_data)
		for z = all_min.z, all_max.z do
			for y = all_min.y, all_max.y do
				for x = all_min.x, all_max.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end

		data[area:index(marker.x, marker.y, marker.z)] = cid_a
		if param2_data then param2_data[area:index(marker.x, marker.y, marker.z)] = 5 end

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(false)

		local map = octmap.new(src_min, src_max, {
			store_chunk_blobs = true,
			cache_mb = 4,
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		})

		assert(map:place(dst_min, {
			force_batches = true,
			cache_mb = 4,
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		}), "map:place failed")

		core.load_area(dst_min, dst_max)
		local vm = core.get_voxel_manip()
		local e1, e2 = vm:read_from_map(dst_min, dst_max)
		local va = VoxelArea(e1, e2)
		local out = {}
		local out_p2 = {}
		vm:get_data(out)
		if vm.get_param2_data then
			vm:get_param2_data(out_p2)
		end
		if vm.close then vm:close() end

		local idx = va:index(dst_marker.x, dst_marker.y, dst_marker.z)
		assert(out[idx] == cid_a, "map:place marker mismatch")
		assert((out_p2[idx] or 0) == 5, "map:place param2 mismatch")
	end)
end)


map_octree.tests.register("map api: place_async wrapper supports opts", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("map:place_async wrapper") then
		return
	end

	local base = vector.round(ctx.player:get_pos())
	local src_center = octchunk.snap_to_center(vector.add(base, {x = 96, y = 0, z = 9728}))
	local dst_center = vector.add(src_center, {x = octchunk.SIZE * 2, y = 0, z = 0})

	local src_min, src_max = get_chunk_bounds(src_center)
	local dst_min, dst_max = get_chunk_bounds(dst_center)

	local all_min = vector.new(
		math.min(src_min.x, dst_min.x),
		math.min(src_min.y, dst_min.y),
		math.min(src_min.z, dst_min.z)
	)
	local all_max = vector.new(
		math.max(src_max.x, dst_max.x),
		math.max(src_max.y, dst_max.y),
		math.max(src_max.z, dst_max.z)
	)

	local node_a = "default:cobble"
	local cid_a = core.get_content_id(node_a)
	local cid_air = core.get_content_id("air")
	assert(type(cid_a) == "number", "missing node: " .. node_a)
	assert(type(cid_air) == "number", "missing node: air")

	local marker = vector.add(src_min, {x = 5, y = 2, z = 1})
	local dst_marker = vector.add(marker, vector.subtract(dst_min, src_min))

	local test_id = "map_place_async_wrapper_" .. os.time()
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	testutil.with_voxel_region_async(all_min, all_max, function(manip, area, data, param2_data, restore)
		for z = all_min.z, all_max.z do
			for y = all_min.y, all_max.y do
				for x = all_min.x, all_max.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end

		data[area:index(marker.x, marker.y, marker.z)] = cid_a
		if param2_data then param2_data[area:index(marker.x, marker.y, marker.z)] = 9 end

		manip:set_data(data)
		manip:set_param2_data(param2_data)
		manip:write_to_map(false)

		local map = octmap.new(src_min, src_max, {
			store_chunk_blobs = true,
			cache_mb = 4,
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		})

		local function finish(ok, err)
			if not ok then
				core.log("error", "[test] " .. test_id .. ": " .. tostring(err))
				if end_async then end_async() end
				restore()
				return
			end

			core.load_area(dst_min, dst_max)
			local vm = core.get_voxel_manip()
			local e1, e2 = vm:read_from_map(dst_min, dst_max)
			local va = VoxelArea(e1, e2)
			local out = {}
			local out_p2 = {}
			vm:get_data(out)
			if vm.get_param2_data then
				vm:get_param2_data(out_p2)
			end
			if vm.close then vm:close() end

			local idx = va:index(dst_marker.x, dst_marker.y, dst_marker.z)
			if out[idx] ~= cid_a then
				core.log("error", "[test] " .. test_id .. ": map:place_async marker mismatch")
				if end_async then end_async() end
				restore()
				return
			end
			if (out_p2[idx] or 0) ~= 9 then
				core.log("error", "[test] " .. test_id .. ": map:place_async param2 mismatch")
				if end_async then end_async() end
				restore()
				return
			end

			if end_async then end_async() end
			restore()
		end

		map:place_async(dst_min, {
			min_delay = 0,
			force_batches = true,
			cache_mb = 4,
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		}, finish)
	end)
end)