local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

-- Async tests cannot block - results are logged via core.after callbacks
local async_results = {}


map_octree.tests.register("place: sync places snapshot at offset", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = vector.round(ctx.player:get_pos())
	local src_center = octchunk.snap_to_center(vector.add(base, {x = 64, y = 0, z = 64}))
	local half = octchunk.SIZE / 2

	local src_min = vector.subtract(src_center, {x = half, y = half, z = half})
	local src_max = vector.add(src_center, {x = half - 1, y = half - 1, z = half - 1})
	local delta = {x = octchunk.SIZE * 4, y = 0, z = 0}
	local dst_min = vector.add(src_min, delta)
	local dst_max = vector.add(src_max, delta)

	local all_min = vector.new(
		math.min(src_min.x, dst_min.x),
		math.min(src_min.y, dst_min.y),
		math.min(src_min.z, dst_min.z)
	)
	local all_max = vector.new(
		math.max(src_max.x, dst_max.x),
		math.max(src_max.y, dst_max.y),
		math.max(src_max.z, dst_max.z)
	)

	local node_fill = "default:stone"
	local node_a = "default:dirt"
	local node_b = "default:cobble"
	local cid_fill = core.get_content_id(node_fill)
	local cid_a = core.get_content_id(node_a)
	local cid_b = core.get_content_id(node_b)
	local cid_air = core.get_content_id("air")
	assert(type(cid_fill) == "number", "missing node: " .. node_fill)
	assert(type(cid_a) == "number", "missing node: " .. node_a)
	assert(type(cid_b) == "number", "missing node: " .. node_b)
	assert(type(cid_air) == "number", "missing node: air")

	local m1 = vector.add(src_min, {x = 1, y = 2, z = 3})
	local m2 = vector.add(src_min, {x = 8, y = 7, z = 6})
	local dst_m1 = vector.add(m1, delta)
	local dst_m2 = vector.add(m2, delta)

	testutil.with_voxel_region(all_min, all_max, function(manip, area, data, param2_data)
		-- Prepare src pattern and clear dst
		for z = all_min.z, all_max.z do
			for y = all_min.y, all_max.y do
				for x = all_min.x, all_max.x do
					local idx = area:index(x, y, z)
					if x >= src_min.x and x <= src_max.x and y >= src_min.y and y <= src_max.y and z >= src_min.z and z <= src_max.z then
						data[idx] = cid_fill
						if param2_data then param2_data[idx] = 0 end
					else
						data[idx] = cid_air
						if param2_data then param2_data[idx] = 0 end
					end
				end
			end
		end

		local idx1 = area:index(m1.x, m1.y, m1.z)
		data[idx1] = cid_a
		if param2_data then param2_data[idx1] = 5 end
		local idx2 = area:index(m2.x, m2.y, m2.z)
		data[idx2] = cid_b
		if param2_data then param2_data[idx2] = 7 end

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(false)

		local map = octmap.new(src_min, src_max, {
			store_chunk_blobs = true,
			cache_mb = 8,
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		})

		-- Sanity: ensure the snapshot actually captured the markers.
		local s1, sp1 = map:get_node_at(m1.x, m1.y, m1.z)
		assert(s1 == node_a and (sp1 or 0) == 5, "snapshot marker1 mismatch")
		local s2, sp2 = map:get_node_at(m2.x, m2.y, m2.z)
		assert(s2 == node_b and (sp2 or 0) == 7, "snapshot marker2 mismatch")

		assert(map_octree.place(map, dst_min, {
			max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		}), "place failed")

		-- Verify via VoxelManip readback
		core.load_area(dst_min, dst_max)
		local vm = core.get_voxel_manip()
		local e1, e2 = vm:read_from_map(dst_min, dst_max)
		local va = VoxelArea(e1, e2)
		local out = {}
		local out_p2 = {}
		vm:get_data(out)
		if vm.get_param2_data then
			vm:get_param2_data(out_p2)
		end
		if vm.close then vm:close() end

		local oidx1 = va:index(dst_m1.x, dst_m1.y, dst_m1.z)
		if out[oidx1] ~= cid_a then
			-- Try to find the marker somewhere in the destination region to detect offset bugs.
			local found = nil
			for z = dst_min.z, dst_max.z do
				for y = dst_min.y, dst_max.y do
					for x = dst_min.x, dst_max.x do
						local idx = va:index(x, y, z)
						if out[idx] == cid_a then
							found = {x = x, y = y, z = z}
							break
						end
					end
					if found then break end
				end
				if found then break end
			end
			local got_name = tostring(core.get_name_from_content_id(out[oidx1]))
			if found then
				error("marker1 content_id mismatch: got=" .. got_name .. ", found at=" .. core.pos_to_string(found))
			end
			error("marker1 content_id mismatch: got=" .. got_name .. ", not found anywhere in dst")
		end
		assert((out_p2[oidx1] or 0) == 5, "marker1 param2 mismatch")

		local oidx2 = va:index(dst_m2.x, dst_m2.y, dst_m2.z)
		if out[oidx2] ~= cid_b then
			local found = nil
			for z = dst_min.z, dst_max.z do
				for y = dst_min.y, dst_max.y do
					for x = dst_min.x, dst_max.x do
						local idx = va:index(x, y, z)
						if out[idx] == cid_b then
							found = {x = x, y = y, z = z}
							break
						end
					end
					if found then break end
				end
				if found then break end
			end
			local got_name = tostring(core.get_name_from_content_id(out[oidx2]))
			if found then
				error("marker2 content_id mismatch: got=" .. got_name .. ", found at=" .. core.pos_to_string(found))
			end
			error("marker2 content_id mismatch: got=" .. got_name .. ", not found anywhere in dst")
		end
		assert((out_p2[oidx2] or 0) == 7, "marker2 param2 mismatch")
	end)
end)


map_octree.tests.register("place: async places snapshot at offset", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("place async") then
		return
	end

	local base = vector.round(ctx.player:get_pos())
	local src_center = octchunk.snap_to_center(vector.add(base, {x = 96, y = 0, z = 64}))
	local half = octchunk.SIZE / 2

	local src_min = vector.subtract(src_center, {x = half, y = half, z = half})
	local src_max = vector.add(src_center, {x = half - 1, y = half - 1, z = half - 1})
	local delta = {x = octchunk.SIZE * 4, y = 0, z = 0}
	local dst_min = vector.add(src_min, delta)
	local dst_max = vector.add(src_max, delta)

	local all_min = vector.new(
		math.min(src_min.x, dst_min.x),
		math.min(src_min.y, dst_min.y),
		math.min(src_min.z, dst_min.z)
	)
	local all_max = vector.new(
		math.max(src_max.x, dst_max.x),
		math.max(src_max.y, dst_max.y),
		math.max(src_max.z, dst_max.z)
	)

	local node_fill = "default:stone"
	local node_a = "default:dirt"
	local node_b = "default:cobble"
	local cid_fill = core.get_content_id(node_fill)
	local cid_a = core.get_content_id(node_a)
	local cid_b = core.get_content_id(node_b)
	local cid_air = core.get_content_id("air")
	assert(type(cid_fill) == "number", "missing node: " .. node_fill)
	assert(type(cid_a) == "number", "missing node: " .. node_a)
	assert(type(cid_b) == "number", "missing node: " .. node_b)
	assert(type(cid_air) == "number", "missing node: air")

	local m1 = vector.add(src_min, {x = 1, y = 2, z = 3})
	local m2 = vector.add(src_min, {x = 8, y = 7, z = 6})
	local dst_m1 = vector.add(m1, delta)
	local dst_m2 = vector.add(m2, delta)

	core.load_area(all_min, all_max)
	local manip = core.get_voxel_manip()
	local e1, e2 = manip:read_from_map(all_min, all_max)
	local area = VoxelArea(e1, e2)
	local data = {}
	local param2_data = {}
	manip:get_data(data)
	manip:get_param2_data(param2_data)

	-- Snapshot original for restore after async callback.
	local original = {}
	for i = 1, #data do
		original[i] = data[i]
	end
	local original_p2 = {}
	for i = 1, #param2_data do
		original_p2[i] = param2_data[i]
	end

	for z = all_min.z, all_max.z do
		for y = all_min.y, all_max.y do
			for x = all_min.x, all_max.x do
				local idx = area:index(x, y, z)
				if x >= src_min.x and x <= src_max.x and y >= src_min.y and y <= src_max.y and z >= src_min.z and z <= src_max.z then
					data[idx] = cid_fill
					param2_data[idx] = 0
				else
					data[idx] = cid_air
					param2_data[idx] = 0
				end
			end
		end
	end

	local idx1 = area:index(m1.x, m1.y, m1.z)
	data[idx1] = cid_a
	param2_data[idx1] = 5
	local idx2 = area:index(m2.x, m2.y, m2.z)
	data[idx2] = cid_b
	param2_data[idx2] = 7

	manip:set_data(data)
	manip:set_param2_data(param2_data)
	manip:write_to_map(false)
	manip:close()

	local map = octmap.new(src_min, src_max, {
		store_chunk_blobs = true,
		cache_mb = 8,
		max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
	})

	-- Sanity: ensure the snapshot actually captured the markers.
	local s1, sp1 = map:get_node_at(m1.x, m1.y, m1.z)
	assert(s1 == node_a and (sp1 or 0) == 5, "snapshot marker1 mismatch")
	local s2, sp2 = map:get_node_at(m2.x, m2.y, m2.z)
	assert(s2 == node_b and (sp2 or 0) == 7, "snapshot marker2 mismatch")

	local test_id = "place_async_" .. os.time()
	async_results[test_id] = {status = "pending", start = core.get_us_time()}
	local end_async = nil
	if map_octree.tests.async_start then
		end_async = map_octree.tests.async_start(test_id)
	end

	map_octree.place_async(map, dst_min, {min_delay = 0}, function(ok, err)
		local res = async_results[test_id]
		if not ok then
			res.status = "FAIL"
			res.error = tostring(err)
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			-- restore even on failure
			core.load_area(all_min, all_max)
			local rm = core.get_voxel_manip()
			rm:read_from_map(all_min, all_max)
			rm:set_data(original)
			if original_p2 and rm.set_param2_data then
				rm:set_param2_data(original_p2)
			end
			rm:write_to_map(false)
			if rm.close then rm:close() end
			return
		end

		core.load_area(dst_min, dst_max)
		local vm = core.get_voxel_manip()
		local r1, r2 = vm:read_from_map(dst_min, dst_max)
		local va = VoxelArea(r1, r2)
		local out = {}
		local out_p2 = {}
		vm:get_data(out)
		if vm.get_param2_data then
			vm:get_param2_data(out_p2)
		end
		if vm.close then vm:close() end

		local oidx1 = va:index(dst_m1.x, dst_m1.y, dst_m1.z)
		if out[oidx1] ~= cid_a then
			res.status = "FAIL"
			local got_name = tostring(core.get_name_from_content_id(out[oidx1]))
			local found = nil
			for z = dst_min.z, dst_max.z do
				for y = dst_min.y, dst_max.y do
					for x = dst_min.x, dst_max.x do
						local idx = va:index(x, y, z)
						if out[idx] == cid_a then
							found = {x = x, y = y, z = z}
							break
						end
					end
					if found then break end
				end
				if found then break end
			end
			if found then
				res.error = "marker1 mismatch: got=" .. got_name .. ", found at=" .. core.pos_to_string(found)
			else
				res.error = "marker1 mismatch: got=" .. got_name .. ", not found anywhere in dst"
			end
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end
		if (out_p2[oidx1] or 0) ~= 5 then
			res.status = "FAIL"
			res.error = "marker1 param2 mismatch"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		local oidx2 = va:index(dst_m2.x, dst_m2.y, dst_m2.z)
		if out[oidx2] ~= cid_b then
			res.status = "FAIL"
			local got_name = tostring(core.get_name_from_content_id(out[oidx2]))
			local found = nil
			for z = dst_min.z, dst_max.z do
				for y = dst_min.y, dst_max.y do
					for x = dst_min.x, dst_max.x do
						local idx = va:index(x, y, z)
						if out[idx] == cid_b then
							found = {x = x, y = y, z = z}
							break
						end
					end
					if found then break end
				end
				if found then break end
			end
			if found then
				res.error = "marker2 mismatch: got=" .. got_name .. ", found at=" .. core.pos_to_string(found)
			else
				res.error = "marker2 mismatch: got=" .. got_name .. ", not found anywhere in dst"
			end
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end
		if (out_p2[oidx2] or 0) ~= 7 then
			res.status = "FAIL"
			res.error = "marker2 param2 mismatch"
			core.log("error", "[test] " .. test_id .. ": " .. res.error)
			if end_async then end_async() end
			return
		end

		res.status = "PASS"
		local elapsed_ms = (core.get_us_time() - res.start) / 1000
		core.log("action", string.format("[test] %s: PASS - %.1fms", test_id, elapsed_ms))
		if end_async then end_async() end

		-- restore original area
		core.load_area(all_min, all_max)
		local rm = core.get_voxel_manip()
		rm:read_from_map(all_min, all_max)
		rm:set_data(original)
		if original_p2 and rm.set_param2_data then
			rm:set_param2_data(original_p2)
		end
		rm:write_to_map(false)
		if rm.close then rm:close() end
	end)

	core.log("action", "[test] " .. test_id .. ": async operation started (result will be logged)")
end)