--- OctreeManip: low-GC area snapshot with batched writes.
-- Reads never require volume-sized Lua arrays.
-- Writes are recorded as sparse deltas and applied in batches.
-- Writing uses VoxelManip per-chunk (octchunk.SIZE^3) so temporary tables are bounded.
-- Snapshot is refreshed only for touched chunks.
-- Async operations use time-budgeting (50ms per step) to keep server responsive.

--#region types
---@class OctreeManip
---@field pos1? vector Original read position 1
---@field pos2? vector Original read position 2
---@field _snapshot? OctMap The underlying octmap snapshot
---@field _grid_minp? vector Snapped grid origin
---@field _grid_maxp? vector Snapped grid max
---@field _dirty_trees string[] List of dirty chunk keys
---@field _tree_deltas table<string, table<integer, integer>> Chunk key to deltas
---@field _tree_delta_counts table<string, integer> Chunk key to delta count
---@field _write_inflight boolean Whether a write is in progress
---@field _write_callback_queue function[] Queued callbacks for write completion
---@field _apply_inflight boolean Whether an async apply is in progress

---@alias OctreeManipOpts {build_opts?: MapCreationOpts, frozen_snapshot?: boolean}

---@class ApplyAsyncContext
---@field self OctreeManip
---@field snap OctMap
---@field callback fun(ok: boolean, changed_count: integer, err?: string)
---@field fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil
---@field time_budget_us integer
---@field grid_minp vector
---@field gx_min integer
---@field gy_min integer
---@field gz_min integer
---@field gx_max integer
---@field gy_max integer
---@field gz_max integer
---@field minx integer
---@field miny integer
---@field minz integer
---@field maxx integer
---@field maxy integer
---@field maxz integer
---@field default_cid integer
---@field gx integer
---@field gy integer
---@field gz integer
---@field chunk_min vector
---@field chunk_max vector
---@field x integer
---@field y integer
---@field z integer
---@field changed integer
---@field step_fn fun()

---@class ApplyRangesAsyncTask
---@field gx integer
---@field gy integer
---@field gz integer
---@field chunk_min vector
---@field chunk_max vector
---@field req_min vector
---@field req_max vector

---@class ApplyRangesAsyncContext
---@field self OctreeManip
---@field snap OctMap
---@field callback fun(ok: boolean, changed_count: integer, err?: string)
---@field fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil
---@field default_cid integer
---@field time_budget_us integer
---@field tasks ApplyRangesAsyncTask[]
---@field task_idx integer
---@field x integer|nil
---@field y integer|nil
---@field z integer|nil
---@field changed integer
---@field step_fn fun()

--#endregion

local OctreeManip = {}
OctreeManip.__index = OctreeManip

local get_requested_bounds = map_octree.get_requested_bounds

local floor = math.floor

local DEFAULT_STEP_DELAY = 0.5
local SETTING_WRITE_BUDGET_MS = "map_octree_write_budget_ms"
local DEFAULT_WRITE_BUDGET_MS = 50

local get_async_budget_us
local sorted_pos
local chunk_coords_from_world = map_octree.chunk_coords_from_world
local local_index_in_chunk
local chunk_world_bounds = map_octree.chunk_world_bounds
local chunk_key
local parse_chunk_key
local pack_cid_p2
local unpack_cid_p2
local pack_cid_p2_p1
local unpack_cid_p2_p1
local build_write_batches
local apply_write_batch
local refresh_snapshot_if_needed
local resolve_default_content_id
local write_delta_cid
local apply_async_finish
local apply_async_step
local apply_async_step_impl
local apply_ranges_async_finish
local apply_ranges_async_step
local apply_ranges_async_step_impl
local get_write_limits
local compute_write_batch_size_nodes
local maybe_collect_write_gc

local BATCH_SIZE = 80


---Create a new OctreeManip instance.
---@param opts? OctreeManipOpts
---@return OctreeManip
function OctreeManip.new(opts)
	opts = opts or {}
	local self = setmetatable({}, OctreeManip)
	self._build_opts = opts.build_opts or {}
	self._frozen_snapshot = opts.frozen_snapshot == true
	self._dirty_trees = {}
	self._tree_deltas = {}
	self._tree_delta_counts = {}
	self._write_inflight = false
	self._write_callback_queue = {}
	return self
end



---Read a map region into the manipulator.
---@param pos1 vector
---@param pos2 vector
---@return vector pos1, vector pos2 Sorted positions
function OctreeManip:read_from_map(pos1, pos2)
	assert(pos1 and pos2, "pos1 and pos2 required")
	self.pos1, self.pos2 = sorted_pos(pos1, pos2)
	self._snapshot = octmap.new(self.pos1, self.pos2, self._build_opts)

	-- octmap.new() snaps its grid to chunk centers; store the snapped origin for
	-- all gx/gy/gz calculations so writes match the snapshot grid even when
	-- callers pass arbitrary corners (e.g. WorldEdit selections).
	self._grid_minp = self._snapshot.minp
	self._grid_maxp = self._snapshot.maxp
	self._dirty_trees = {}
	self._tree_deltas = {}
	self._tree_delta_counts = {}
	return self.pos1, self.pos2
end



---Async variant of read_from_map() that ensures the region has been generated.
---Required for far/never-visited areas: core.load_area() doesn't trigger mapgen.
---@param pos1 vector
---@param pos2 vector
---@param callback fun(ok: boolean, err?: string)
function OctreeManip:read_from_map_async(pos1, pos2, callback)
	assert(pos1 and pos2, "pos1 and pos2 required")
	assert(type(callback) == "function", "callback required")
	self.pos1, self.pos2 = sorted_pos(pos1, pos2)

	octmap.new_async(self.pos1, self.pos2, self._build_opts, function(map, err)
		if not map then
			callback(false, err)
			return
		end
		self._snapshot = map
		self._grid_minp = map.minp
		self._grid_maxp = map.maxp
		self._dirty_trees = {}
		self._tree_deltas = {}
		self._tree_delta_counts = {}
		callback(true)
	end)
end



---Get the emerged area bounds.
---@return vector|nil, vector|nil
function OctreeManip:get_emerged_area()
	if not self._snapshot then
		return nil, nil
	end
	return self._snapshot:get_emerged_area()
end



---Get the size of the emerged area.
---@return vector|nil
function OctreeManip:size()
	if not self._snapshot then
		return nil
	end
	return self._snapshot:size()
end



-- Low-GC single node read from snapshot (x,y,z, no tables)
---Get node at position (reads from snapshot, not live map).
---@param x number
---@param y number
---@param z number
---@return string|nil name, integer param2, integer param1
function OctreeManip:get_node_at(x, y, z)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	local req_min, req_max = get_requested_bounds(snap)
	if x < req_min.x or y < req_min.y or z < req_min.z
		or x > req_max.x or y > req_max.y or z > req_max.z then
		return nil, 0, 0
	end

	-- Overlay pending deltas on top of the snapshot so reads reflect staged edits.
	if self.pos1 and self._tree_deltas and next(self._tree_deltas) ~= nil then
		x, y, z = floor(x), floor(y), floor(z)
		local grid_minp = self._grid_minp or self.pos1
		local gx, gy, gz = chunk_coords_from_world(grid_minp, x, y, z)
		if gx >= 1 and gy >= 1 and gz >= 1
			and gx <= snap.trees.size.x and gy <= snap.trees.size.y and gz <= snap.trees.size.z then
			local key = chunk_key(gx, gy, gz)
			local deltas = self._tree_deltas[key]
			if deltas and next(deltas) then
				local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)
				if x >= chunk_min.x and x <= chunk_max.x
					and y >= chunk_min.y and y <= chunk_max.y
					and z >= chunk_min.z and z <= chunk_max.z then
					local idx0 = local_index_in_chunk(chunk_min, x, y, z)
					local packed = deltas[idx0]
					if packed ~= nil then
						local cid, p2, p1
						if packed < 0 then
							cid, p2, p1 = unpack_cid_p2_p1(packed)
						else
							cid, p2 = unpack_cid_p2(packed)
							local _, _, orig_p1 = snap:get_node_cid_at(x, y, z)
							p1 = orig_p1
						end

						local name = snap.content_id_map and snap.content_id_map[cid] or core.get_name_from_content_id(cid)
						return name, p2, p1
					end
				end
			end
		end
	end

	return snap:get_node_at(x, y, z)
end



---Iterate over nodes in a box (from snapshot).
---@param pos1 vector
---@param pos2 vector
---@param cb fun(x: number, y: number, z: number, name: string, param2: integer, param1: integer)
---@return integer count
function OctreeManip:for_each_node(pos1, pos2, cb)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	return snap:for_each_node(pos1, pos2, cb)
end



---Iterate nodes over selected ranges (from snapshot).
---@param ranges OctreeRange[]
---@param cb fun(x: number, y: number, z: number, content_id: integer, param2: integer, param1: integer)
---@param opts? RangeIterateOpts
---@return integer count
function OctreeManip:read_ranges(ranges, cb, opts)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	---@diagnostic disable: undefined-field
	return snap:read_ranges(ranges, cb, opts)
end



---Low-GC write intent: store (name,param2,param1) delta. Snapshot is not modified.
---@param x number
---@param y number
---@param z number
---@param name string Node name
---@param param2? integer
---@param param1? integer
---@return boolean success
function OctreeManip:set_node_at(x, y, z, name, param2, param1)
	assert(self._snapshot and self.pos1, "call :read_from_map(pos1, pos2) first")
	local req_min, req_max = get_requested_bounds(self._snapshot)
	if x < req_min.x or y < req_min.y or z < req_min.z
		or x > req_max.x or y > req_max.y or z > req_max.z then
		return false
	end
	assert(type(name) == "string", "name must be a node name")
	x, y, z = floor(x), floor(y), floor(z)
	local snap = assert(self._snapshot)
	local cid = core.get_content_id(name)
	assert(type(cid) == "number", "unknown node name: " .. name)

	local grid_minp = self._grid_minp or self.pos1
	local gx, gy, gz = chunk_coords_from_world(grid_minp, x, y, z)
	if gx < 1 or gy < 1 or gz < 1 or gx > self._snapshot.trees.size.x or gy > self._snapshot.trees.size.y or gz > self._snapshot.trees.size.z then
		return false
	end

	local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)
	if x < chunk_min.x or x > chunk_max.x or y < chunk_min.y or y > chunk_max.y or z < chunk_min.z or z > chunk_max.z then
		return false
	end

	local key = chunk_key(gx, gy, gz)
	local idx0 = local_index_in_chunk(chunk_min, x, y, z)

	-- If this write would not change the effective value, don't mark the tree dirty.
	local snap_cid, snap_p2, snap_p1 = snap:get_node_cid_at(x, y, z)
	if snap_cid ~= nil then
		local deltas = self._tree_deltas[key]
		local cur_cid, cur_p2, cur_p1 = snap_cid, snap_p2, snap_p1
		local existing = deltas and deltas[idx0]
		if existing ~= nil then
			if existing < 0 then
				cur_cid, cur_p2, cur_p1 = unpack_cid_p2_p1(existing)
			else
				cur_cid, cur_p2 = unpack_cid_p2(existing)
				cur_p1 = snap_p1
			end
		end

		local desired_p2 = tonumber(param2) or 0
		if desired_p2 < 0 then desired_p2 = 0 end
		if desired_p2 > 255 then desired_p2 = 255 end
		local desired_p1
		if param1 ~= nil then
			desired_p1 = tonumber(param1) or 0
			if desired_p1 < 0 then desired_p1 = 0 end
			if desired_p1 > 255 then desired_p1 = 255 end
		else
			-- param1 omitted: effective param1 comes from snapshot (clears any pending param1 delta)
			desired_p1 = tonumber(snap_p1) or 0
		end

		if cur_cid == cid and cur_p2 == desired_p2 and cur_p1 == desired_p1 then
			return true
		end
	end

	local packed = (param1 ~= nil) and pack_cid_p2_p1(cid, param2, param1) or pack_cid_p2(cid, param2)
	local deltas = self._tree_deltas[key]
	if not deltas then
		deltas = {}
		self._tree_deltas[key] = deltas
		self._tree_delta_counts[key] = 0
		self._dirty_trees[#self._dirty_trees + 1] = key
	end

	if deltas[idx0] == nil then
		self._tree_delta_counts[key] = self._tree_delta_counts[key] + 1
	end
	deltas[idx0] = packed
	return true
end



---Low-GC write intent: store (content_id,param2,param1) delta. Snapshot is not modified.
---@param x number
---@param y number
---@param z number
---@param cid integer Content ID
---@param param2? integer
---@param param1? integer
---@return boolean success
function OctreeManip:set_node_cid_at(x, y, z, cid, param2, param1)
	assert(self._snapshot and self.pos1, "call :read_from_map(pos1, pos2) first")
	local req_min, req_max = get_requested_bounds(self._snapshot)
	if x < req_min.x or y < req_min.y or z < req_min.z
		or x > req_max.x or y > req_max.y or z > req_max.z then
		return false
	end
	assert(type(cid) == "number", "cid must be a content_id")
	x, y, z = floor(x), floor(y), floor(z)
	local snap = assert(self._snapshot)

	local grid_minp = self._grid_minp or self.pos1
	local gx, gy, gz = chunk_coords_from_world(grid_minp, x, y, z)
	if gx < 1 or gy < 1 or gz < 1 or gx > self._snapshot.trees.size.x or gy > self._snapshot.trees.size.y or gz > self._snapshot.trees.size.z then
		return false
	end

	local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)
	if x < chunk_min.x or x > chunk_max.x or y < chunk_min.y or y > chunk_max.y or z < chunk_min.z or z > chunk_max.z then
		return false
	end

	local key = chunk_key(gx, gy, gz)
	local idx0 = local_index_in_chunk(chunk_min, x, y, z)

	-- If this write would not change the effective value, don't mark the tree dirty.
	local snap_cid, snap_p2, snap_p1 = snap:get_node_cid_at(x, y, z)
	if snap_cid ~= nil then
		local deltas = self._tree_deltas[key]
		local cur_cid, cur_p2, cur_p1 = snap_cid, snap_p2, snap_p1
		local existing = deltas and deltas[idx0]
		if existing ~= nil then
			if existing < 0 then
				cur_cid, cur_p2, cur_p1 = unpack_cid_p2_p1(existing)
			else
				cur_cid, cur_p2 = unpack_cid_p2(existing)
				cur_p1 = snap_p1
			end
		end

		local desired_p2 = tonumber(param2) or 0
		if desired_p2 < 0 then desired_p2 = 0 end
		if desired_p2 > 255 then desired_p2 = 255 end
		local desired_p1
		if param1 ~= nil then
			desired_p1 = tonumber(param1) or 0
			if desired_p1 < 0 then desired_p1 = 0 end
			if desired_p1 > 255 then desired_p1 = 255 end
		else
			desired_p1 = tonumber(snap_p1) or 0
		end

		if cur_cid == cid and cur_p2 == desired_p2 and cur_p1 == desired_p1 then
			return true
		end
	end

	local packed = (param1 ~= nil) and pack_cid_p2_p1(cid, param2, param1) or pack_cid_p2(cid, param2)
	local deltas = self._tree_deltas[key]
	if not deltas then
		deltas = {}
		self._tree_deltas[key] = deltas
		self._tree_delta_counts[key] = 0
		self._dirty_trees[#self._dirty_trees + 1] = key
	end

	if deltas[idx0] == nil then
		self._tree_delta_counts[key] = self._tree_delta_counts[key] + 1
	end
	deltas[idx0] = packed
	return true
end



---Resolve default content id for a map.
---@param map OctMap
---@return integer
function resolve_default_content_id(map)
	if map.default_content_id ~= nil then
		return map.default_content_id
	end
	local nm = map.default_node or "air"
	map.default_content_id = core.get_content_id(nm)
	return map.default_content_id
end



---Write a delta entry if it changes the effective value.
---@param self OctreeManip
---@param key string
---@param idx0 integer
---@param snap_cid integer|nil
---@param snap_p2 integer
---@param snap_p1 integer
---@param cid integer
---@param p2 integer
---@param p1 integer
---@return boolean
function write_delta_cid(self, key, idx0, snap_cid, snap_p2, snap_p1, cid, p2, p1)
	local desired_p2 = tonumber(p2) or 0
	if desired_p2 < 0 then desired_p2 = 0 end
	if desired_p2 > 255 then desired_p2 = 255 end
	local desired_p1 = tonumber(p1) or 0
	if desired_p1 < 0 then desired_p1 = 0 end
	if desired_p1 > 255 then desired_p1 = 255 end

	if snap_cid ~= nil then
		local deltas = self._tree_deltas[key]
		local cur_cid, cur_p2, cur_p1 = snap_cid, snap_p2, snap_p1
		local existing = deltas and deltas[idx0]
		if existing ~= nil then
			if existing < 0 then
				cur_cid, cur_p2, cur_p1 = unpack_cid_p2_p1(existing)
			else
				cur_cid, cur_p2 = unpack_cid_p2(existing)
				cur_p1 = snap_p1
			end
		end

		if cur_cid == cid and cur_p2 == desired_p2 and cur_p1 == desired_p1 then
			return false
		end
	end

	local packed = pack_cid_p2_p1(cid, desired_p2, desired_p1)
	local deltas = self._tree_deltas[key]
	if not deltas then
		deltas = {}
		self._tree_deltas[key] = deltas
		self._tree_delta_counts[key] = 0
		self._dirty_trees[#self._dirty_trees + 1] = key
	end

	if deltas[idx0] == nil then
		self._tree_delta_counts[key] = self._tree_delta_counts[key] + 1
	end
	deltas[idx0] = packed
	return true
end



---Generic bulk transform helper.
---@param pos1 vector
---@param pos2 vector
---@param fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil Transform function
---@return integer count Number of nodes changed
function OctreeManip:apply(pos1, pos2, fn)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	assert(pos1 and pos2, "pos1 and pos2 required")
	assert(type(fn) == "function", "fn required")

	pos1, pos2 = vector.sort(pos1, pos2)
	pos1 = assert(vector.round(pos1))
	pos2 = assert(vector.round(pos2))

	local e1, e2 = snap:get_emerged_area()
	if not e1 or not e2 then
		error("snapshot has no emerged area")
	end
	---@cast e1 vector
	---@cast e2 vector
	local minx = math.max(pos1.x, e1.x)
	local miny = math.max(pos1.y, e1.y)
	local minz = math.max(pos1.z, e1.z)
	local maxx = math.min(pos2.x, e2.x)
	local maxy = math.min(pos2.y, e2.y)
	local maxz = math.min(pos2.z, e2.z)

	if minx > maxx or miny > maxy or minz > maxz then
		return 0
	end

	local grid_minp = self._grid_minp or self.pos1
	local gx_min, gy_min, gz_min = chunk_coords_from_world(grid_minp, minx, miny, minz)
	local gx_max, gy_max, gz_max = chunk_coords_from_world(grid_minp, maxx, maxy, maxz)
	if gx_min < 1 then gx_min = 1 end
	if gy_min < 1 then gy_min = 1 end
	if gz_min < 1 then gz_min = 1 end
	if gx_max > snap.trees.size.x then gx_max = snap.trees.size.x end
	if gy_max > snap.trees.size.y then gy_max = snap.trees.size.y end
	if gz_max > snap.trees.size.z then gz_max = snap.trees.size.z end

	local default_cid = resolve_default_content_id(snap)

	local changed = 0
	for gz = gz_min, gz_max do
		for gy = gy_min, gy_max do
			for gx = gx_min, gx_max do
				local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)
				local req_min = {
					x = math.max(minx, chunk_min.x),
					y = math.max(miny, chunk_min.y),
					z = math.max(minz, chunk_min.z),
				}
				local req_max = {
					x = math.min(maxx, chunk_max.x),
					y = math.min(maxy, chunk_max.y),
					z = math.min(maxz, chunk_max.z),
				}

				local key = chunk_key(gx, gy, gz)
				local cell = matrix3d.get(snap.trees, gx, gy, gz)
				if cell == nil then
					for z = req_min.z, req_max.z do
						for y = req_min.y, req_max.y do
							for x = req_min.x, req_max.x do
								local cid, p2, p1 = default_cid, 0, 0
								local new_cid, new_p2, new_p1 = fn(x, y, z, cid, p2, p1)
								if new_cid ~= nil then
									new_p2 = new_p2 and tonumber(new_p2) or p2
									new_p1 = new_p1 and tonumber(new_p1) or p1
									local idx0 = local_index_in_chunk(chunk_min, x, y, z)
									if write_delta_cid(self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
										changed = changed + 1
									end
								end
							end
						end
					end
				else
					local tree = octmap._get_tree_cached(snap, gx, gy, gz)
					---@cast tree Octchunk
					octchunk.for_each_in_bounds(tree, req_min, req_max, function(x, y, z, cid, p2, p1)
						local new_cid, new_p2, new_p1 = fn(x, y, z, cid, p2, p1)
						if new_cid ~= nil then
							new_p2 = new_p2 and tonumber(new_p2) or p2
							new_p1 = new_p1 and tonumber(new_p1) or p1
							local idx0 = local_index_in_chunk(chunk_min, x, y, z)
							if write_delta_cid(self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
								changed = changed + 1
							end
						end
					end)
				end
			end
		end
	end

	return changed
end



---Apply a transform over selected ranges (node-grid or chunk-grid).
---@param ranges OctreeRange[]
---@param fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil
---@param opts? RangeIterateOpts
---@return integer count
function OctreeManip:apply_ranges(ranges, fn, opts)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	assert(type(ranges) == "table", "ranges required")
	assert(type(fn) == "function", "fn required")

	local grid_minp = self._grid_minp or self.pos1
	local default_cid = resolve_default_content_id(snap)
	local changed = 0

	map_octree.iterate_chunk_ranges(grid_minp, snap.trees.size, ranges, opts, function(gx, gy, gz, chunk_min, chunk_max, req_min, req_max)
		local key = chunk_key(gx, gy, gz)
		local cell = matrix3d.get(snap.trees, gx, gy, gz)
		if cell == nil then
			for z = req_min.z, req_max.z do
				for y = req_min.y, req_max.y do
					for x = req_min.x, req_max.x do
						local cid, p2, p1 = default_cid, 0, 0
						local new_cid, new_p2, new_p1 = fn(x, y, z, cid, p2, p1)
						if new_cid ~= nil then
							new_p2 = new_p2 and tonumber(new_p2) or p2
							new_p1 = new_p1 and tonumber(new_p1) or p1
							local idx0 = local_index_in_chunk(chunk_min, x, y, z)
							if write_delta_cid(self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
								changed = changed + 1
							end
						end
					end
				end
			end
		else
			local tree = octmap._get_tree_cached(snap, gx, gy, gz)
			---@cast tree Octchunk
			octchunk.for_each_in_bounds(tree, req_min, req_max, function(x, y, z, cid, p2, p1)
				local new_cid, new_p2, new_p1 = fn(x, y, z, cid, p2, p1)
				if new_cid ~= nil then
					new_p2 = new_p2 and tonumber(new_p2) or p2
					new_p1 = new_p1 and tonumber(new_p1) or p1
					local idx0 = local_index_in_chunk(chunk_min, x, y, z)
					if write_delta_cid(self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
						changed = changed + 1
					end
				end
			end)
		end
	end)

	return changed
end



---Async version of :apply().
---Processes a limited number of nodes per step and yields using core.after.
---callback(ok, changed_count, err)
---@async
---@param pos1 vector
---@param pos2 vector
---@param fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil
---@param opts? {}
---@param callback fun(ok: boolean, changed_count: integer, err?: string)
function OctreeManip:apply_async(pos1, pos2, fn, opts, callback)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	assert(pos1 and pos2, "pos1 and pos2 required")
	assert(type(fn) == "function", "fn required")

	if type(opts) == "function" and callback == nil then
		callback = opts
		opts = nil
	end
	assert(type(callback) == "function", "callback required")

	if self._apply_inflight then
		callback(false, 0, "apply already in progress")
		return
	end

	opts = opts or {}
	local time_budget_us = get_async_budget_us()

	pos1, pos2 = vector.sort(pos1, pos2)
	pos1 = assert(vector.round(pos1))
	pos2 = assert(vector.round(pos2))

	local e1, e2 = snap:get_emerged_area()
	if not e1 or not e2 then
		self._apply_inflight = false
		callback(false, 0, "snapshot has no emerged area")
		return
	end
	---@cast e1 vector
	---@cast e2 vector
	local minx = math.max(pos1.x, e1.x)
	local miny = math.max(pos1.y, e1.y)
	local minz = math.max(pos1.z, e1.z)
	local maxx = math.min(pos2.x, e2.x)
	local maxy = math.min(pos2.y, e2.y)
	local maxz = math.min(pos2.z, e2.z)

	if minx > maxx or miny > maxy or minz > maxz then
		callback(true, 0)
		return
	end

	self._apply_inflight = true
	local grid_minp = self._grid_minp or self.pos1
	local gx_min, gy_min, gz_min = chunk_coords_from_world(grid_minp, minx, miny, minz)
	local gx_max, gy_max, gz_max = chunk_coords_from_world(grid_minp, maxx, maxy, maxz)
	if gx_min < 1 then gx_min = 1 end
	if gy_min < 1 then gy_min = 1 end
	if gz_min < 1 then gz_min = 1 end
	if gx_max > snap.trees.size.x then gx_max = snap.trees.size.x end
	if gy_max > snap.trees.size.y then gy_max = snap.trees.size.y end
	if gz_max > snap.trees.size.z then gz_max = snap.trees.size.z end

	local default_cid = resolve_default_content_id(snap)
	local gx, gy, gz = gx_min, gy_min, gz_min
	local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)
	local x, y, z = math.max(minx, chunk_min.x), math.max(miny, chunk_min.y), math.max(minz, chunk_min.z)
	local ctx = {
		self = self,
		snap = snap,
		callback = callback,
		fn = fn,
		time_budget_us = time_budget_us,
		grid_minp = grid_minp,
		gx_min = gx_min,
		gy_min = gy_min,
		gz_min = gz_min,
		gx_max = gx_max,
		gy_max = gy_max,
		gz_max = gz_max,
		minx = minx,
		miny = miny,
		minz = minz,
		maxx = maxx,
		maxy = maxy,
		maxz = maxz,
		default_cid = default_cid,
		gx = gx,
		gy = gy,
		gz = gz,
		chunk_min = chunk_min,
		chunk_max = chunk_max,
		x = x,
		y = y,
		z = z,
		changed = 0,
	}

	map_octree.run_budgeted({
		budget_ms = time_budget_us / 1000,
		delay = DEFAULT_STEP_DELAY,
		step_fn = function()
			return apply_async_step(ctx)
		end,
		done_fn = function()
			if ctx._err then
				apply_async_finish(ctx, false, ctx._err)
			else
				apply_async_finish(ctx, true)
			end
		end,
	})
end



---Async version of apply_ranges.
---@async
---@param ranges OctreeRange[]
---@param fn fun(x: number, y: number, z: number, cid: integer, p2: integer, p1: integer): integer|nil, integer|nil, integer|nil
---@param opts? RangeIterateOpts
---@param callback fun(ok: boolean, changed_count: integer, err?: string)
function OctreeManip:apply_ranges_async(ranges, fn, opts, callback)
	local snap = assert(self._snapshot, "call :read_from_map(pos1, pos2) first")
	assert(type(ranges) == "table", "ranges required")
	assert(type(fn) == "function", "fn required")
	assert(type(callback) == "function", "callback required")

	if self._apply_inflight then
		callback(false, 0, "apply already in progress")
		return
	end

	local grid_minp = self._grid_minp or self.pos1
	local default_cid = resolve_default_content_id(snap)
	local time_budget_us = get_async_budget_us()
	local tasks = {}
	map_octree.iterate_chunk_ranges(grid_minp, snap.trees.size, ranges, opts, function(gx, gy, gz, chunk_min, chunk_max, req_min, req_max)
		tasks[#tasks + 1] = {
			gx = gx,
			gy = gy,
			gz = gz,
			chunk_min = chunk_min,
			chunk_max = chunk_max,
			req_min = req_min,
			req_max = req_max,
		}
	end)

	if #tasks == 0 then
		callback(true, 0)
		return
	end

	self._apply_inflight = true
	local task_idx = 1
	local ctx = {
		self = self,
		snap = snap,
		callback = callback,
		fn = fn,
		default_cid = default_cid,
		time_budget_us = time_budget_us,
		tasks = tasks,
		task_idx = task_idx,
		x = nil,
		y = nil,
		z = nil,
		changed = 0,
	}

	map_octree.run_budgeted({
		budget_ms = time_budget_us / 1000,
		delay = DEFAULT_STEP_DELAY,
		step_fn = function()
			return apply_ranges_async_step(ctx)
		end,
		done_fn = function()
			if ctx._err then
				apply_ranges_async_finish(ctx, false, ctx._err)
			else
				apply_ranges_async_finish(ctx, true)
			end
		end,
	})
end



function OctreeManip:clear_deltas()
	self._dirty_trees = {}
	self._tree_deltas = {}
	self._tree_delta_counts = {}
end



function OctreeManip:_refresh_snapshot_tree(gx, gy, gz)
	local snap = assert(self._snapshot)
	local grid_minp = self._grid_minp or self.pos1
	local p1, p2 = chunk_world_bounds(grid_minp, gx, gy, gz)
	local map = octmap.new(p1, p2, self._build_opts)
	-- copy the only tree into our snapshot (same grid coords)
	local cell = matrix3d.get(map.trees, 1, 1, 1)
	matrix3d.set(snap.trees, gx, gy, gz, cell)
	-- merge cid->name knowledge
	for cid, nm in pairs(map.content_id_map) do
		if not snap.content_id_map[cid] then
			snap.content_id_map[cid] = nm
		end
	end
end



---Write all recorded deltas to the world map.
function OctreeManip:write_to_map()
	assert(self._snapshot and self.pos1, "call :read_from_map(pos1, pos2) first")
	if #self._dirty_trees == 0 then
		return true
	end

	local grid_minp = self._grid_minp or self.pos1
	local max_vm_volume, gc_limit_kb = get_write_limits()
	local batch_size_nodes = compute_write_batch_size_nodes(max_vm_volume)
	local batch_list = build_write_batches(self._dirty_trees, grid_minp, batch_size_nodes)

	for i = 1, #batch_list do
		apply_write_batch(self, batch_list[i], grid_minp)
		maybe_collect_write_gc(gc_limit_kb)
	end

	refresh_snapshot_if_needed(self)

	self:clear_deltas()
	maybe_collect_write_gc(gc_limit_kb)
	return true
end



---Async write all recorded deltas to the world map.
---@param callback fun(ok: boolean, err?: string)
function OctreeManip:write_to_map_async(callback)
	local grid_minp = self._grid_minp or self.pos1
	assert(self._snapshot and self.pos1, "call :read_from_map(pos1, pos2) first")
	assert(type(callback) == "function", "callback required")

	if self._write_inflight then
		table.insert(self._write_callback_queue, callback)
		return
	end

	if #self._dirty_trees == 0 then
		callback(true)
		return
	end

	self._write_inflight = true

	local max_vm_volume, gc_limit_kb = get_write_limits()
	local batch_size_nodes = compute_write_batch_size_nodes(max_vm_volume)
	local batch_list = build_write_batches(self._dirty_trees, grid_minp, batch_size_nodes)

	local batch_idx = 1

	---Finalize async write and release state.
	---@param ok boolean
	---@param err? string
	local function finish(ok, err)
		maybe_collect_write_gc(gc_limit_kb)
		self:clear_deltas()
		self._write_inflight = false

		if map_octree.on_async_write_complete and self.pos1 and self.pos2 then
			core.after(0, function()
				map_octree.on_async_write_complete(self.pos1, self.pos2)
			end)
		end

		callback(ok, err)

		if #self._write_callback_queue > 0 then
			local next_callback = table.remove(self._write_callback_queue, 1)
			core.after(0, function()
				self:write_to_map_async(next_callback)
			end)
		end
	end

	map_octree.run_budgeted({
		budget_ms = get_async_budget_us() / 1000,
		delay = DEFAULT_STEP_DELAY,
		step_fn = function()
			if batch_idx > #batch_list then
				return true
			end
			local batch = batch_list[batch_idx]
			batch_idx = batch_idx + 1
			apply_write_batch(self, batch, grid_minp)
			maybe_collect_write_gc(gc_limit_kb)
			return batch_idx > #batch_list
		end,
		done_fn = function()
			refresh_snapshot_if_needed(self)
			finish(true)
		end,
	})
end



---Create a new OctreeManip instance (convenience wrapper).
---@param opts? OctreeManipOpts
---@return OctreeManip
function map_octree.new_octree_manip(opts)
	return OctreeManip.new(opts)
end



---Finalize apply_async.
---@param ctx ApplyAsyncContext
---@param ok boolean
---@param err? string
function apply_async_finish(ctx, ok, err)
	ctx.self._apply_inflight = false
	if ok then
		ctx.callback(true, ctx.changed)
	else
		ctx.callback(false, ctx.changed, err)
	end
end



---Step the apply_async state machine.
---@param ctx ApplyAsyncContext
---@return boolean done
function apply_async_step(ctx)
	if ctx.gz > ctx.gz_max then
		return true
	end

	local step_start = core.get_us_time()
	local ok, err = pcall(apply_async_step_impl, ctx, step_start)
	if not ok then
		ctx._err = tostring(err)
		return true
	end

	return ctx.gz > ctx.gz_max
end



---Execute a timed slice of apply_async work.
---@param ctx ApplyAsyncContext
---@param step_start integer
function apply_async_step_impl(ctx, step_start)
	while ctx.gz <= ctx.gz_max do
		local key = chunk_key(ctx.gx, ctx.gy, ctx.gz)
		local cell = matrix3d.get(ctx.snap.trees, ctx.gx, ctx.gy, ctx.gz)
		local tree = cell ~= nil and octmap._get_tree_cached(ctx.snap, ctx.gx, ctx.gy, ctx.gz) or nil

		local z_end = math.min(ctx.maxz, ctx.chunk_max.z)
		local y_end = math.min(ctx.maxy, ctx.chunk_max.y)
		local x_end = math.min(ctx.maxx, ctx.chunk_max.x)
		while ctx.z <= z_end do
			while ctx.y <= y_end do
				while ctx.x <= x_end do
					local cid, p2, p1
					if tree then
						---@cast tree Octchunk
						cid, p2, p1 = octchunk.get_node_value_xyz(tree, ctx.x, ctx.y, ctx.z)
					else
						cid, p2, p1 = ctx.default_cid, 0, 0
					end

					local new_cid, new_p2, new_p1 = ctx.fn(ctx.x, ctx.y, ctx.z, cid, p2, p1)
					if new_cid ~= nil then
						new_p2 = new_p2 and tonumber(new_p2) or p2
						new_p1 = new_p1 and tonumber(new_p1) or p1
						local idx0 = local_index_in_chunk(ctx.chunk_min, ctx.x, ctx.y, ctx.z)
						if write_delta_cid(ctx.self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
							ctx.changed = ctx.changed + 1
						end
					end

					ctx.x = ctx.x + 1
					local elapsed = core.get_us_time() - step_start
					if elapsed >= ctx.time_budget_us then
						return
					end
				end
				ctx.x = math.max(ctx.minx, ctx.chunk_min.x)
				ctx.y = ctx.y + 1
			end
			ctx.y = math.max(ctx.miny, ctx.chunk_min.y)
			ctx.z = ctx.z + 1
		end

		ctx.gx = ctx.gx + 1
		if ctx.gx > ctx.gx_max then
			ctx.gx = ctx.gx_min
			ctx.gy = ctx.gy + 1
			if ctx.gy > ctx.gy_max then
				ctx.gy = ctx.gy_min
				ctx.gz = ctx.gz + 1
			end
		end
		if ctx.gz > ctx.gz_max then
			return
		end
		ctx.chunk_min, ctx.chunk_max = chunk_world_bounds(ctx.grid_minp, ctx.gx, ctx.gy, ctx.gz)
		ctx.x = math.max(ctx.minx, ctx.chunk_min.x)
		ctx.y = math.max(ctx.miny, ctx.chunk_min.y)
		ctx.z = math.max(ctx.minz, ctx.chunk_min.z)
	end
end



---Finalize apply_ranges_async.
---@param ctx ApplyRangesAsyncContext
---@param ok boolean
---@param err? string
function apply_ranges_async_finish(ctx, ok, err)
	ctx.self._apply_inflight = false
	if ok then
		ctx.callback(true, ctx.changed)
	else
		ctx.callback(false, ctx.changed, err)
	end
end



---Step the apply_ranges_async state machine.
---@param ctx ApplyRangesAsyncContext
---@return boolean done
function apply_ranges_async_step(ctx)
	if ctx.task_idx > #ctx.tasks then
		return true
	end

	local step_start = core.get_us_time()
	local ok, err = pcall(apply_ranges_async_step_impl, ctx, step_start)
	if not ok then
		ctx._err = tostring(err)
		return true
	end

	return ctx.task_idx > #ctx.tasks
end



---Execute a timed slice of apply_ranges_async work.
---@param ctx ApplyRangesAsyncContext
---@param step_start integer
function apply_ranges_async_step_impl(ctx, step_start)
	while ctx.task_idx <= #ctx.tasks do
		local t = ctx.tasks[ctx.task_idx]
		local key = chunk_key(t.gx, t.gy, t.gz)
		local cell = matrix3d.get(ctx.snap.trees, t.gx, t.gy, t.gz)
		local tree = cell ~= nil and octmap._get_tree_cached(ctx.snap, t.gx, t.gy, t.gz) or nil

		if not ctx.x then
			ctx.x = t.req_min.x
			ctx.y = t.req_min.y
			ctx.z = t.req_min.z
		end

		local x_end = t.req_max.x
		local y_end = t.req_max.y
		local z_end = t.req_max.z
		while ctx.z <= z_end do
			while ctx.y <= y_end do
				while ctx.x <= x_end do
					local cid, p2, p1
					if tree then
						---@cast tree Octchunk
						cid, p2, p1 = octchunk.get_node_value_xyz(tree, ctx.x, ctx.y, ctx.z)
					else
						cid, p2, p1 = ctx.default_cid, 0, 0
					end

					local new_cid, new_p2, new_p1 = ctx.fn(ctx.x, ctx.y, ctx.z, cid, p2, p1)
					if new_cid ~= nil then
						new_p2 = new_p2 and tonumber(new_p2) or p2
						new_p1 = new_p1 and tonumber(new_p1) or p1
						local idx0 = local_index_in_chunk(t.chunk_min, ctx.x, ctx.y, ctx.z)
						if write_delta_cid(ctx.self, key, idx0, cid, p2, p1, new_cid, new_p2, new_p1) then
							ctx.changed = ctx.changed + 1
						end
					end

					ctx.x = ctx.x + 1
					local elapsed = core.get_us_time() - step_start
					if elapsed >= ctx.time_budget_us then
						return
					end
				end
				ctx.x = t.req_min.x
				ctx.y = ctx.y + 1
			end
			ctx.y = t.req_min.y
			ctx.z = ctx.z + 1
		end

		ctx.task_idx = ctx.task_idx + 1
		ctx.x, ctx.y, ctx.z = nil, nil, nil
	end
end



---Get async write budget in microseconds.
---@return integer
function get_async_budget_us()
	local budget_ms = tonumber(core.settings:get(SETTING_WRITE_BUDGET_MS) or "")
	if not budget_ms or budget_ms <= 0 then
		budget_ms = DEFAULT_WRITE_BUDGET_MS
	end
	return budget_ms * 1000
end



---Sort two positions.
---@param p1 vector
---@param p2 vector
---@return vector
---@return vector
function sorted_pos(p1, p2)
	return vector.sort(p1, p2)
end



---Get linear index inside a chunk.
---@param chunk_min vector
---@param x number
---@param y number
---@param z number
---@return integer
function local_index_in_chunk(chunk_min, x, y, z)
	local dx = x - chunk_min.x
	local dy = y - chunk_min.y
	local dz = z - chunk_min.z
	local size = octchunk.SIZE
	return (dz * size + dy) * size + dx
end



---Build a chunk key from grid coords.
---@param gx integer
---@param gy integer
---@param gz integer
---@return string
function chunk_key(gx, gy, gz)
	return tostring(gx) .. ":" .. tostring(gy) .. ":" .. tostring(gz)
end



---Parse a chunk key into grid coords.
---@param key string
---@return integer|nil
---@return integer|nil
---@return integer|nil
function parse_chunk_key(key)
	local a, b, c = key:match("^(%d+):(%d+):(%d+)$")
	return tonumber(a), tonumber(b), tonumber(c)
end



---Pack content id and param2 into an integer.
---@param cid integer
---@param p2 integer|nil
---@return integer
function pack_cid_p2(cid, p2)
	p2 = tonumber(p2) or 0
	if p2 < 0 then p2 = 0 end
	if p2 > 255 then p2 = 255 end
	return cid * 256 + p2
end



---Unpack content id and param2.
---@param packed integer
---@return integer cid
---@return integer p2
function unpack_cid_p2(packed)
	local p = tonumber(packed) or 0
	local cid = floor(p / 256)
	local p2 = p - cid * 256
	return cid, p2
end



---Pack content id, param2, and param1 into an integer.
---@param cid integer
---@param p2 integer|nil
---@param p1 integer|nil
---@return integer
function pack_cid_p2_p1(cid, p2, p1)
	p2 = tonumber(p2) or 0
	p1 = tonumber(p1) or 0
	if p2 < 0 then p2 = 0 end
	if p2 > 255 then p2 = 255 end
	if p1 < 0 then p1 = 0 end
	if p1 > 255 then p1 = 255 end
	-- Pack to single integer to avoid GC pressure (deltas can be millions of nodes).
	-- Negative values flag param1 presence (bit 31 as discriminator).
	-- This avoids ambiguity with (cid,p2)-only packed values, and keeps storage low-GC.
	return -(cid * 65536 + p2 * 256 + p1 + 1)
end



---Unpack content id, param2, and param1.
---@param packed integer
---@return integer cid
---@return integer p2
---@return integer p1
function unpack_cid_p2_p1(packed)
	local p = tonumber(packed) or 0
	p = -p - 1
	local cid = floor(p / 65536)
	local remainder = p - cid * 65536
	local p2 = floor(remainder / 256)
	local p1 = remainder - p2 * 256
	return cid, p2, p1
end



---Build write batches from dirty chunk keys.
---@param dirty_trees string[]
---@param grid_minp vector
---@param batch_size_nodes? integer
---@return table[]
function build_write_batches(dirty_trees, grid_minp, batch_size_nodes)
	local batches = {}
	local batch_size = math.max(octchunk.SIZE, math.floor(tonumber(batch_size_nodes) or BATCH_SIZE))

	for i = 1, #dirty_trees do
		local key = dirty_trees[i]
		local gx, gy, gz = parse_chunk_key(key)
		assert(gx and gy and gz, "invalid chunk key: " .. tostring(key))
		local chunk_min, chunk_max = chunk_world_bounds(grid_minp, gx, gy, gz)

		local batch_x = floor(chunk_min.x / batch_size)
		local batch_y = floor(chunk_min.y / batch_size)
		local batch_z = floor(chunk_min.z / batch_size)
		local batch_key = batch_x .. "," .. batch_y .. "," .. batch_z

		if not batches[batch_key] then
			batches[batch_key] = {
				trees = {},
				min = vector.new(chunk_min.x, chunk_min.y, chunk_min.z),
				max = vector.new(chunk_max.x, chunk_max.y, chunk_max.z)
			}
		end

		local batch = batches[batch_key]
		batch.trees[#batch.trees + 1] = {key = key, gx = gx, gy = gy, gz = gz}
		batch.min.x = math.min(batch.min.x, chunk_min.x)
		batch.min.y = math.min(batch.min.y, chunk_min.y)
		batch.min.z = math.min(batch.min.z, chunk_min.z)
		batch.max.x = math.max(batch.max.x, chunk_max.x)
		batch.max.y = math.max(batch.max.y, chunk_max.y)
		batch.max.z = math.max(batch.max.z, chunk_max.z)
	end

	local batch_list = {}
	for _, b in pairs(batches) do
		batch_list[#batch_list + 1] = b
	end

	return batch_list
end



function get_write_limits()
	local opts = octmap.apply_write_limits({}, 1, 3)
	return opts.max_voxelmanip_volume, opts._octmap_gc_limit_kb
end



function compute_write_batch_size_nodes(max_vm_volume)
	local size = octchunk.SIZE
	local root = math.floor(max_vm_volume ^ (1 / 3))
	local max_chunks = math.max(1, math.floor((root - 1) / size))
	return max_chunks * size
end



function maybe_collect_write_gc(limit_kb)
	if not limit_kb or limit_kb <= 0 then
		return
	end
	if collectgarbage("count") >= limit_kb then
		collectgarbage("collect")
	end
end



---Apply a write batch into the map.
---@param self OctreeManip
---@param batch table
---@param grid_minp vector
function apply_write_batch(self, batch, grid_minp)
	local param1_data
	core.load_area(batch.min, batch.max)
	local manip = core.get_voxel_manip()
	local e1, e2 = manip:read_from_map(batch.min, batch.max)
	local area = VoxelArea(e1, e2)
	local data = {}
	local param2_data = {}
	manip:get_data(data)
	manip:get_param2_data(param2_data)

	for _, tree_info in ipairs(batch.trees) do
		local deltas = self._tree_deltas[tree_info.key]
		if deltas and next(deltas) then
			local chunk_min = chunk_world_bounds(grid_minp, tree_info.gx, tree_info.gy, tree_info.gz)
			for idx0, packed in pairs(deltas) do
				local cid, p2, p1
				if packed < 0 then
					cid, p2, p1 = unpack_cid_p2_p1(packed)
				else
					cid, p2 = unpack_cid_p2(packed)
				end
				local dx = idx0 % octchunk.SIZE
				local dy = floor(idx0 / octchunk.SIZE) % octchunk.SIZE
				local dz = floor(idx0 / (octchunk.SIZE * octchunk.SIZE))
				local x = chunk_min.x + dx
				local y = chunk_min.y + dy
				local z = chunk_min.z + dz
				local idx = area:index(x, y, z)
				data[idx] = cid
				param2_data[idx] = p2
				if p1 ~= nil then
					if not param1_data then
						param1_data = manip:get_light_data()
					end
					if param1_data then
						param1_data[idx] = p1
					end
				end
			end
		end
	end

	manip:set_data(data)
	manip:set_param2_data(param2_data)
	if param1_data then
		manip:set_light_data(param1_data)
	end
	manip:write_to_map(true)
	manip:close()
end



---Refresh snapshot trees that were modified, if not frozen.
---@param self OctreeManip
function refresh_snapshot_if_needed(self)
	if self._frozen_snapshot then
		return
	end
	for i = 1, #self._dirty_trees do
		local key = self._dirty_trees[i]
		local gx, gy, gz = parse_chunk_key(key)
		self:_refresh_snapshot_tree(gx, gy, gz)
	end
end
