local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")


map_octree.tests.register("bounds + file I/O", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	local pos = assert(ctx.player:get_pos(), "player position required")
	---@cast pos vector
	pos = vector.round(pos)
	---@cast pos vector
	local center = octchunk.snap_to_center(pos)

	local map = octmap.new(center, center, {
		max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
		force_batches = true,
	})

	local outside = vector.add(center, {x = 64 * 10, y = 0, z = 0})
	local got = octmap.get_node_name(map, outside)
	if got ~= nil then
		error("expected nil outside bounds")
	end

	local testfile = core.get_worldpath() .. "/octree_map_test.bin"
	local blob, err = octmap.serialize(map)
	assert(blob, err or "serialize failed")
	core.safe_file_write(testfile, blob)

	local loaded, err_load = octmap.read_from_file(testfile)
	assert(type(loaded) == "table", err_load or "expected loaded map")

	os.remove(testfile)

	-- spot check a few positions
	local positions = {
		center,
		vector.subtract(center, {x = 32, y = 32, z = 32}),
		vector.add(center, {x = 31, y = 31, z = 31}),
	}
	for _, p in ipairs(positions) do
		local id1 = octmap.get_node_name(map, p)
		local id2 = octmap.get_node_name(loaded, p)
		if id1 ~= id2 then
			error("loaded map mismatch at " .. core.pos_to_string(p))
		end
	end
end)


map_octree.tests.register("save/load bounds exact", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = testutil.get_test_region(ctx.player)
	local center = octchunk.snap_to_center(base)
	local half = octchunk.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})

	local req_min = vector.add(pmin, {x = 3, y = 5, z = 7})
	local req_max = vector.subtract(pmax, {x = 2, y = 1, z = 4})

	local file_name = "bounds_invariant_" .. os.time()
	local subdir = "tests"
	local map, err = map_octree.save_to_file(req_min, req_max, {
		file_name = file_name,
		subdir = subdir,
		store_chunk_blobs = true,
		force_batches = true,
		max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
	})
	assert(map, err or "save_to_file failed")

	local function assert_exact(label, minp, maxp)
		assert(minp.x == req_min.x and minp.y == req_min.y and minp.z == req_min.z,
			label .. " min is not exact")
		assert(maxp.x == req_max.x and maxp.y == req_max.y and maxp.z == req_max.z,
			label .. " max is not exact")
	end

	local e1, e2 = map:get_emerged_area()
	assert_exact("saved map", e1, e2)

	local outside = vector.subtract(req_min, {x = 1, y = 0, z = 0})
	local name = map:get_node_at(outside.x, outside.y, outside.z)
	assert(name == nil, "expected nil outside requested bounds")

	local loaded = map_octree.load_map(file_name, {subdir = subdir, set_current = false})
	assert(loaded, "expected loaded map")
	local l1, l2 = loaded:get_emerged_area()
	assert_exact("loaded map", l1, l2)

	local dir = map_octree.get_storage_dir(subdir)
	os.remove(dir .. "/" .. file_name .. ".bin")
end)
