local tutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")


map_octree.tests.register("param1: place preserves lighting (not dark)", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = vector.round(ctx.player:get_pos())
	local src_center = octchunk.snap_to_center(vector.add(base, {x = 128, y = 0, z = 64}))
	local half = octchunk.SIZE / 2

	local src_min = vector.subtract(src_center, {x = half, y = half, z = half})
	local src_max = vector.add(src_center, {x = half - 1, y = half - 1, z = half - 1})
	local delta = {x = octchunk.SIZE * 4, y = 0, z = 0}
	local dst_min = vector.add(src_min, delta)
	local dst_max = vector.add(src_max, delta)

	local all_min = vector.new(
		math.min(src_min.x, dst_min.x),
		math.min(src_min.y, dst_min.y),
		math.min(src_min.z, dst_min.z)
	)
	local all_max = vector.new(
		math.max(src_max.x, dst_max.x),
		math.max(src_max.y, dst_max.y),
		math.max(src_max.z, dst_max.z)
	)

	local light_node = "default:meselamp"
	local cid_light = core.get_content_id(light_node)
	local cid_air = core.get_content_id("air")
	if type(cid_light) ~= "number" then
		core.log("warning", "[test] skipping param1 place test: missing node " .. light_node)
		return
	end
	assert(type(cid_air) == "number", "missing node: air")

	local light_pos = vector.add(src_min, {x = 4, y = 4, z = 4})
	local check_pos = vector.add(light_pos, {x = 1, y = 0, z = 0})
	local dst_check_pos = vector.add(check_pos, delta)

	tutil.with_voxel_region(all_min, all_max, function(manip, area, data, param2_data)
		if not manip.get_light_data or not manip.set_light_data then
			core.log("warning", "[test] skipping param1 place test: VoxelManip light_data methods not available")
			return
		end

		-- Prepare src with a real light source and clear dst
		for z = all_min.z, all_max.z do
			for y = all_min.y, all_max.y do
				for x = all_min.x, all_max.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end

		data[area:index(light_pos.x, light_pos.y, light_pos.z)] = cid_light
		data[area:index(check_pos.x, check_pos.y, check_pos.z)] = cid_air

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		-- Force lighting to exist in the source so we can validate placement isn't dark.
		manip:write_to_map(true)

		-- Save snapshot from src
		local map = octmap.new(src_min, src_max, {
			store_chunk_blobs = true,
			cache_mb = 8,
		})

		-- Place at dst
		map_octree.place(map, dst_min)

		-- Verify placed snapshot is not dark near the light source
		core.load_area(dst_min, dst_max)
		local manip2 = core.get_voxel_manip()
		local e1, e2 = manip2:read_from_map(dst_min, dst_max)
		local area2 = VoxelArea(e1, e2)
		local param1_verify = manip2:get_light_data()

		local dst_idx = area2:index(dst_check_pos.x, dst_check_pos.y, dst_check_pos.z)
		local got_light = (param1_verify and param1_verify[dst_idx]) or 0
		if got_light <= 0 then
			error(string.format("placed lighting is dark at %s: expected >0, got %d",
				core.pos_to_string(dst_check_pos), got_light))
		end


	end)
end)


map_octree.tests.register("param1: get_node_at returns param1", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = vector.round(ctx.player:get_pos())
	local center = octchunk.snap_to_center(vector.add(base, {x = 96, y = 0, z = 96}))
	local half = octchunk.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})

	local light_node = "default:torch"
	local cid_light = core.get_content_id(light_node)
	if type(cid_light) ~= "number" then
		core.log("warning", "[test] skipping param1 get_node_at test: missing node " .. light_node)
		return
	end

	local test_pos = vector.add(pmin, {x = 4, y = 4, z = 4})

	tutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		if not manip.get_light_data or not manip.set_light_data then
			core.log("warning", "[test] skipping param1 get_node_at test: light_data methods not available")
			return
		end
		local cid_air = core.get_content_id("air")
		assert(type(cid_air) == "number", "missing node: air")
		for z = pmin.z, pmax.z do
			for y = pmin.y, pmax.y do
				for x = pmin.x, pmax.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end

		local test_idx = area:index(test_pos.x, test_pos.y, test_pos.z)
		data[test_idx] = cid_light
		if param2_data then param2_data[test_idx] = 1 end

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(true)

		-- After write_to_map(true), lighting is computed in the engine.
		-- Read back from the map to obtain the authoritative packed param1.
		core.load_area(pmin, pmax)
		local manip2 = core.get_voxel_manip()
		local e1, e2 = manip2:read_from_map(pmin, pmax)
		local area2 = VoxelArea(e1, e2)
		local param1_verify = manip2:get_light_data()
		local expected_param1 = (param1_verify and param1_verify[area2:index(test_pos.x, test_pos.y, test_pos.z)]) or 0


		local map = octmap.new(pmin, pmax, {store_chunk_blobs = true, cache_mb = 4})

		local name, p2, p1 = map:get_node_at(test_pos.x, test_pos.y, test_pos.z)
		assert(name == light_node, string.format("get_node_at name mismatch: expected %s, got %s", light_node, tostring(name)))
		assert(p2 == 1, string.format("get_node_at param2 mismatch: expected 1, got %d", p2 or -1))
		assert(p1 == expected_param1, string.format("get_node_at param1 mismatch: expected %d, got %d", expected_param1, p1 or -1))

		local cid, p2c, p1c = map:get_node_cid_at(test_pos.x, test_pos.y, test_pos.z)
		assert(cid == cid_light, "get_node_cid_at content_id mismatch")
		assert(p2c == 1, string.format("get_node_cid_at param2 mismatch: expected 1, got %d", p2c or -1))
		assert(p1c == expected_param1, string.format("get_node_cid_at param1 mismatch: expected %d, got %d", expected_param1, p1c or -1))
	end)
end)


map_octree.tests.register("param1: for_each_node passes param1 to callback", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = vector.round(ctx.player:get_pos())
	local center = octchunk.snap_to_center(vector.add(base, {x = 112, y = 0, z = 112}))
	local half = octchunk.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})

	local light_node = "default:torch"
	local cid_light = core.get_content_id(light_node)
	if type(cid_light) ~= "number" then
		core.log("warning", "[test] skipping param1 for_each_node test: missing node " .. light_node)
		return
	end

	local test_pos = vector.add(pmin, {x = 3, y = 3, z = 3})

	tutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		if not manip.get_light_data or not manip.set_light_data then
			core.log("warning", "[test] skipping param1 for_each_node test: light_data methods not available")
			return
		end
		local cid_air = core.get_content_id("air")
		assert(type(cid_air) == "number", "missing node: air")
		for z = pmin.z, pmax.z do
			for y = pmin.y, pmax.y do
				for x = pmin.x, pmax.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end

		local test_idx = area:index(test_pos.x, test_pos.y, test_pos.z)
		data[test_idx] = cid_light
		if param2_data then param2_data[test_idx] = 2 end

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(true)

		core.load_area(pmin, pmax)
		local manip2 = core.get_voxel_manip()
		local e1, e2 = manip2:read_from_map(pmin, pmax)
		local area2 = VoxelArea(e1, e2)
		local param1_verify = manip2:get_light_data()
		local expected_param1 = (param1_verify and param1_verify[area2:index(test_pos.x, test_pos.y, test_pos.z)]) or 0


		local map = octmap.new(pmin, pmax, {store_chunk_blobs = true, cache_mb = 4})

		local found_test_node = false
		---@diagnostic disable-next-line: undefined-field
		map:for_each_node(pmin, pmax, function(x, y, z, name, p2, p1)
			if x == test_pos.x and y == test_pos.y and z == test_pos.z then
				assert(name == light_node, "for_each_node name mismatch")
				assert(p2 == 2, string.format("for_each_node param2 mismatch: expected 2, got %d", p2 or -1))
				assert(p1 == expected_param1, string.format("for_each_node param1 mismatch: expected %d, got %d", expected_param1, p1 or -1))
				found_test_node = true
			end
		end)
		assert(found_test_node, "test node not found in for_each_node scan")

		found_test_node = false
		---@diagnostic disable-next-line: undefined-field
		map:for_each_node_cid(pmin, pmax, function(x, y, z, cid, p2, p1)
			if x == test_pos.x and y == test_pos.y and z == test_pos.z then
				assert(cid == cid_light, "for_each_node_cid content_id mismatch")
				assert(p2 == 2, string.format("for_each_node_cid param2 mismatch: expected 2, got %d", p2 or -1))
				assert(p1 == expected_param1, string.format("for_each_node_cid param1 mismatch: expected %d, got %d", expected_param1, p1 or -1))
				found_test_node = true
			end
		end)
		assert(found_test_node, "test node not found in for_each_node_cid scan")
	end)
end)