--- Shared utility functions.
-- Safe to load in async environments (globals may not exist).

map_octree = map_octree or {}

--#region types
---@class OctreeRange
---@field pos1? vector
---@field pos2? vector
---@field min? vector
---@field max? vector
---@field [1]? vector
---@field [2]? vector

---@alias RangeIterateOpts {grid?: "node"|"chunk", dedup?: boolean}

---@class BudgetedOpts
---@field budget_ms number Per-tick budget in milliseconds.
---@field step_fn fun(): boolean Executes one unit of work; returns true when all work is done.
---@field done_fn? fun() Optional callback invoked once when work completes.
---@field cancel_fn? fun(): boolean Optional callback checked before each tick; return true to cancel.
---@field delay? number Optional delay between ticks (seconds).
--#endregion


---Apply function to each value in table, returning new table with same keys.
---@param t table<any, any>
---@param f fun(v: any): any
---@return table<any, any>
function map_octree.table_map(t, f)
	local result = {}
	for k, v in pairs(t) do
		result[k] = f(v)
	end
	return result
end



---Count entries in a table (works for non-sequential tables).
---@param t table<any, any>
---@return integer
function map_octree.table_count(t)
	local c = 0
	for _ in pairs(t) do
		c = c + 1
	end
	return c
end



---Round n up to the nearest power of 2.
---@param n number
---@return integer
function map_octree.round_to_pow_of_2(n)
	local mul = 2
	while mul < n do
		mul = mul * 2
	end
	return mul
end



---Get requested bounds for a map (exact), with fallback to snapped bounds.
---@param map OctMap
---@return vector minp
---@return vector maxp
function map_octree.get_requested_bounds(map)
	if map and map.requested_pos1 and map.requested_pos2 then
		local p1, p2 = vector.sort(map.requested_pos1, map.requested_pos2)
		p1 = assert(vector.round(p1))
		p2 = assert(vector.round(p2))
		return p1, p2
	end
	local half = octchunk.SIZE / 2
	local minp = vector.subtract(map.minp, {x = half, y = half, z = half})
	local maxp = vector.add(map.maxp, {x = half - 1, y = half - 1, z = half - 1})
	return minp, maxp
end



---Get the center of the chunk containing pos.
---@param pos vector
---@return vector
function map_octree.get_chunk_center(pos)
	local chunk_size = octchunk.SIZE
	local half_chunk_size = chunk_size / 2
	return {
		x = math.floor(pos.x / chunk_size) * chunk_size + half_chunk_size,
		y = math.floor(pos.y / chunk_size) * chunk_size + half_chunk_size,
		z = math.floor(pos.z / chunk_size) * chunk_size + half_chunk_size
	}
end



---Run a budgeted loop over multiple ticks.
---@param opts BudgetedOpts
function map_octree.run_budgeted(opts)
	assert(opts and type(opts.step_fn) == "function", "step_fn required")
	local budget_ms = tonumber(opts.budget_ms)
	assert(budget_ms and budget_ms > 0, "budget_ms must be > 0")
	local delay = tonumber(opts.delay) or 0
	local step_fn = opts.step_fn
	local done_fn = opts.done_fn
	local cancel_fn = opts.cancel_fn

	local function tick()
		if cancel_fn and cancel_fn() then
			return
		end

		local t0 = core.get_us_time()
		while (core.get_us_time() - t0) < (budget_ms * 1000) do
			local done = step_fn()
			if done then
				if done_fn then
					done_fn()
				end
				return
			end
		end

		core.after(delay, tick)
	end

	core.after(delay, tick)
end



---Get the minimum corner of the chunk containing pos.
---@param pos vector
---@return vector
function map_octree.get_chunk_pos(pos)
	local chunk_size = octchunk.SIZE
	return {
		x = math.floor(pos.x / chunk_size) * chunk_size,
		y = math.floor(pos.y / chunk_size) * chunk_size,
		z = math.floor(pos.z / chunk_size) * chunk_size
	}
end



---Convert world coordinates to chunk grid indices (gx, gy, gz).
---@param minp vector Grid origin (snap min)
---@param x number
---@param y number
---@param z number
---@return integer gx
---@return integer gy
---@return integer gz
function map_octree.chunk_coords_from_world(minp, x, y, z)
	local size = octchunk.SIZE
	local half = size / 2
	local sx = math.floor(x / size) * size + half
	local sy = math.floor(y / size) * size + half
	local sz = math.floor(z / size) * size + half
	local gx = 1 + math.floor((sx - minp.x) / size)
	local gy = 1 + math.floor((sy - minp.y) / size)
	local gz = 1 + math.floor((sz - minp.z) / size)
	return gx, gy, gz
end



---Get world bounds of a chunk given grid coords.
---@param minp vector Grid origin (snap min)
---@param gx integer
---@param gy integer
---@param gz integer
---@return vector minp
---@return vector maxp
function map_octree.chunk_world_bounds(minp, gx, gy, gz)
	local size = octchunk.SIZE
	local half = size / 2
	local center = vector.new(
		minp.x + (gx - 1) * size,
		minp.y + (gy - 1) * size,
		minp.z + (gz - 1) * size
	)
	center = octchunk.snap_to_center(center)
	local p1 = vector.subtract(center, {x = half, y = half, z = half})
	local p2 = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})
	return p1, p2
end



---Iterate chunk ranges, optionally based on node-space ranges.
---@param minp vector Grid origin (snap min)
---@param chunk_size vector Chunk grid size
---@param ranges OctreeRange[]
---@param opts? RangeIterateOpts
---@param cb fun(gx: integer, gy: integer, gz: integer, chunk_min: vector, chunk_max: vector, req_min: vector, req_max: vector, range: OctreeRange)
---@return integer count
function map_octree.iterate_chunk_ranges(minp, chunk_size, ranges, opts, cb)
	assert(type(ranges) == "table", "ranges required")
	assert(type(cb) == "function", "callback required")
	local grid = opts and opts.grid or "node"
	local dedup = opts and opts.dedup
	if dedup == nil then
		dedup = (grid == "chunk")
	end
	local visited = {}
	local count = 0

	---Clamp grid bounds to chunk size.
	local function clamp_range(gx1, gy1, gz1, gx2, gy2, gz2)
		local minx = math.min(gx1, gx2)
		local miny = math.min(gy1, gy2)
		local minz = math.min(gz1, gz2)
		local maxx = math.max(gx1, gx2)
		local maxy = math.max(gy1, gy2)
		local maxz = math.max(gz1, gz2)
		if minx < 1 then minx = 1 end
		if miny < 1 then miny = 1 end
		if minz < 1 then minz = 1 end
		if maxx > chunk_size.x then maxx = chunk_size.x end
		if maxy > chunk_size.y then maxy = chunk_size.y end
		if maxz > chunk_size.z then maxz = chunk_size.z end
		return minx, miny, minz, maxx, maxy, maxz
	end

	for i = 1, #ranges do
		local range = ranges[i]
		if grid == "chunk" then
			local minv = range.min or range.pos1 or range[1]
			local maxv = range.max or range.pos2 or range[2]
			assert(minv and maxv, "chunk range requires min/max")
			assert(minv.x and minv.y and minv.z, "chunk range min must be a vector")
			assert(maxv.x and maxv.y and maxv.z, "chunk range max must be a vector")
			local gx_min, gy_min, gz_min, gx_max, gy_max, gz_max = clamp_range(
				math.floor(minv.x),
				math.floor(minv.y),
				math.floor(minv.z),
				math.floor(maxv.x),
				math.floor(maxv.y),
				math.floor(maxv.z)
			)

			for gz = gz_min, gz_max do
				for gy = gy_min, gy_max do
					for gx = gx_min, gx_max do
						local skip = false
						if dedup then
							local key = gx .. ":" .. gy .. ":" .. gz
							if visited[key] then
								skip = true
							else
								visited[key] = true
							end
						end

						if not skip then
							local chunk_min, chunk_max = map_octree.chunk_world_bounds(minp, gx, gy, gz)
							count = count + 1
							cb(gx, gy, gz, chunk_min, chunk_max, chunk_min, chunk_max, range)
						end
					end
				end
			end
		else
			local pos1 = assert(range.pos1 or range.min or range[1], "node range requires pos1")
			local pos2 = assert(range.pos2 or range.max or range[2], "node range requires pos2")
			pos1, pos2 = vector.sort(pos1, pos2)
			pos1 = assert(vector.round(pos1))
			pos2 = assert(vector.round(pos2))
			local p1x = assert(pos1.x, "node range pos1.x required")
			local p1y = assert(pos1.y, "node range pos1.y required")
			local p1z = assert(pos1.z, "node range pos1.z required")
			local p2x = assert(pos2.x, "node range pos2.x required")
			local p2y = assert(pos2.y, "node range pos2.y required")
			local p2z = assert(pos2.z, "node range pos2.z required")
			local gx1, gy1, gz1 = map_octree.chunk_coords_from_world(minp, p1x, p1y, p1z)
			local gx2, gy2, gz2 = map_octree.chunk_coords_from_world(minp, p2x, p2y, p2z)
			local gx_min, gy_min, gz_min, gx_max, gy_max, gz_max = clamp_range(gx1, gy1, gz1, gx2, gy2, gz2)

			for gz = gz_min, gz_max do
				for gy = gy_min, gy_max do
					for gx = gx_min, gx_max do
						local chunk_min, chunk_max = map_octree.chunk_world_bounds(minp, gx, gy, gz)
						local req_min = {
							x = math.max(p1x, chunk_min.x),
							y = math.max(p1y, chunk_min.y),
							z = math.max(p1z, chunk_min.z),
						}
						local req_max = {
							x = math.min(p2x, chunk_max.x),
							y = math.min(p2y, chunk_max.y),
							z = math.min(p2z, chunk_max.z),
						}
						if req_min.x <= req_max.x and req_min.y <= req_max.y and req_min.z <= req_max.z then
							count = count + 1
							cb(gx, gy, gz, chunk_min, chunk_max, req_min, req_max, range)
						end
					end
				end
			end
		end
	end

	return count
end
