local tree_cache_key
local lru_init
local lru_touch
local lru_pop_oldest
local resolve_cid_cached


-- Internal: lazy decompression with LRU cache (best-effort size estimate).
-- map.trees contains either compressed strings (tree blobs) or deserialized trees.
---Get a tree from cache or decompress it on demand.
---@param map OctMap
---@param gx integer
---@param gy integer
---@param gz integer
---@return OctNode|string|nil
local function get_tree_cached(map, gx, gy, gz)
	local cell = matrix3d.get(map.trees, gx, gy, gz)
	if type(cell) ~= "string" then
		return cell
	end

	if not map._tree_cache then
		map._tree_cache = {}
		lru_init(map)
		map._tree_cache_bytes = {}
		map._cache_total_bytes = 0
		map._cache_peak_bytes = 0
		map._cache_evictions = 0
	end

	local key = tree_cache_key(map, gx, gy, gz)
	local cached = map._tree_cache[key]
	if cached then
		lru_touch(map, key)
		return cached
	end

	-- Use compressed size as memory estimate (best-effort, avoids double decompression)
	local tree_bytes = #cell
	local tree = octchunk.deserialize(cell)

	-- Remap stored content_ids to current engine IDs (for restart stability).
	-- Snapshots are serialized with engine-specific numeric IDs, which can change between runs.
	if map.content_id_map then
		if not map._cid_remap then
			map._cid_remap = {}
		end
		octchunk.remap_tree_ids(tree, resolve_cid_cached, map)
	end

	map._tree_cache[key] = tree
	map._tree_cache_bytes[key] = tree_bytes
	map._cache_total_bytes = map._cache_total_bytes + tree_bytes
	if map._cache_total_bytes > map._cache_peak_bytes then
		map._cache_peak_bytes = map._cache_total_bytes
	end
	lru_touch(map, key)

	-- Evict until under memory limit
	local cache_bytes_limit = (map.cache_mb or octmap.DEFAULT_CACHE_MB) * 1024 * 1024
	while map._cache_total_bytes > cache_bytes_limit
		and map._tree_cache_lru
		and map._tree_cache_lru.head
		and map._tree_cache_lru.head ~= map._tree_cache_lru.tail do
		local old_key = lru_pop_oldest(map)
		if not old_key then
			break
		end
		local old_bytes = map._tree_cache_bytes[old_key] or 0
		map._cache_total_bytes = map._cache_total_bytes - old_bytes
		map._tree_cache[old_key] = nil
		map._tree_cache_bytes[old_key] = nil
		map._cache_evictions = map._cache_evictions + 1
	end

	return tree
end

-- Internal: exposed for placement/utility code.
octmap._get_tree_cached = get_tree_cached



---Build a linear cache key for a tree position.
---@param map OctMap
---@param gx integer
---@param gy integer
---@param gz integer
---@return integer
function tree_cache_key(map, gx, gy, gz)
	local size = map.trees.size
	return ((gx - 1) * size.y + (gy - 1)) * size.z + gz
end



---Initialize LRU structures on the map.
---@param map OctMap
function lru_init(map)
	map._tree_cache_lru = {head = nil, tail = nil, nodes = {}}
end



---Mark a cache key as most recently used.
---@param map OctMap
---@param key integer
function lru_touch(map, key)
	if not map._tree_cache_lru then
		lru_init(map)
	end
	local lru = map._tree_cache_lru
	local nodes = lru.nodes
	local node = nodes[key]
	if node then
		local prev = node.prev
		local next = node.next
		if prev then
			nodes[prev].next = next
		else
			lru.head = next
		end
		if next then
			nodes[next].prev = prev
		else
			lru.tail = prev
		end
	else
		node = {prev = nil, next = nil}
		nodes[key] = node
	end

	local tail = lru.tail
	if tail then
		nodes[tail].next = key
		node.prev = tail
	else
		lru.head = key
		node.prev = nil
	end
	node.next = nil
	lru.tail = key
end



---Pop the least recently used key.
---@param map OctMap
---@return integer|nil
function lru_pop_oldest(map)
	local lru = map._tree_cache_lru
	if not lru or not lru.head then
		return nil
	end
	local key = lru.head
	local node = lru.nodes[key]
	local next = node and node.next or nil
	lru.head = next
	if next then
		lru.nodes[next].prev = nil
	else
		lru.tail = nil
	end
	if node then
		lru.nodes[key] = nil
	end
	return key
end



---Resolve content ID using cached remap table.
---@param old_cid integer
---@param map OctMap
---@return integer
function resolve_cid_cached(old_cid, map)
	local cached_cid = map._cid_remap[old_cid]
	if cached_cid ~= nil then
		return cached_cid
	end
	local nm = map.content_id_map[old_cid] or core.get_name_from_content_id(old_cid)
	local ok, new_cid = pcall(core.get_content_id, nm)
	if not ok then
		new_cid = core.CONTENT_IGNORE
	end
	map._cid_remap[old_cid] = new_cid
	return new_cid
end
