local serialization = map_octree.serialization
local FORMAT_VERSION = serialization.LATEST_VERSION
-- Format version as single byte for fixed-length header
local MAGIC = serialization.make_header(FORMAT_VERSION)



---Serialize an OctMap to a string for storage.
---@param map OctMap
---@return string|nil serialized_data Serialized map data
---@return string|nil err Error message on failure
function octmap.serialize(map)
	local size = map.trees.size
	local out = {
		requested_pos1 = map.requested_pos1,
		requested_pos2 = map.requested_pos2,
		minp = map.minp,
		maxp = map.maxp,
		default_node = map.default_node,
		content_id_map = map.content_id_map,         -- global content_id->name (nil if using palettes)
		trees = matrix3d.new(size.x, size.y, size.z, nil), -- sparse
	}

	matrix3d.iterate(map.trees, function(x, y, z)
		local tree = matrix3d.get(map.trees, x, y, z)
		if tree == nil then
			-- sparse: keep nil
			return
		elseif type(tree) == "string" then
			matrix3d.set(out.trees, x, y, z, tree)
		else
			matrix3d.set(out.trees, x, y, z, octchunk.serialize(tree))
		end
	end)

	-- Trees use zstd level 3 for memory efficiency.
	-- Compress entire map with zstd level 9 for disk storage (disk I/O is infrequent).
	local ok_ser, raw = pcall(core.serialize, out)
	if not ok_ser then
		return nil, "serialize failed"
	end
	---@diagnostic disable-next-line: param-type-mismatch
	local ok_comp, compressed = pcall(core.compress, raw, "zstd", 9)
	if not ok_comp then
		return nil, "compress failed"
	end
	return MAGIC .. compressed, nil
end



---Deserialize an OctMap from a string.
---@param data string Compressed serialized map data
---@param opts? {deserialize_trees?: boolean}
---@return OctMap|nil map The deserialized map object
---@return string|nil err Error message on failure
function octmap.deserialize(data, opts)
	opts = opts or {}
	local prefix = serialization.MAGIC_PREFIX
	if data:sub(1, #prefix) ~= prefix then
		return nil, "invalid or unsupported map header"
	end
	local version = data:byte(#prefix + 1)
	if not version then
		return nil, "invalid map header"
	end
	if version > serialization.LATEST_VERSION then
		return nil, "unsupported map format"
	end
	local payload = data:sub(#prefix + 2)
	local ok_decomp, decompressed = pcall(core.decompress, payload, "zstd")
	if not ok_decomp then
		return nil, "decompress failed"
	end
	local ok_deser, map = pcall(core.deserialize, decompressed)
	if not ok_deser or type(map) ~= "table" then
		return nil, "deserialize failed"
	end

	if version < serialization.LATEST_VERSION then
		local migrated, err = serialization.migrate(map, version)
		if not migrated then
			return nil, err
		end
		map = migrated
	end

	-- Deserialize trees if requested (expensive - only for debugging)
	if opts.deserialize_trees then
		matrix3d.iterate(map.trees, function(x, y, z)
			local tree = matrix3d.get(map.trees, x, y, z)
			if type(tree) == "string" then
				matrix3d.set(map.trees, x, y, z, octchunk.deserialize(tree))
			end
		end)
	end

	return octmap.attach_methods(map), nil
end



---Load an OctMap from a file.
---@param name string File path
---@return OctMap|nil map The loaded map object
---@return string|nil err Error message on failure
function octmap.read_from_file(name)
	local file = io.open(name, "rb")
	if file then
		local t0 = core.get_us_time()
		local raw = file:read("*all")
		file:close()
		local t_read = core.get_us_time()
		local map, err = octmap.deserialize(raw)
		local t_deser = core.get_us_time()
		if map then
			core.log("action", string.format("[octmap] Load timing: read=%.1fms, deserialize=%.1fms, file_size=%.1fKB",
				(t_read - t0) / 1000, (t_deser - t_read) / 1000, #raw / 1024))
			return map
		end
		core.log("warning", "[octmap] Load failed: " .. tostring(err))
		return nil, err
	end
	core.log("warning", "[octmap] File doesn't exist: " .. name)
end
