local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

local function get_chunk_bounds(center)
	local half = octchunk.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.add(center, {x = half - 1, y = half - 1, z = half - 1})
	return pmin, pmax
end


map_octree.tests.register("OctreeManip: build_opts, read_ranges, size", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = testutil.get_test_region(ctx.player)
	local center = octchunk.snap_to_center(vector.add(base, {x = 0, y = 0, z = 9856}))
	local pmin, pmax = get_chunk_bounds(center)

	testutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		local cid_air = core.get_content_id("air")
		local cid_dirt = core.get_content_id("default:dirt")
		assert(type(cid_air) == "number", "missing node: air")
		assert(type(cid_dirt) == "number", "missing node: default:dirt")

		for z = pmin.z, pmax.z do
			for y = pmin.y, pmax.y do
				for x = pmin.x, pmax.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_air
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end
		data[area:index(center.x, center.y, center.z)] = cid_dirt
		if param2_data then param2_data[area:index(center.x, center.y, center.z)] = 4 end

		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip({
			build_opts = {
				store_chunk_blobs = false,
				cache_mb = 2,
				force_batches = true,
				max_voxelmanip_volume = (octchunk.SIZE + 1) ^ 3 * 2,
			},
		})
		m:read_from_map(pmin, pmax)

		local e1, e2 = m:get_emerged_area()
		local size = m:size()
		assert(size.x == (e2.x - e1.x + 1), "OctreeManip:size x mismatch")
		assert(size.y == (e2.y - e1.y + 1), "OctreeManip:size y mismatch")
		assert(size.z == (e2.z - e1.z + 1), "OctreeManip:size z mismatch")

		local ranges = {{pos1 = e1, pos2 = e2}}
		local count = m:read_ranges(ranges, function() end, {grid = "node"})
		assert(count == octchunk.SIZE ^ 3, "OctreeManip:read_ranges count mismatch")
	end)
end)


map_octree.tests.register("OctreeManip: clear_deltas + setters", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")

	local base = testutil.get_test_region(ctx.player)
	local center = octchunk.snap_to_center(vector.add(base, {x = 0, y = 0, z = 9984}))
	local pmin, pmax = get_chunk_bounds(center)

	local node_a = "default:stone"
	local node_b = "default:dirt"
	local cid_a = core.get_content_id(node_a)
	local cid_b = core.get_content_id(node_b)
	assert(type(cid_a) == "number", "missing node: " .. node_a)
	assert(type(cid_b) == "number", "missing node: " .. node_b)

	local target = vector.add(pmin, {x = 6, y = 6, z = 6})

	testutil.with_voxel_region(pmin, pmax, function(manip, area, data, param2_data)
		for z = pmin.z, pmax.z do
			for y = pmin.y, pmax.y do
				for x = pmin.x, pmax.x do
					local idx = area:index(x, y, z)
					data[idx] = cid_a
					if param2_data then param2_data[idx] = 0 end
				end
			end
		end
		manip:set_data(data)
		if manip.set_param2_data and param2_data then
			manip:set_param2_data(param2_data)
		end
		manip:write_to_map(false)

		local m = map_octree.new_octree_manip()
		m:read_from_map(pmin, pmax)

		assert(m:set_node_at(target.x, target.y, target.z, node_b, 4), "set_node_at failed")
		local name1, p2_1 = m:get_node_at(target.x, target.y, target.z)
		assert(name1 == node_b, "set_node_at name mismatch")
		assert(p2_1 == 4, "set_node_at param2 mismatch")

		assert(m:set_node_cid_at(target.x, target.y, target.z, cid_a, 7), "set_node_cid_at failed")
		local name2, p2_2 = m:get_node_at(target.x, target.y, target.z)
		assert(name2 == node_a, "set_node_cid_at name mismatch")
		assert(p2_2 == 7, "set_node_cid_at param2 mismatch")

		m:clear_deltas()
		local name3 = select(1, m:get_node_at(target.x, target.y, target.z))
		assert(name3 == node_a, "clear_deltas should restore snapshot view")

		assert(m:write_to_map(), "write_to_map after clear_deltas failed")

		core.load_area(pmin, pmax)
		local vm = core.get_voxel_manip()
		local e1, e2 = vm:read_from_map(pmin, pmax)
		local va = VoxelArea(e1, e2)
		local out = {}
		vm:get_data(out)
		if vm.close then vm:close() end

		local idx = va:index(target.x, target.y, target.z)
		assert(out[idx] == cid_a, "clear_deltas should prevent live changes")
	end)
end)