--- Chat commands for testing map_octree.
-- Provides commands for saving/loading snapshots, placing them, and debugging.
-- Local functions used during registration are defined inline.

local modname = core.get_current_modname()
local S = core.get_translator(modname)

local get_player_positions
local parse_pos_param
local parse_load_flags
local build_map_key_for_command

local defs = {}
---Register a chatcommand definition.
---@param name string
---@param def table
local function register(name, def)
    defs[name] = def
end

-- Position markers (like WorldEdit)
map_octree.markers = {}



-- Get player positions with fallback to WorldEdit if available
register("octree_pos1", {
    description = S("Set octree position 1 to your current location (or copy from WorldEdit)"),
    privs = {server = true},
    func = function(name)
        local player = core.get_player_by_name(name)
        if not player then return false, S("Player not found") end

        local pos = vector.round(player:get_pos())
        map_octree.markers[name] = map_octree.markers[name] or {}
        map_octree.markers[name].pos1 = pos
        ---@cast pos vector
        map_octree.add_node_marker(pos)

        return true, S("Position 1 set: @1", core.pos_to_string(pos))
    end
})



register("octree_pos2", {
    description = S("Set octree position 2 to your current location (or copy from WorldEdit)"),
    privs = {server = true},
    func = function(name)
        local player = core.get_player_by_name(name)
        if not player then return false, S("Player not found") end

        local pos = vector.round(player:get_pos())
        map_octree.markers[name] = map_octree.markers[name] or {}
        map_octree.markers[name].pos2 = pos
        ---@cast pos vector
        map_octree.add_node_marker(pos)

        return true, S("Position 2 set: @1", core.pos_to_string(pos))
    end
})



register("octree_save", {
    params = "<name> [<radius> [<height>]] [-sync] [inflight=<num>]",
    description = S("Save the selected area to a snapshot file under worldpath/map_octree"),
    privs = {server = true},
    func = function(name, param)
        local player = core.get_player_by_name(name)
        if not player then return false, S("Player not found") end

        param = param or ""
        local file_name, rest = param:match("^%s*(%S+)%s*(.-)%s*$")
        if not file_name or file_name == "" then
            return false, S("Missing name. Usage: /octree_save <name> [<radius> [<height>]] [-sync] [inflight=<n>]")
        end

        local is_sync = rest:match("-sync") ~= nil
        rest = rest:gsub("-sync", ""):match("^%s*(.-)%s*$")

        -- Extract inflight parameter before parsing radius/height
        local inflight_override = nil
        local inflight_match = rest:match("inflight=(%d+)")
        if inflight_match then
            local inflight_value = tonumber(inflight_match) or 1
            inflight_override = math.max(1, math.floor(inflight_value))
            rest = rest:gsub("inflight=%d+", ""):match("^%s*(.-)%s*$")
        end

        local radius, height
        local use_radius = false
        do
            local r, h = rest:match("^(%d+)%s*(%d*)$")
            if r then
                use_radius = true
                radius = tonumber(r)
                height = tonumber(h)
            end
        end

        local pos1, pos2 = get_player_positions(name)
        local source_desc

        -- If radius is provided, it overrides any existing pos1/pos2 selection.
        if use_radius or not pos1 or not pos2 then
            radius = radius or 100
            height = height or 128
            local ppos = player:get_pos()
            pos1 = vector.new(ppos.x - radius, ppos.y - height, ppos.z - radius)
            pos2 = vector.new(ppos.x + radius, ppos.y + height, ppos.z + radius)
            source_desc = S("radius=@1 height=@2", radius, height)
        else
            source_desc = S("pos1/pos2")
        end

        local start_time = core.get_us_time()

        if is_sync then
            local map, err = map_octree.save_to_file(pos1, pos2, {file_name = file_name})
            if not map then
                return false, S("Save failed: @1", tostring(err))
            end
            local p1, p2 = vector.sort(pos1, pos2)
            local elapsed = (core.get_us_time() - start_time) / 1000
            local chunk_count = map.trees.size.x * map.trees.size.y * map.trees.size.z
            map_octree.last_map = map
            return true, S(
                "Saved @1 chunks in @2ms from @3 to @4 (sync, @5)",
                chunk_count, string.format("%.1f", elapsed), core.pos_to_string(p1), core.pos_to_string(p2), source_desc
            )
        end

        -- Default: use engine's async worker capacity (num_cpus - 2)
        ---@diagnostic disable-next-line: undefined-field
        local inflight = math.max(1, core.get_async_threading_capacity() or 4)
        -- Check for autotune override (if set and non-zero, use it instead)
        local override = tonumber(core.settings:get("map_octree_async_inflight_override")) or 0
        if override > 0 then
            inflight = override
        end
        -- Apply command parameter override if specified
        if inflight_override then
            inflight = inflight_override
        end

        core.chat_send_player(name, S("Starting async save (@1)...", source_desc))
        local plan = map_octree.save_to_file_async(pos1, pos2, {async_inflight = inflight, file_name = file_name}, function(ok, result, stats)
            local elapsed = (core.get_us_time() - start_time) / 1000
            if not ok then
                core.chat_send_player(name, S("Async save FAILED: @1", tostring(result)))
                return
            end
            ---@cast result OctMap
            local chunk_count = result.trees.size.x * result.trees.size.y * result.trees.size.z
            core.chat_send_player(name, S(
                "Async save complete: @1 chunks in @2ms (batches=@3, inflight=@4)",
                chunk_count, string.format("%.1f", elapsed), stats.batch_count, inflight
            ))
            map_octree.last_map = result
        end)

        if not plan then
            return false, S("Async save failed to start")
        end
        return true, S("Async save '@1' started with @2 batches (inflight=@3)", file_name, #plan.batches, inflight)
    end
})



register("octree_autotune", {
    params = "[budget_ms] [-apply]",
    description = S("Empirical benchmark: measures real save_to_file_async throughput for each batch size"),
    privs = {server = true},
    func = map_octree.autotune.run,
})



register("octree_load", {
    params = "<name> [-subdir=DIR] [-no-current]",
    description = S("Load a snapshot file from worldpath/map_octree into memory"),
    privs = {server = true},
    func = function(name, param)
        local cleaned, flags = parse_load_flags(param)
        local file_name = cleaned:match("^%s*(%S+)%s*$")
        if not file_name or file_name == "" then
            return false, "Missing name. Usage: /octree_load <name> [-subdir=DIR] [-no-current]"
        end

        local start_time = core.get_us_time()
        local map, err = map_octree.load_map(file_name, {
            subdir = flags.subdir,
            set_current = not flags.no_current,
        })
        local elapsed = (core.get_us_time() - start_time) / 1000

        if map then
            local where = flags.subdir and flags.subdir ~= "" and (" (subdir=" .. flags.subdir .. ")") or ""
            return true, string.format("Loaded map '%s'%s in %.1fms", file_name, where, elapsed)
        else
            return false, err or "No saved map found"
        end
    end
})



register("octree_maps", {
    params = "[-subdir=DIR]",
    description = S("List snapshot files saved on disk in worldpath/map_octree"),
    privs = {server = true},
    func = function(p_name, param)
        local _, flags = parse_load_flags(param)
        local subdir = flags.subdir

        local dir
        if map_octree.get_storage_dir then
            dir = map_octree.get_storage_dir(subdir)
        else
            local base = core.get_worldpath() .. "/map_octree"
            dir = subdir and subdir ~= "" and (base .. "/" .. subdir) or base
        end

        local files = core.get_dir_list(dir, false)
        local maps = {}
        for _, fname in ipairs(files) do
            if fname:sub(-4) == ".bin" then
                maps[#maps + 1] = fname:sub(1, -5)
            end
        end
        table.sort(maps)

        if #maps == 0 then
            return true, S("No saved octree maps found")
        end

        return true, S("Saved octree maps (@1):\n@2", #maps, table.concat(maps, "\n"))
    end
})



register("octree_loaded", {
    description = S("List snapshots currently loaded in memory"),
    privs = {server = true},
    func = function()
        local keys = map_octree.list_loaded_maps()
        if #keys == 0 then
            return true, S("No loaded octree maps in memory")
        end
        return true, S("Loaded octree maps (@1):\n@2", #keys, table.concat(keys, "\n"))
    end
})



register("octree_unload", {
    params = "<key>",
    description = S("Unload a snapshot from memory"),
    privs = {server = true},
    func = function(p_name, param)
        local cleaned = parse_load_flags(param)
        local key = cleaned:match("^%s*(%S+)%s*$")
        if not key or key == "" then
            return false, "Missing key. Usage: /octree_unload <key>"
        end

        local ok, err = map_octree.unload_map(key)
        if ok then
            return true, S("Unloaded map '@1'", key)
        end
        return false, "Unload failed: " .. tostring(err)
    end
})



register("octree_place", {
    params = "<name> [x,y,z] [-sync] [-subdir=DIR]",
    description = S("Paste a saved snapshot into the world (defaults to its original position)"),
    privs = {server = true},
    func = function(name, param)
        local player = core.get_player_by_name(name)
        if not player then return false, "Player not found" end

        local cleaned, flags = parse_load_flags(param)
        local file_name, rest = cleaned:match("^%s*(%S+)%s*(.-)%s*$")
        if not file_name or file_name == "" then
            return false, "Missing name. Usage: /octree_place <name> [x,y,z] [-sync] [-subdir=DIR]"
        end

        local is_sync = rest:match("-sync") ~= nil
        rest = rest:gsub("-sync", ""):match("^%s*(.-)%s*$")

        local pos = parse_pos_param(rest)
        if not pos then
            -- No position specified - use original saved position (nil = place at original location)
            pos = nil
        end

        local map = map_octree.load_map(file_name, {subdir = flags.subdir})
        local label = file_name
        if not map then
            return false, string.format("Failed to load map '%s'", label)
        end

        local start_time = core.get_us_time()

        if is_sync then
            local ok, err = map_octree.place(map, pos, {})
            local elapsed = (core.get_us_time() - start_time) / 1000
            if ok then
                local placed_at = pos or (map:get_emerged_area())
                return true, string.format("Placed '%s' at %s in %.1fms (sync)", label, core.pos_to_string(placed_at), elapsed)
            else
                return false, string.format("Place failed: %s", err)
            end
        end

        core.chat_send_player(name, "Starting async place...")
        map_octree.place_async(map, pos, {}, function(ok, err)
            local elapsed = (core.get_us_time() - start_time) / 1000
            if not ok then
                core.chat_send_player(name, "Async place FAILED: " .. tostring(err))
                return
            end
            local placed_at = pos or (map:get_emerged_area())
            core.chat_send_player(name, string.format(
                "Async place complete: '%s' at %s in %.1fms",
                label, core.pos_to_string(placed_at), elapsed
            ))
        end)

        return true, string.format("Async place '%s' started", label)
    end
})



register("octree_query", {
    description = S("Query the node at your current position from the loaded snapshot"),
    privs = {server = true},
    func = function(name)
        local player = core.get_player_by_name(name)
        if not player then
            return false, "Player not found"
        end
        assert(map_octree.current_map, "No map in memory")

        local pos = vector.round(player:get_pos())
        local get_node = assert(map_octree.get_node_at, "get_node_at unavailable")
        ---@diagnostic disable-next-line: need-check-nil
        local node_name, param2 = get_node(pos.x, pos.y, pos.z)

        if not node_name then
            return false, string.format("Position %s is outside the saved area (min=%s, max=%s)",
                core.pos_to_string(pos),
                core.pos_to_string(map_octree.current_map.minp),
                core.pos_to_string(map_octree.current_map.maxp))
        end

        return true, string.format("Node at %s: %s (param2=%d)",
            core.pos_to_string(pos), node_name, tonumber(param2) or 0)
    end
})



register("octree_info", {
    description = S("Show info about the currently loaded snapshot"),
    privs = {server = true},
    func = function(name)
        local map = map_octree.current_map
        local size = map.trees.size
        local chunk_count = size.x * size.y * size.z
        local area_size = vector.subtract(map.maxp, map.minp)

        return true, string.format(
            "Map info:\n" ..
            "  Min: %s\n" ..
            "  Max: %s\n" ..
            "  Area: %dx%dx%d blocks\n" ..
            "  Chunks: %d (%dx%dx%d grid)",
            core.pos_to_string(map.minp),
            core.pos_to_string(map.maxp),
            area_size.x, area_size.y, area_size.z,
            chunk_count, size.x, size.y, size.z
        )
    end
})



register("octree_clear", {
    description = S("Clear the current snapshot from memory"),
    privs = {server = true},
    func = function()
        local before_kb = collectgarbage("count")
        map_octree.current_map = nil
        map_octree.last_map = nil
        octcache.delete()
        collectgarbage("collect")
        collectgarbage("collect")
        local after_kb = collectgarbage("count")
        return true, string.format("Octree map cleared. Lua memory: %.1f MB -> %.1f MB (freed %.1f MB)",
            before_kb / 1024, after_kb / 1024, (before_kb - after_kb) / 1024)
    end
})



register("octree_mem", {
    description = S("Show Lua memory usage"),
    privs = {server = true},
    func = function()
        collectgarbage("collect")
        local kb = collectgarbage("count")
        local map_info = ""
        if map_octree.current_map then
            local m = map_octree.current_map
            local chunk_count = m.trees.size.x * m.trees.size.y * m.trees.size.z
            map_info = string.format(" | Map: %d chunks", chunk_count)
            if m._cache_total_bytes then
                map_info = map_info .. string.format(", cache: %.1f MB", m._cache_total_bytes / 1024 / 1024)
            end
        end

        local cache_count = 0
        if octcache and octcache.trees_cache then
            for _ in pairs(octcache.trees_cache) do
                cache_count = cache_count + 1
            end
        end

        return true, string.format("Lua: %.1f MB | Octcache: %d entries%s", kb / 1024, cache_count, map_info)
    end
})



register("octree_test", {
    description = S("Run octree self-tests"),
    privs = {server = true},
    func = function(name)
        local player = core.get_player_by_name(name)
        if not player then
            return false, S("Player not found")
        end

        if not (map_octree.tests and map_octree.tests.run_all) then
            local modpath = core.get_modpath("map_octree")
            dofile(modpath .. "/src/tests/runner.lua")
        end

        local result = map_octree.tests.run_all({player = player})
        if result.failed == 0 then
            return true, S("All tests passed (@1) in @2ms", result.total, string.format("%.1f", result.ms))
        end

        local msg = S("Tests failed: @1/@2 failed (@3ms)\n", result.failed, result.total, string.format("%.1f", result.ms))
        msg = msg .. table.concat(result.failures, "\n")
        return false, msg
    end
})



register("octree_debug", {
    params = "[on|off]",
    description = S("Toggle or set wireframes/markers (debug visuals)"),
    privs = {server = true},
    func = function(name, param)
        if param == "on" then
            map_octree.wireframe_enabled = true
        elseif param == "off" then
            map_octree.wireframe_enabled = false
        else
            map_octree.wireframe_enabled = not map_octree.wireframe_enabled
        end
        return true, "Wireframes: " .. (map_octree.wireframe_enabled and "ON" or "OFF")
    end
})



register("octree_log", {
    params = "[on|off]",
    description = S("Toggle or set debug logging (timings/perf logs)"),
    privs = {server = true},
    func = function(name, param)
        if param == "on" then
            map_octree.log_enabled = true
        elseif param == "off" then
            map_octree.log_enabled = false
        else
            map_octree.log_enabled = not map_octree.log_enabled
        end
        return true, "Debug logging: " .. (map_octree.log_enabled and "ON" or "OFF")
    end
})
map_octree._bench = {}



register("octree_bench", {
    params = "[<count>]",
    description = S("Full benchmark: hot, cold, cold cache=0. Reports total_us + call_us + call_p50/call_p95."),
    privs = {server = true},
    func = function(name, param)
        if map_octree._bench[name] then
            return false, "Benchmark already running"
        end

        local map = map_octree.current_map or map_octree.last_map
        if not map then
            return false, "No map in memory. Use /octree_save or /octree_load first."
        end

        local count = tonumber(param:match("^(%d+)")) or 2000
        count = math.max(100, math.floor(count))

        local player = core.get_player_by_name(name)
        local hot_pos = player and vector.round(player:get_pos()) or map.minp

        local size = map.trees.size
        local total_chunks = size.x * size.y * size.z
        local old_cache_mb = map.cache_mb
        local default_cache_mb = (old_cache_mb ~= nil) and old_cache_mb or (octmap and octmap.DEFAULT_CACHE_MB) or 256

        ---Compute cache key for chunk coordinates.
        ---@param gx integer
        ---@param gy integer
        ---@param gz integer
        ---@return integer
        local function chunk_key(gx, gy, gz)
            return ((gx - 1) * size.y + (gy - 1)) * size.z + gz
        end

        ---Clear tree cache and counters on the map.
        local function clear_cache()
            map._tree_cache = nil
            map._tree_cache_lru = nil
            map._tree_cache_bytes = nil
            map._cache_total_bytes = nil
            map._cache_peak_bytes = nil
            map._cache_evictions = nil
        end

        ---Reset cache counters while keeping cached data.
        local function reset_cache_counters_keep_data()
            -- Keep the hot cache (if any), but reset counters so phases report only their own work.
            map._cache_evictions = 0
            map._cache_peak_bytes = map._cache_total_bytes or 0
        end

        ---Convert a position to chunk grid coordinates.
        ---@param pos vector
        ---@return integer gx
        ---@return integer gy
        ---@return integer gz
        local function pos_to_chunk_gxyz(pos)
            local snapped = octchunk.snap_to_center(pos)
            local gx = 1 + math.floor((snapped.x - map.minp.x) / octchunk.SIZE)
            local gy = 1 + math.floor((snapped.y - map.minp.y) / octchunk.SIZE)
            local gz = 1 + math.floor((snapped.z - map.minp.z) / octchunk.SIZE)
            -- Clamp to valid grid bounds
            local size = map.trees.size
            gx = math.max(1, math.min(size.x, gx))
            gy = math.max(1, math.min(size.y, gy))
            gz = math.max(1, math.min(size.z, gz))
            return gx, gy, gz
        end

        ---Pick a random position and its chunk coordinates.
        ---@param i integer
        ---@return vector pos
        ---@return integer gx
        ---@return integer gy
        ---@return integer gz
        local function cold_pos_for(i)
            -- Random position within map bounds (not sequential)
            local rx = map.minp.x + math.random(0, map.maxp.x - map.minp.x)
            local ry = map.minp.y + math.random(0, map.maxp.y - map.minp.y)
            local rz = map.minp.z + math.random(0, map.maxp.z - map.minp.z)
            local pos = vector.new(rx, ry, rz)
            local gx, gy, gz = pos_to_chunk_gxyz(pos)
            return pos, gx, gy, gz
        end

        -- For contiguous test: pick a random region center, then iterate within it
        local contig_region_size = 64 -- read 64³ contiguous nodes per region
        local contig_idx = 0
        local contig_positions = nil

        ---Initialize contiguous region sampling positions.
        local function init_contiguous_region()
            -- Clamp region size to fit map dimensions (minimum 1)
            local map_sx = map.maxp.x - map.minp.x + 1
            local map_sy = map.maxp.y - map.minp.y + 1
            local map_sz = map.maxp.z - map.minp.z + 1
            local region = math.max(1, math.min(contig_region_size, map_sx, map_sy, map_sz))
            local half = math.floor(region / 2)
            -- Pick random region center within map (clamped to valid range)
            local cx = math.random(map.minp.x, map.maxp.x)
            local cy = math.random(map.minp.y, map.maxp.y)
            local cz = math.random(map.minp.z, map.maxp.z)
            contig_idx = 0
            contig_positions = {}
            -- Generate positions around center, clamped to map bounds
            for x = -half, half do
                for y = -half, half do
                    for z = -half, half do
                        local px = math.max(map.minp.x, math.min(map.maxp.x, cx + x))
                        local py = math.max(map.minp.y, math.min(map.maxp.y, cy + y))
                        local pz = math.max(map.minp.z, math.min(map.maxp.z, cz + z))
                        contig_positions[#contig_positions + 1] = vector.new(px, py, pz)
                    end
                end
            end
            -- Shuffle to avoid perfectly sequential access
            for i = #contig_positions, 2, -1 do
                local j = math.random(i)
                contig_positions[i], contig_positions[j] = contig_positions[j], contig_positions[i]
            end
        end

        ---Get next position in contiguous sampling mode.
        ---@param i integer
        ---@return vector pos
        ---@return integer gx
        ---@return integer gy
        ---@return integer gz
        local function contiguous_pos_for(i)
            contig_idx = contig_idx + 1
            if contig_idx > #contig_positions then
                init_contiguous_region()
                contig_idx = 1
            end
            ---@diagnostic disable-next-line: need-check-nil
            local pos = contig_positions[contig_idx]
            if not pos then
                init_contiguous_region()
                contig_idx = 1
                ---@diagnostic disable-next-line: need-check-nil
                pos = contig_positions[contig_idx]
            end
            pos = assert(pos, "contiguous position missing")
            ---@cast pos vector
            local gx, gy, gz = pos_to_chunk_gxyz(pos)
            return pos, gx, gy, gz
        end

        ---Return percentile from a sorted array.
        ---@param sorted number[]
        ---@param p number
        ---@return number
        local function percentile(sorted, p)
            local n = #sorted
            if n == 0 then return 0 end
            local idx = math.max(1, math.ceil(n * p))
            return sorted[idx]
        end

        local phases = {
            {name = "hot",                cache_mb = default_cache_mb, is_hot = true,  mode = "hot"},
            {name = "cold_random",        cache_mb = default_cache_mb, is_hot = false, mode = "random"},
            {name = "cold_contiguous",    cache_mb = default_cache_mb, is_hot = false, mode = "contiguous"},
            {name = "cold_random_nc",     cache_mb = 0,                is_hot = false, mode = "random"},
            {name = "cold_contiguous_nc", cache_mb = 0,                is_hot = false, mode = "contiguous"},
        }
        local results = {}

        local phase_idx = 0
        local state = nil

        ---Run a benchmark phase.
        local function run_phase()
            phase_idx = phase_idx + 1
            if phase_idx > #phases then
                map.cache_mb = old_cache_mb
                map_octree._bench[name] = nil

                core.chat_send_player(name, "=== Benchmark Results ===")
                for _, r in ipairs(results) do
                    core.chat_send_player(name, string.format(
                        "%s: call=%.1fus p50=%.1fus p95=%.1fus | hit=%d miss=%d sparse=%d evict=%d peak=%.1fMB",
                        r.name, r.call_us, r.call_p50_us, r.call_p95_us, r.hit, r.miss, r.sparse, r.evictions, r.cache_peak_mb
                    ))
                end
                core.chat_send_player(name, "=========================")
                return
            end

            local phase = phases[phase_idx]
            map.cache_mb = phase.cache_mb
            clear_cache()

            if phase.is_hot then
                octmap.get_node_name(map, hot_pos)
                reset_cache_counters_keep_data()
            end

            local hot_gx, hot_gy, hot_gz = pos_to_chunk_gxyz(hot_pos)

            state = {
                i = 0,
                total = count,
                hit = 0,
                miss = 0,
                sparse = 0,
                sum_total_us = 0,
                sum_call_us = 0,
                call_samples = {},
                call_sample_rate = (count <= 20000) and 1 or math.max(1, math.floor(count / 2000)),
                phase = phase,
            }

            -- Init contiguous region if needed
            if phase.mode == "contiguous" then
                init_contiguous_region()
            end

            core.chat_send_player(name, string.format(
                "Running %s: %d queries | cache=%dMB",
                phase.name, count, phase.cache_mb
            ))

            ---Process a batch of benchmark queries.
            local function step()
                local per_step = 200
                local processed = 0

                while state.i < state.total and processed < per_step do
                    state.i = state.i + 1
                    processed = processed + 1

                    local pos, gx, gy, gz
                    if phase.is_hot then
                        pos = hot_pos
                        gx, gy, gz = hot_gx, hot_gy, hot_gz
                    elseif phase.mode == "contiguous" then
                        pos, gx, gy, gz = contiguous_pos_for(state.i)
                    else
                        pos, gx, gy, gz = cold_pos_for(state.i)
                    end

                    -- Count cache stats: sparse cells don't deserialize, don't count as miss
                    local cell = matrix3d.get(map.trees, gx, gy, gz)
                    if cell == nil then
                        state.sparse = state.sparse + 1
                    else
                        local cached = map._tree_cache and map._tree_cache[chunk_key(gx, gy, gz)] ~= nil
                        if cached then
                            state.hit = state.hit + 1
                        else
                            state.miss = state.miss + 1
                        end
                    end

                    local iter_t0 = core.get_us_time()
                    local call_dt
                    if phase.is_hot then
                        -- For very fast hot-path calls, batch multiple calls to reduce timer quantization.
                        local reps = 16
                        local t0 = core.get_us_time()
                        for _ = 1, reps do
                            octmap.get_node_name(map, pos)
                        end
                        call_dt = (core.get_us_time() - t0) / reps
                    else
                        local t0 = core.get_us_time()
                        octmap.get_node_name(map, pos)
                        call_dt = core.get_us_time() - t0
                    end

                    local iter_dt = core.get_us_time() - iter_t0
                    state.sum_total_us = state.sum_total_us + iter_dt
                    state.sum_call_us = state.sum_call_us + call_dt

                    if state.i % state.call_sample_rate == 0 then
                        state.call_samples[#state.call_samples + 1] = call_dt
                    end
                end

                if state.i >= state.total then
                    local total_us = state.sum_total_us / state.total
                    local call_us = state.sum_call_us / state.total

                    table.sort(state.call_samples)
                    local call_p50 = percentile(state.call_samples, 0.50)
                    local call_p95 = percentile(state.call_samples, 0.95)

                    results[#results + 1] = {
                        name = phase.name,
                        total_us = total_us,
                        call_us = call_us,
                        call_p50_us = call_p50,
                        call_p95_us = call_p95,
                        hit = state.hit,
                        miss = state.miss,
                        sparse = state.sparse,
                        evictions = map._cache_evictions or 0,
                        cache_peak_mb = (map._cache_peak_bytes or 0) / (1024 * 1024),
                    }

                    core.after(0, run_phase)
                    return
                end

                core.after(0, step)
            end

            core.after(0, step)
        end

        map_octree._bench[name] = true
        core.chat_send_player(name, string.format(
            "Starting full benchmark: %d queries/phase | chunks=%d",
            count, total_chunks
        ))
        core.after(0, run_phase)
        return true, "Benchmark started"
    end
})


register("octree_track", {
    params = "<name> [-subdir=DIR]",
    description = S("Enable restore tracking for a saved snapshot"),
    privs = {server = true},
    func = function(p_name, param)
        local cleaned, flags = parse_load_flags(param)
        local file_name = cleaned:match("^%s*(%S+)%s*$")
        if not file_name or file_name == "" then
            return false, "Usage: /octree_track <name> [-subdir=DIR]"
        end

        local key = build_map_key_for_command(file_name, flags.subdir)
        local map = map_octree.get_loaded_map(key)
        if not map then
            map = map_octree.load_map(file_name, {subdir = flags.subdir, set_current = false})
        end

        if not map then
            return false, "Failed to load map '" .. key .. "'"
        end

        map:enable_tracking()
        map_octree._active_tracker_map = map
        map_octree._active_tracker_key = key

        return true, string.format(
            "Tracking enabled for '%s' at %s",
            key, core.pos_to_string(map.minp)
        )
    end
})


register("octree_track_status", {
    params = "[-key=KEY]",
    description = S("Show restore tracking status for a snapshot"),
    privs = {server = true},
    func = function(p_name, param)
        local _, flags = parse_load_flags(param)
        local key = flags.key or map_octree._active_tracker_key
        local map = key and map_octree.get_loaded_map(key) or map_octree._active_tracker_map
        if not map or not map._tracker_id then
            return false, "No active tracker. Use /octree_track first."
        end

        local status = map:get_tracking_status()
        if not status then
            return false, "Tracker not found"
        end

        local size_str = "?"
        if status.chunk_size then
            size_str = string.format("%dx%dx%d", status.chunk_size.x, status.chunk_size.y, status.chunk_size.z)
        end

        local bounds_str = "bounds=?"
        if status.base_corner and status.max_corner then
            bounds_str = string.format(
                "bounds=%s..%s",
                core.pos_to_string(status.base_corner),
                core.pos_to_string(status.max_corner)
            )
        end

        local block_bounds_str = "block_bounds=?"
        if status.min_blockpos and status.max_blockpos then
            block_bounds_str = string.format(
                "block_bounds=%s..%s",
                core.pos_to_string(status.min_blockpos),
                core.pos_to_string(status.max_blockpos)
            )
        end

        local last_str = "last=none"
        if status.last_change then
            last_str = string.format(
                "last: blocks=%d seen=%d in=%d out=%d enq=%d",
                status.last_change.modified_block_count,
                status.last_change.blocks_seen or 0,
                status.last_change.blocks_in_bounds,
                status.last_change.blocks_out_of_bounds or 0,
                status.last_change.unique_chunks_enqueued
            )
        end

        return true, string.format(
            "Tracker: pending=%d, dirty=%d, chunks=%s, pos1=%s, %s, %s, %s",
            status.pending,
            status.dirty,
            size_str,
            core.pos_to_string(map.minp),
            bounds_str,
            block_bounds_str,
            last_str
        )
    end
})



register("octree_restore", {
    params = "[-sync] [-key=KEY]",
    description = S("Restore modified nodes from a tracked snapshot"),
    privs = {server = true},
    func = function(p_name, param)
        local cleaned, flags = parse_load_flags(param)
        local is_sync = cleaned:match("-sync") ~= nil
        local key = flags.key or map_octree._active_tracker_key
        local map = key and map_octree.get_loaded_map(key) or map_octree._active_tracker_map
        if not map or not map._tracker_id then
            return false, "No active tracker. Use /octree_track first."
        end

        local status = map:get_tracking_status()
        if not status then
            return false, "Tracker not found"
        end

        if status.dirty == 0 then
            return true, "No dirty chunks to restore"
        end

        local start_time = core.get_us_time()

        if is_sync then
            local ok, err = map:restore()
            local elapsed = (core.get_us_time() - start_time) / 1000
            if ok then
                return true, string.format("Restored %d chunks in %.1fms (sync)", status.dirty, elapsed)
            end
            return false, "Restore failed: " .. tostring(err)
        end

        local dirty_count = status.dirty
        core.chat_send_player(p_name, string.format("Restoring %d dirty chunks...", dirty_count))
        map:schedule_restore(function(ok, err)
            local elapsed = (core.get_us_time() - start_time) / 1000
            if ok then
                core.chat_send_player(p_name, string.format(
                    "Restored %d chunks in %.1fms (async)", dirty_count, elapsed
                ))
            else
                core.chat_send_player(p_name, "Restore failed: " .. tostring(err))
            end
        end)
        return true, "Async restore started"
    end
})



register("octree_untrack", {
    params = "[-key=KEY]",
    description = S("Disable restore tracking for a snapshot"),
    privs = {server = true},
    func = function(p_name, param)
        local _, flags = parse_load_flags(param)
        local key = flags.key or map_octree._active_tracker_key
        local map = key and map_octree.get_loaded_map(key) or map_octree._active_tracker_map
        if not map or not map._tracker_id then
            return false, "No active tracker"
        end

        map:disable_tracking()
        if map_octree._active_tracker_map == map then
            map_octree._active_tracker_map = nil
            map_octree._active_tracker_key = nil
        end
        return true, "Tracking disabled"
    end
})



register("octree_track_list", {
    description = S("List snapshots with tracking enabled"),
    privs = {server = true},
    func = function(p_name)
        local keys = map_octree.list_loaded_maps()
        local lines = {}
        for _, key in ipairs(keys) do
            local map = map_octree.get_loaded_map(key)
            if map and map._tracker_id then
                local status = map:get_tracking_status()
                if status then
                    lines[#lines + 1] = string.format("%s (pending=%d, dirty=%d)", key, status.pending, status.dirty)
                end
            end
        end

        if #lines == 0 then
            return true, "No tracked maps"
        end

        return true, "Tracked maps:\n" .. table.concat(lines, "\n")
    end
})



for name, def in pairs(defs) do
    core.register_chatcommand(name, def)
end



-- Get player positions with fallback to WorldEdit if available
function get_player_positions(player_name)
    local markers = map_octree.markers[player_name]
    if markers and markers.pos1 and markers.pos2 then
        return markers.pos1, markers.pos2
    end

    local we = worldedit
    if we and we.pos1 and we.pos2 then
        local we_pos1 = we.pos1[player_name]
        local we_pos2 = we.pos2[player_name]
        if we_pos1 and we_pos2 then
            return we_pos1, we_pos2
        end
    end

    return nil, nil
end



function parse_pos_param(param)
    param = tostring(param or "")
    local x, y, z = param:match("^%s*(-?%d+)%s*[, ]%s*(-?%d+)%s*[, ]%s*(-?%d+)%s*$")
    if x and y and z then
        return vector.new(tonumber(x), tonumber(y), tonumber(z))
    end
    return nil
end



function parse_load_flags(param)
    param = tostring(param or "")
    local subdir = param:match("%-subdir=([^%s]+)")
    local key = param:match("%-key=([^%s]+)")
    local no_current = param:match("%-no%-current") ~= nil
    local cleaned = param
    cleaned = cleaned:gsub("%-subdir=%S+", "")
    cleaned = cleaned:gsub("%-key=%S+", "")
    cleaned = cleaned:gsub("%-no%-current", "")
    cleaned = cleaned:match("^%s*(.-)%s*$")
    return cleaned, {
        subdir = subdir,
        key = key,
        no_current = no_current,
    }
end



function build_map_key_for_command(file_name, subdir)
    file_name = tostring(file_name or "")
    file_name = file_name:match("^%s*(.-)%s*$")
    if subdir and subdir ~= "" then
        return subdir .. "/" .. file_name
    end
    return file_name
end
