local clamp
local choose_batch_chunk_sizes

--#region types
---@class BatchPlanEntry
---@field x_idx integer
---@field y_idx integer
---@field z_idx integer
---@field x_end integer
---@field y_end integer
---@field z_end integer
---@field minp vector
---@field maxp vector
---@field padded_min vector
---@field padded_max vector
---@field volume integer

---@class BatchPlan
---@field minp vector
---@field maxp vector
---@field size vector
---@field max_voxelmanip_volume integer
---@field batches BatchPlanEntry[]
--#endregion


---Pure planning: returns batches (index ranges + world bounds) without touching VoxelManip.
---Useful for tests to validate partitioning and volume constraints.
---@param minp vector
---@param maxp vector
---@param opts? MapCreationOpts
---@return BatchPlan
function octmap.plan_batches(minp, maxp, opts)
	opts = opts or {}
	-- Cap by server budget assuming inflight=1 unless caller already applied limits
	-- (async save applies limits using async_inflight).
	if not opts._octmap_limits_applied then
		octmap.apply_server_limits(opts, 1)
	end
	local max_voxelmanip_volume = opts.max_voxelmanip_volume
	---@cast max_voxelmanip_volume integer

	minp, maxp = vector.sort(minp, maxp)
	minp, maxp = octchunk.snap_to_center(minp), octchunk.snap_to_center(maxp)

	local size_x = math.floor((maxp.x - minp.x) / octchunk.SIZE) + 1
	local size_y = math.floor((maxp.y - minp.y) / octchunk.SIZE) + 1
	local size_z = math.floor((maxp.z - minp.z) / octchunk.SIZE) + 1

	local padded_min = vector.subtract(minp, octchunk.SIZE / 2)
	local padded_max = vector.add(maxp, octchunk.SIZE / 2)
	local size = vector.subtract(padded_max, padded_min)
	local total_volume = (size.x + 1) * (size.y + 1) * (size.z + 1)

	local batches = {}

	if not opts.force_batches and total_volume <= max_voxelmanip_volume then
		batches[1] = {
			x_idx = 1,
			y_idx = 1,
			z_idx = 1,
			x_end = size_x,
			y_end = size_y,
			z_end = size_z,
			minp = minp,
			maxp = maxp,
			padded_min = padded_min,
			padded_max = padded_max,
			volume = total_volume,
		}
		return {
			minp = minp,
			maxp = maxp,
			size = vector.new(size_x, size_y, size_z),
			max_voxelmanip_volume = max_voxelmanip_volume,
			batches = batches,
		}
	end

	local batch_chunks_x, batch_chunks_y, batch_chunks_z
	if type(opts.choose_batch_chunk_sizes) == "function" then
		batch_chunks_x, batch_chunks_y, batch_chunks_z = opts.choose_batch_chunk_sizes(size_x, size_y, size_z, max_voxelmanip_volume)
	else
		batch_chunks_x, batch_chunks_y, batch_chunks_z = choose_batch_chunk_sizes(size_x, size_y, size_z, max_voxelmanip_volume)
	end

	for x_idx = 1, size_x, batch_chunks_x do
		local x_end = math.min(x_idx + batch_chunks_x - 1, size_x)
		for y_idx = 1, size_y, batch_chunks_y do
			local y_end = math.min(y_idx + batch_chunks_y - 1, size_y)
			for z_idx = 1, size_z, batch_chunks_z do
				local z_end = math.min(z_idx + batch_chunks_z - 1, size_z)

				local batch_minp = vector.new(
					minp.x + (x_idx - 1) * octchunk.SIZE,
					minp.y + (y_idx - 1) * octchunk.SIZE,
					minp.z + (z_idx - 1) * octchunk.SIZE
				)
				local batch_maxp = vector.new(
					minp.x + (x_end - 1) * octchunk.SIZE,
					minp.y + (y_end - 1) * octchunk.SIZE,
					minp.z + (z_end - 1) * octchunk.SIZE
				)
				batch_minp = octchunk.snap_to_center(batch_minp)
				batch_maxp = octchunk.snap_to_center(batch_maxp)

				local padded_bmin = vector.subtract(batch_minp, octchunk.SIZE / 2)
				local padded_bmax = vector.add(batch_maxp, octchunk.SIZE / 2)
				local bsize = vector.subtract(padded_bmax, padded_bmin)
				local volume = (bsize.x + 1) * (bsize.y + 1) * (bsize.z + 1)

				batches[#batches + 1] = {
					x_idx = x_idx,
					y_idx = y_idx,
					z_idx = z_idx,
					x_end = x_end,
					y_end = y_end,
					z_end = z_end,
					minp = batch_minp,
					maxp = batch_maxp,
					padded_min = padded_bmin,
					padded_max = padded_bmax,
					volume = volume,
				}
			end
		end
	end

	return {
		minp = minp,
		maxp = maxp,
		size = vector.new(size_x, size_y, size_z),
		max_voxelmanip_volume = max_voxelmanip_volume,
		batches = batches,
	}
end



---Clamp a value to the provided bounds.
---@param v number
---@param min_v number
---@param max_v number
---@return number
function clamp(v, min_v, max_v)
	if v < min_v then return min_v end
	if v > max_v then return max_v end
	return v
end



---Pick a 3D batch size in chunk units so a VoxelManip read stays within limits.
---Returns (bx, by, bz) in number-of-chunks.
---@param total_x integer
---@param total_y integer
---@param total_z integer
---@param max_voxelmanip_volume? integer
---@return integer bx
---@return integer by
---@return integer bz
function choose_batch_chunk_sizes(total_x, total_y, total_z, max_voxelmanip_volume)
	max_voxelmanip_volume = max_voxelmanip_volume or octmap.DEFAULT_MAX_VOXELMANIP_VOLUME
	local size_nodes = octchunk.SIZE + 1
	local max_chunks_per_batch = math.floor(max_voxelmanip_volume / (size_nodes * size_nodes * size_nodes))
	max_chunks_per_batch = math.max(max_chunks_per_batch, 1)

	local base = math.max(1, math.floor(max_chunks_per_batch ^ (1 / 3)))
	local bx = clamp(base, 1, total_x)
	local by = clamp(base, 1, total_y)
	local bz = clamp(base, 1, total_z)

	---Check if a batch size fits volume constraints.
	---@param nx integer
	---@param ny integer
	---@param nz integer
	---@return boolean
	local function product_fits(nx, ny, nz)
		local x_nodes = nx * octchunk.SIZE + 1
		local y_nodes = ny * octchunk.SIZE + 1
		local z_nodes = nz * octchunk.SIZE + 1
		return (x_nodes * y_nodes * z_nodes) <= max_voxelmanip_volume
	end

	while not product_fits(bx, by, bz) do
		if bx >= by and bx >= bz and bx > 1 then
			bx = bx - 1
		elseif by >= bx and by >= bz and by > 1 then
			by = by - 1
		elseif bz > 1 then
			bz = bz - 1
		else
			break
		end
	end

	local grew = true
	while grew do
		grew = false
		if bx < total_x and product_fits(bx + 1, by, bz) then
			bx = bx + 1
			grew = true
		end
		if by < total_y and product_fits(bx, by + 1, bz) then
			by = by + 1
			grew = true
		end
		if bz < total_z and product_fits(bx, by, bz + 1) then
			bz = bz + 1
			grew = true
		end
	end

	return bx, by, bz
end
