local DEFAULT_MAX_TOTAL_RAM_BUDGET_MB = 1500
local DEFAULT_INFLIGHT_WORKERS = 4
local DEFAULT_VM_ARRAYS = 1
-- Engine mapblock cache adds overhead, allocate 1/6 for Lua data arrays
-- VoxelManip data: ~8 bytes/node per array, divided by inflight workers
local DEFAULT_MAX_VOXELMANIP_VOLUME = math.floor((DEFAULT_MAX_TOTAL_RAM_BUDGET_MB / 6 * 1024 * 1024) / (8 * DEFAULT_VM_ARRAYS) / DEFAULT_INFLIGHT_WORKERS)
-- Cache limit: 5% of budget, more than enough for typical usage
local DEFAULT_CACHE_MB = math.floor(DEFAULT_MAX_TOTAL_RAM_BUDGET_MB * 0.05)

octmap.DEFAULT_CACHE_MB = DEFAULT_CACHE_MB
octmap.DEFAULT_MAX_VOXELMANIP_VOLUME = DEFAULT_MAX_VOXELMANIP_VOLUME

local SETTING_MAX_TOTAL_RAM_BUDGET_MB = "map_octree_max_total_ram_budget_mb"
local SETTING_MAX_VOXELMANIP_VOLUME = "map_octree_max_voxelmanip_volume"



---Get configured RAM budget in MB.
---@return integer
local function get_server_ram_budget_mb()
	local v = tonumber(core.settings:get(SETTING_MAX_TOTAL_RAM_BUDGET_MB) or "")
	if not v or v <= 0 then
		return DEFAULT_MAX_TOTAL_RAM_BUDGET_MB
	end
	if v < 128 then
		v = 128
	end
	return math.floor(v)
end


---Compute a GC memory limit for write-heavy operations (in KB).
---@return integer
function octmap.get_gc_limit_kb()
	local total_mb = get_server_ram_budget_mb()
	local limit_mb = math.max(128, math.floor(total_mb * 0.9))
	return limit_mb * 1024
end



---Compute cache cap in MB from RAM budget.
---@param total_ram_budget_mb number
---@return integer
local function compute_cache_mb_cap(total_ram_budget_mb)
	return math.max(1, math.floor(total_ram_budget_mb * 0.05))
end




---Compute max VoxelManip volume for the given budget.
---@param total_ram_budget_mb number
---@param inflight_workers? number
---@param array_count? number
---@return integer
local function compute_max_voxelmanip_volume(total_ram_budget_mb, inflight_workers, array_count)
	inflight_workers = math.max(1, math.floor(tonumber(inflight_workers) or DEFAULT_INFLIGHT_WORKERS))
	array_count = math.max(1, math.floor(tonumber(array_count) or DEFAULT_VM_ARRAYS))
	-- Engine mapblock cache adds ~5x overhead, allocate 1/6 for Lua data arrays.
	-- VoxelManip data: ~8 bytes/node per array, divided by inflight workers.
	return math.floor((total_ram_budget_mb / 6 * 1024 * 1024) / (8 * array_count) / inflight_workers)
end



---Apply server-owned RAM caps and fill safe defaults.
---@param opts? MapCreationOpts Options (mutated in place)
---@param inflight_workers? integer Expected async concurrency (default 1)
---@return MapCreationOpts opts The modified options
function octmap.apply_server_limits(opts, inflight_workers)
	opts = opts or {}
	local total_mb = get_server_ram_budget_mb()
	local max_cache_mb = compute_cache_mb_cap(total_mb)
	local array_count = tonumber(opts._octmap_vm_arrays) or DEFAULT_VM_ARRAYS
	local max_vm_volume = compute_max_voxelmanip_volume(total_mb, inflight_workers, array_count)
	local vm_override = tonumber(core.settings:get(SETTING_MAX_VOXELMANIP_VOLUME) or "")
	if vm_override and vm_override > 0 then
		max_vm_volume = math.min(max_vm_volume, math.floor(vm_override))
	end

	if opts.cache_mb == nil then
		opts.cache_mb = max_cache_mb
	else
		local cache_mb = tonumber(opts.cache_mb)
		if cache_mb then
			opts.cache_mb = math.min(max_cache_mb, math.max(1, math.floor(cache_mb)))
		else
			opts.cache_mb = max_cache_mb
		end
	end

	if opts.max_voxelmanip_volume == nil then
		opts.max_voxelmanip_volume = max_vm_volume
	else
		local vm_volume = tonumber(opts.max_voxelmanip_volume)
		if vm_volume then
			opts.max_voxelmanip_volume = math.min(max_vm_volume, math.max(1, math.floor(vm_volume)))
		else
			opts.max_voxelmanip_volume = max_vm_volume
		end
	end

	opts._octmap_limits_applied = true

	return opts
end


---Apply write-focused limits and defaults (mutates opts).
---@param opts? MapCreationOpts
---@param inflight_workers? integer
---@param array_count? integer
---@return MapCreationOpts
function octmap.apply_write_limits(opts, inflight_workers, array_count)
	opts = opts or {}
	if opts._octmap_vm_arrays == nil then
		opts._octmap_vm_arrays = array_count or DEFAULT_VM_ARRAYS
	end
	if opts._octmap_gc_limit_kb == nil then
		opts._octmap_gc_limit_kb = octmap.get_gc_limit_kb()
	end
	return octmap.apply_server_limits(opts, inflight_workers)
end
