--- Map storage and placement operations.
-- Handles saving map snapshots to disk, loading them, and placing them back into the world.
-- Supports both synchronous and async operations with automatic RAM budgeting.

local world_dir = core.get_worldpath() .. "/map_octree"

local build_batch_job
local acquire_unload_timeout_guard
local release_unload_timeout_guard

local unload_timeout_guard = {
    count = 0,
    original = nil,
    desired = nil,
    storage_before = nil,
    storage_had = false,
    storage_guard_value = nil,
}

local get_requested_bounds = map_octree.get_requested_bounds

local SETTING_READ_BUDGET_MS = "map_octree_read_budget_ms"
local DEFAULT_READ_BUDGET_MS = 50
local SETTING_REDUCE_UNLOAD_TIMEOUT = "map_octree_reduce_unload_timeout"

--#region types
---@alias LoadMapOpts {subdir?: string, key?: string, set_current?: boolean}

---@class StorageSaveOpts : MapCreationOpts
---@field file_name? string
---@field subdir? string
---@field async_inflight? integer
---@field flush_async_workers? boolean
---@field read_budget_ms? number

---@class StorageSaveInfo
---@field batch_count integer

---@class MapPlacementOpts : MapCreationOpts
---@field min_delay? number

---@alias PlaceAsyncOpts MapPlacementOpts|fun(ok: boolean, err?: string)

--#endregion


---Ensure the world directory exists.
---@param dir string|nil
local function ensure_world_dir(dir)
    -- mkdir() may return false if it already exists; we don't treat that as fatal.
    core.mkdir(dir or world_dir)
end



---Normalize and validate a subdir.
---@param subdir string|nil
---@return string
local function normalize_subdir(subdir)
    subdir = tostring(subdir or "")
    subdir = subdir:match("^%s*(.-)%s*$")
    assert(subdir ~= "", "subdir required")
    assert(not subdir:find("[/\\]"), "subdir must not contain path separators")
    return subdir
end



---Resolve storage directory for a subdir.
---@param subdir string|nil
---@return string
function map_octree.get_storage_dir(subdir)
    if not subdir or subdir == "" then
        ensure_world_dir(world_dir)
        return world_dir
    end
    local clean = normalize_subdir(subdir)
    local dir = world_dir .. "/" .. clean
    ensure_world_dir(dir)
    return dir
end



---Normalize and validate a file name.
---@param file_name string|nil
---@return string
local function normalize_file_name(file_name)
    file_name = tostring(file_name or "")
    file_name = file_name:match("^%s*(.-)%s*$")
    assert(file_name ~= "", "file_name required")
    assert(not file_name:find("[/\\]"), "file_name must not contain path separators")
    if not file_name:match("%.bin$") then
        file_name = file_name .. ".bin"
    end
    return file_name
end

---Get full map file path.
---@param file_name string|nil
---@param subdir string|nil
---@return string
local function get_mapfile_path(file_name, subdir)
    local dir = map_octree.get_storage_dir(subdir)
    return dir .. "/" .. normalize_file_name(file_name)
end



---Build a stable registry key for a stored map.
---@param file_name string|nil
---@param subdir string|nil
---@return string
local function build_map_key(file_name, subdir)
    local base = normalize_file_name(file_name):gsub("%.bin$", "")
    if subdir and subdir ~= "" then
        local clean = normalize_subdir(subdir)
        return clean .. "/" .. base
    end
    return base
end



-- Stored map reference for queries (global, last-writer-wins)
map_octree.current_map = nil
map_octree.loaded_maps = map_octree.loaded_maps or {}


---Save a map region to disk.
---@param pos1 vector
---@param pos2 vector
---@param opts? StorageSaveOpts Options including file_name, store_chunk_blobs, cache_mb, etc.
---@return OctMap|nil map The created map object
---@return string|nil err Error message on failure
function map_octree.save_to_file(pos1, pos2, opts)
    assert(pos1 and pos2, "pos1 and pos2 required")
    opts = opts or {}
    local mapfile = get_mapfile_path(opts.file_name, opts.subdir)
    if opts.store_chunk_blobs == nil then
        opts.store_chunk_blobs = true
    end
    octmap.apply_server_limits(opts, 1)

    local req_pos1, req_pos2 = vector.sort(pos1, pos2)
    req_pos1 = assert(vector.round(req_pos1))
    req_pos2 = assert(vector.round(req_pos2))
    local map = octmap.new(pos1, pos2, opts)
    map.requested_pos1 = req_pos1
    map.requested_pos2 = req_pos2
    local t0 = core.get_us_time()
    local serialized, err = octmap.serialize(map)
    if not serialized then
        return nil, err
    end
    local t_serialize = core.get_us_time() - t0

    t0 = core.get_us_time()
    core.safe_file_write(mapfile, serialized)
    local t_write = core.get_us_time() - t0
    map_octree.current_map = map
    map_octree.debug(string.format("Serialize: %.1fms", t_serialize / 1000))
    map_octree.debug(string.format("Write: %.1fms", t_write / 1000))
    core.log("action", "[map_octree] Saved area to " .. mapfile)
    return map, nil
end



-- Async build + save. Calls `callback(ok, map_or_err, info)` when finished.
-- map_or_err is either the built map (ok=true) or an error string (ok=false).
---Async save a map region to disk.
---@param pos1 vector
---@param pos2 vector
---@param opts? StorageSaveOpts
---@param callback fun(ok: boolean, map_or_err: OctMap|string|nil, info?: StorageSaveInfo)
---@return BatchPlan|nil plan Batch plan
function map_octree.save_to_file_async(pos1, pos2, opts, callback)
    assert(pos1 and pos2, "pos1 and pos2 required")
    assert(type(callback) == "function", "callback required")

    opts = opts or {}
    local mapfile = get_mapfile_path(opts.file_name, opts.subdir)
    if opts.store_chunk_blobs == nil then
        opts.store_chunk_blobs = true
    end
    local inflight_limit = tonumber(opts.async_inflight)
    if inflight_limit == nil then
        -- Default: use engine's async worker capacity (num_cpus - 2)
        ---@diagnostic disable-next-line: undefined-field
        inflight_limit = core.get_async_threading_capacity() or 4
        -- Check for autotune override (if set and non-zero, use it instead)
        local override = tonumber(core.settings:get("map_octree_async_inflight_override")) or 0
        if override > 0 then
            inflight_limit = override
        end
    end
    inflight_limit = math.max(1, math.floor(inflight_limit))
    octmap.apply_server_limits(opts, inflight_limit)
    local flush_workers = opts.flush_async_workers
    if flush_workers == nil then
        flush_workers = true
    end

    ---Flush async worker heaps.
    ---@param count number
    local function flush_async_workers(count)
        if not flush_workers then
            return
        end
        count = math.max(1, math.floor(tonumber(count) or inflight_limit))
        for _ = 1, count do
            core.handle_async(
                function()
                    if map_octree_async and type(map_octree_async.flush) == "function" then
                        return map_octree_async.flush()
                    end
                    collectgarbage("collect")
                    return true
                end,
                function()
                    -- ignore result
                end
            )
        end
    end

    ---Safely invoke async callback.
    ---@param ... any
    local function safe_callback(...)
        local args = {...}
        local ok, err = xpcall(function()
            callback(unpack(args))
        end, debug.traceback)
        if not ok then
            core.log("error", "[map_octree] save_to_file_async callback error: " .. tostring(err))
        end
    end

    local map = {}
    pos1, pos2 = vector.sort(pos1, pos2)
    local req_min = assert(vector.round(pos1))
    local req_max = assert(vector.round(pos2))
    local minp = octchunk.snap_to_center(pos1)
    local maxp = octchunk.snap_to_center(pos2)
    map.minp = minp
    map.maxp = maxp
    map.requested_pos1 = req_min
    map.requested_pos2 = req_max
    map.trees = matrix3d.new(
        math.floor((maxp.x - minp.x) / octchunk.SIZE) + 1,
        math.floor((maxp.y - minp.y) / octchunk.SIZE) + 1,
        math.floor((maxp.z - minp.z) / octchunk.SIZE) + 1,
        nil -- sparse default
    )
    map.default_node = nil
    map.content_id_map = {} -- global content_id->name map

    local plan = octmap.plan_batches(minp, maxp, opts)
    local reduce_timeout = core.settings:get_bool(SETTING_REDUCE_UNLOAD_TIMEOUT, true) == true
    local use_unload_guard = reduce_timeout and (#plan.batches > 1)

    -- Temporarily reduce server mapblock unload timeout to free memory faster.
    -- This is a global setting, so we guard it with a ref-count to support concurrent jobs.
    if use_unload_guard then
        acquire_unload_timeout_guard()
    end
    local released = false
    ---Release unload timeout guard only once.
    local function release_once()
        if released then
            return
        end
        released = true
        if use_unload_guard then
            release_unload_timeout_guard()
        end
    end
    local next_batch = 1
    local inflight = 0
    local failed = false
    local on_done_err

    -- Collect uniform tree info from workers for zero-deserialize sparsify
    local uniform_counts = {}
    local uniform_positions = {}

    ---Finalize async save with success.
    local function on_done_ok()
        release_once()

        -- Use pre-collected uniform positions for zero-deserialize sparsify
        local t0 = core.get_us_time()
        octmap.sparsify_with_positions(map, uniform_counts, uniform_positions)
        octmap.attach_methods(map)
        local t_sparsify = core.get_us_time() - t0

        t0 = core.get_us_time()
        local serialized, err = octmap.serialize(map)
        if not serialized then
            on_done_err(err or "serialize failed")
            return
        end
        local t_serialize = core.get_us_time() - t0

        t0 = core.get_us_time()
        core.safe_file_write(mapfile, serialized)
        local t_write = core.get_us_time() - t0

        map_octree.debug(string.format("Sparsify: %.1fms", t_sparsify / 1000))
        map_octree.debug(string.format("Serialize: %.1fms", t_serialize / 1000))
        map_octree.debug(string.format("Write: %.1fms", t_write / 1000))

        collectgarbage("collect")
        map_octree.current_map = map
        collectgarbage("collect")
        flush_async_workers(inflight_limit)
        safe_callback(true, map, {
            batch_count = #plan.batches,
        })
    end

    ---Finalize async save with error.
    ---@param err string
    function on_done_err(err)
        release_once()
        flush_async_workers(inflight_limit)
        safe_callback(false, err)
    end



    local read_budget_ms = opts.read_budget_ms
    if read_budget_ms == nil then
        local v = tonumber(core.settings:get(SETTING_READ_BUDGET_MS) or "")
        read_budget_ms = (v and v > 0) and v or DEFAULT_READ_BUDGET_MS
    end

    ---Handle async batch completion.
    ---@param ok boolean
    ---@param payload any
    ---@param x_idx integer
    ---@param y_idx integer
    ---@param z_idx integer
    ---@param trees_x integer
    ---@param trees_y integer
    ---@param trees_z integer
    local function on_batch_done(ok, payload, x_idx, y_idx, z_idx, trees_x, trees_y, trees_z)
        local ok_cb, err_cb = xpcall(function()
            inflight = inflight - 1
            if failed then return end
            if not ok then
                failed = true
                on_done_err("async batch failed: " .. tostring(payload))
                return
            end
            local entries = payload
            local idx = 1
            for bx = 0, trees_x - 1 do
                for by = 0, trees_y - 1 do
                    for bz = 0, trees_z - 1 do
                        local entry = entries[idx]
                        local blob = entry[1]
                        local uniform_cid = entry[2]
                        local content_ids = entry[3]
                        local gx, gy, gz = x_idx + bx, y_idx + by, z_idx + bz
                        matrix3d.set(map.trees, gx, gy, gz, blob)

                        if uniform_cid then
                            uniform_counts[uniform_cid] = (uniform_counts[uniform_cid] or 0) + 1
                            uniform_positions[#uniform_positions + 1] = {gx, gy, gz, uniform_cid}
                        end

                        if content_ids then
                            for cid, node_name in pairs(content_ids) do
                                if not map.content_id_map[cid] then
                                    map.content_id_map[cid] = node_name
                                end
                            end
                        end

                        idx = idx + 1
                    end
                end
            end
            collectgarbage("collect")
        end, debug.traceback)
        if not ok_cb then
            failed = true
            on_done_err("async callback error: " .. tostring(err_cb))
        end
    end

    ---Schedule one batch for async processing.
    ---@param b BatchPlanEntry
    local function kick_one_batch(b)
        core.load_area(b.padded_min, b.padded_max)
        local manip = core.get_voxel_manip()
        local emerged_pos1, emerged_pos2 = manip:read_from_map(b.padded_min, b.padded_max)

        local trees_x = b.x_end - b.x_idx + 1
        local trees_y = b.y_end - b.y_idx + 1
        local trees_z = b.z_end - b.z_idx + 1

        inflight = inflight + 1
        core.handle_async(
            build_batch_job,
            on_batch_done,
            manip, emerged_pos1, emerged_pos2, b.minp, trees_x, trees_y, trees_z, b.x_idx, b.y_idx, b.z_idx
        )
    end

    local schedule_more
    ---Schedule more batches within the read budget.
    ---@type fun()
    schedule_more = function()
        if failed then return end
        if next_batch > #plan.batches and inflight == 0 then
            on_done_ok()
            return
        end

        -- Budget-based loop: kick multiple batches in this tick until budget exhausted
        local tick_start = core.get_us_time()
        local budget_us = read_budget_ms * 1000
        while inflight < inflight_limit and next_batch <= #plan.batches do
            local b = plan.batches[next_batch]
            if not b then break end
            next_batch = next_batch + 1
            kick_one_batch(b)

            local elapsed = core.get_us_time() - tick_start
            if elapsed >= budget_us then break end
        end

        -- Schedule next tick if there's more work
        if next_batch <= #plan.batches or inflight > 0 then
            core.after(0.1, schedule_more)
        end
    end

    local ok_start, err_start = xpcall(schedule_more, debug.traceback)
    if not ok_start then
        failed = true
        on_done_err("schedule_more error: " .. tostring(err_start))
    end
    return plan
end



function build_batch_job(vm, e1, e2, batch_minp, tx, ty, tz, x_idx, y_idx, z_idx)
    local ok, payload = map_octree_async.build_batch_blobs(vm, e1, e2, batch_minp, tx, ty, tz)
    return ok, payload, x_idx, y_idx, z_idx, tx, ty, tz
end



---Acquire the unload timeout guard (reference counted).
---@param desired_value string|number|nil
function acquire_unload_timeout_guard(desired_value)
    desired_value = tostring(desired_value or "1")
    if unload_timeout_guard.count == 0 then
        unload_timeout_guard.original = core.settings:get("server_unload_unused_data_timeout") or "29"
        local stored = map_octree.storage_get_last_timeout()
        unload_timeout_guard.storage_before = stored
        unload_timeout_guard.storage_had = stored ~= ""
        if stored == "" then
            map_octree.storage_set_last_timeout(unload_timeout_guard.original)
            unload_timeout_guard.storage_guard_value = unload_timeout_guard.original
        else
            unload_timeout_guard.storage_guard_value = nil
        end
        unload_timeout_guard.desired = desired_value
        core.settings:set("server_unload_unused_data_timeout", desired_value)
        core.log("action", "[map_octree] Temporarily set server_unload_unused_data_timeout=" .. desired_value)
    else
        if unload_timeout_guard.original == nil or unload_timeout_guard.original == unload_timeout_guard.desired then
            local saved = map_octree.storage_get_last_timeout()
            if saved ~= "" then
                unload_timeout_guard.original = saved
            end
        end
        local cur = tonumber(unload_timeout_guard.desired)
        local nxt = tonumber(desired_value)
        if cur and nxt and nxt < cur then
            unload_timeout_guard.desired = desired_value
            core.settings:set("server_unload_unused_data_timeout", desired_value)
        end
    end
    unload_timeout_guard.count = unload_timeout_guard.count + 1
end



---Release the unload timeout guard.
function release_unload_timeout_guard()
    if unload_timeout_guard.count <= 0 then
        return
    end
    unload_timeout_guard.count = unload_timeout_guard.count - 1
    if unload_timeout_guard.count == 0 then
        local saved = map_octree.storage_get_last_timeout()
        local restore_value = unload_timeout_guard.original or (saved ~= "" and saved) or "29"
        core.settings:set("server_unload_unused_data_timeout", restore_value)
        core.log("action", "[map_octree] Restored server_unload_unused_data_timeout=" .. tostring(restore_value))
        if unload_timeout_guard.storage_had then
            map_octree.storage_set_last_timeout(unload_timeout_guard.storage_before)
        else
            local cur = map_octree.storage_get_last_timeout()
            if cur == tostring(unload_timeout_guard.storage_guard_value or "") then
                map_octree.storage_clear_last_timeout()
            end
        end
        unload_timeout_guard.original = nil
        unload_timeout_guard.desired = nil
        unload_timeout_guard.storage_before = nil
        unload_timeout_guard.storage_had = false
        unload_timeout_guard.storage_guard_value = nil
    end
end




---Save a cubic region around a player.
---@param player ObjectRef
---@param radius? number Default 100
---@param height? number Default 128
---@param opts? StorageSaveOpts
---@return OctMap|nil map The created map object
---@return string|nil err Error message on failure
function map_octree.save_around_player(player, radius, height, opts)
    radius = radius or 100
    height = height or 128
    opts = opts or {}
    local pos = player:get_pos()
    local pos1 = vector.new(pos.x - radius, pos.y - height, pos.z - radius)
    local pos2 = vector.new(pos.x + radius, pos.y + height, pos.z + radius)
    return map_octree.save_to_file(pos1, pos2, opts)
end




---Load a map from disk.
---@param file_name string
---@param opts? LoadMapOpts|string
---@return OctMap|nil
---@return string|nil
function map_octree.load_map(file_name, opts)
    if type(opts) == "string" then
        opts = {subdir = opts}
    end
    ---@cast opts LoadMapOpts|nil
    opts = opts or {}
    local mapfile = get_mapfile_path(file_name, opts.subdir)
    local map, err = octmap.read_from_file(mapfile)
    if map then
        local key = opts.key or build_map_key(file_name, opts.subdir)
        map_octree.loaded_maps[key] = map
        if opts.set_current ~= false then
            map_octree.current_map = map
        end
        core.log("action", "[map_octree] Loaded map from " .. mapfile)
        return map, nil
    end
    return nil, err
end



---Get a loaded map from the in-memory registry.
---@param key string
---@return OctMap|nil
function map_octree.get_loaded_map(key)
    return map_octree.loaded_maps[key]
end



---Return a sorted list of loaded map keys.
---@return string[]
function map_octree.list_loaded_maps()
    local keys = {}
    for key in pairs(map_octree.loaded_maps) do
        keys[#keys + 1] = key
    end
    table.sort(keys)
    return keys
end



---Unload a map from the in-memory registry.
---@param key string
---@return boolean
---@return string|nil
function map_octree.unload_map(key)
    local map = map_octree.loaded_maps[key]
    if not map then
        return false, "map not loaded"
    end
    local status = map:get_tracking_status()
    if status then
        map:disable_tracking({flush = true})
    end
    map_octree.loaded_maps[key] = nil
    if map_octree.current_map == map then
        map_octree.current_map = nil
    end
    if map_octree._active_tracker_map == map then
        map_octree._active_tracker_map = nil
    end
    return true
end



---Get node from currently loaded map.
---@param x number
---@param y number
---@param z number
---@return string|nil name, integer param2
function map_octree.get_node_at(x, y, z)
    return map_octree.current_map:get_node_at(x, y, z)
end



---Return world limit for placement.
---@return integer
local function place_world_limit()
    return 31000
end

---Clamp placement position to world bounds.
---@param pos1 vector
---@param size vector
---@return vector|nil
---@return string|nil
local function clamp_place_pos1(pos1, size)
    local limit = place_world_limit()
    local max_min_x = limit - (size.x - 1)
    local max_min_y = limit - (size.y - 1)
    local max_min_z = limit - (size.z - 1)
    if max_min_x < -limit or max_min_y < -limit or max_min_z < -limit then
        return nil, "snapshot too large to fit inside world bounds"
    end
    ---Clamp a value between bounds.
    ---@param v number
    ---@param lo number
    ---@param hi number
    ---@return number
    local function clamp(v, lo, hi)
        if v < lo then return lo end
        if v > hi then return hi end
        return v
    end
    return vector.new(
        clamp(pos1.x, -limit, max_min_x),
        clamp(pos1.y, -limit, max_min_y),
        clamp(pos1.z, -limit, max_min_z)
    )
end

---Write a single placement batch into the world (clipped to requested bounds).
---@param map OctMap
---@param b BatchPlanEntry
---@param req_min vector
---@param req_max vector
local function place_write_batch(map, b, req_min, req_max)
    local half = octchunk.SIZE / 2
    local batch_min = vector.subtract(b.minp, {x = half, y = half, z = half})
    local batch_max = vector.add(b.maxp, {x = half - 1, y = half - 1, z = half - 1})
    local src_req_min, _ = get_requested_bounds(map)
    local delta = vector.subtract(req_min, src_req_min)

    local clip_min = {
        x = math.max(batch_min.x, req_min.x),
        y = math.max(batch_min.y, req_min.y),
        z = math.max(batch_min.z, req_min.z),
    }
    local clip_max = {
        x = math.min(batch_max.x, req_max.x),
        y = math.min(batch_max.y, req_max.y),
        z = math.min(batch_max.z, req_max.z),
    }
    if clip_min.x > clip_max.x or clip_min.y > clip_max.y or clip_min.z > clip_max.z then
        return
    end

    core.load_area(batch_min, batch_max)
    local manip = core.get_voxel_manip()
    local e1, e2 = manip:read_from_map(batch_min, batch_max)
    local area = VoxelArea(e1, e2)

    local default_cid = core.get_content_id(map.default_node or "air")
    local data = manip:get_data()
    local param2_data = manip:get_param2_data()
    ---@cast data integer[]
    ---@cast param2_data integer[]

    local size = octchunk.SIZE
    for gx = b.x_idx, b.x_end do
        for gy = b.y_idx, b.y_end do
            for gz = b.z_idx, b.z_end do
                local cell = matrix3d.get(map.trees, gx, gy, gz)
                local src_chunk_min, src_chunk_max = map_octree.chunk_world_bounds(map.minp, gx, gy, gz)
                local chunk_min = vector.add(src_chunk_min, delta)
                local chunk_max = vector.add(src_chunk_max, delta)
                local req_min = {
                    x = math.max(chunk_min.x, clip_min.x),
                    y = math.max(chunk_min.y, clip_min.y),
                    z = math.max(chunk_min.z, clip_min.z),
                }
                local req_max = {
                    x = math.min(chunk_max.x, clip_max.x),
                    y = math.min(chunk_max.y, clip_max.y),
                    z = math.min(chunk_max.z, clip_max.z),
                }
                if req_min.x > req_max.x or req_min.y > req_max.y or req_min.z > req_max.z then
                    goto continue_chunk
                end
                if cell ~= nil then
                    local tree = assert(octmap._get_tree_cached(map, gx, gy, gz))
                    ---@cast tree Octchunk
                    local center = vector.new(
                        b.minp.x + (gx - b.x_idx) * size,
                        b.minp.y + (gy - b.y_idx) * size,
                        b.minp.z + (gz - b.z_idx) * size
                    )
                    octchunk.fill_voxel_area(tree, req_min, req_max, area, data, param2_data, nil, center, size)
                else
                    for z = req_min.z, req_max.z do
                        for y = req_min.y, req_max.y do
                            for x = req_min.x, req_max.x do
                                local idx = area:index(x, y, z)
                                data[idx] = default_cid
                                param2_data[idx] = 0
                            end
                        end
                    end
                end
                ::continue_chunk::
            end
        end
    end

    manip:set_data(data)
    manip:set_param2_data(param2_data)
    -- Recalculate lighting so placed snapshots don't end up dark.
    manip:write_to_map(true)
end


---Collect garbage only when above the configured limit.
---@param opts MapCreationOpts
local function maybe_collect_gc(opts)
    local limit_kb = tonumber(opts._octmap_gc_limit_kb) or 0
    if limit_kb <= 0 then
        return
    end
    if collectgarbage("count") >= limit_kb then
        collectgarbage("collect")
    end
end


---Place a snapshot into the live map.
---pos1 is the target emerged min corner; when omitted, places back to the original position.
---@param map OctMap|vector|nil Map object or target position (uses current_map if vector)
---@param pos1? vector Target position (if map provided)
---@param opts? MapPlacementOpts
---@return boolean success
function map_octree.place(map, pos1, opts)
    if map == nil or (map.x and map.y and map.z) then
        opts = pos1
        pos1 = map
        map = map_octree.current_map
    end
    assert(map, "no map provided/loaded")
    opts = opts or {}
    octmap.apply_write_limits(opts, 1, 3)

    local src_min, _ = map:get_emerged_area()
    local target_pos1 = pos1 or src_min
    local size = map:size()
    ---@cast target_pos1 vector
    local clamped, err = clamp_place_pos1(target_pos1, size)
    assert(clamped, err)
    target_pos1 = clamped

    local delta = vector.subtract(target_pos1, src_min)
    local target_minp = vector.add(map.minp, delta)
    local target_maxp = vector.add(map.maxp, delta)
    local src_req_min, src_req_max = get_requested_bounds(map)
    local target_req_min = vector.add(src_req_min, delta)
    local target_req_max = vector.add(src_req_max, delta)

    local plan = octmap.plan_batches(target_minp, target_maxp, opts)
    for i = 1, #plan.batches do
        place_write_batch(map, plan.batches[i], target_req_min, target_req_max)
        maybe_collect_gc(opts)
    end

    map.minp = target_minp
    map.maxp = target_maxp
    map.requested_pos1 = target_req_min
    map.requested_pos2 = target_req_max
    return true
end



---Async placement (step-based): writes one batch per step and yields via core.after().
---@param map OctMap|vector|nil Map object or target position
---@param pos1? vector Target position or callback
---@param opts? PlaceAsyncOpts
---@param callback? fun(ok: boolean, err?: string)
function map_octree.place_async(map, pos1, opts, callback)
    if type(opts) == "function" and callback == nil then
        ---@cast opts fun(ok: boolean, err?: string)
        callback, opts = opts, nil
    end
    if map == nil or (map.x and map.y and map.z) then
        ---@cast opts fun(ok: boolean, err?: string)
        callback = opts
        opts = pos1
        pos1 = map
        map = map_octree.current_map
    end
    assert(type(callback) == "function", "callback required")
    assert(map, "no map provided/loaded")
    ---@cast opts MapPlacementOpts|nil
    opts = opts or {}
    octmap.apply_write_limits(opts, 1, 3)

    local src_min, _ = map:get_emerged_area()
    local target_pos1 = pos1 or src_min
    local size = map:size()
    ---@cast target_pos1 vector
    local clamped, err = clamp_place_pos1(target_pos1, size)
    if not clamped then
        callback(false, err)
        return
    end
    target_pos1 = clamped

    local delta = vector.subtract(target_pos1, src_min)
    local target_minp = vector.add(map.minp, delta)
    local target_maxp = vector.add(map.maxp, delta)
    local src_req_min, src_req_max = get_requested_bounds(map)
    local target_req_min = vector.add(src_req_min, delta)
    local target_req_max = vector.add(src_req_max, delta)

    local plan = octmap.plan_batches(target_minp, target_maxp, opts)
    local batches = plan.batches
    local idx = 1
    local min_delay = tonumber(opts.min_delay) or 0

    ---Safely invoke placement callback.
    ---@param ok boolean
    ---@param err2? string
    local function safe_callback(ok, err2)
        local ok_cb, e = xpcall(function()
            callback(ok, err2)
        end, debug.traceback)
        if not ok_cb then
            core.log("error", "[map_octree] place_async callback error: " .. tostring(e))
        end
    end

    ---Process the next placement batch.
    local function step()
        if idx > #batches then
            map.minp = target_minp
            map.maxp = target_maxp
            map.requested_pos1 = target_req_min
            map.requested_pos2 = target_req_max
            safe_callback(true)
            return
        end
        local b = batches[idx]
        idx = idx + 1
        local ok_step, err_step = xpcall(function()
            place_write_batch(map, b, target_req_min, target_req_max)
        end, debug.traceback)
        if not ok_step then
            safe_callback(false, tostring(err_step))
            return
        end
        maybe_collect_gc(opts)
        core.after(min_delay, step)
    end

    step()
end
