# skills

A Minetest skills library.

<a name="registering"></a>

## Registering a skill
To register a skill you have to call `skills.register_skill(internal_name, def)`, where `internal_name` is the name you'll use to refer to the skill in the code and it should be formatted like this: "unique_prefix:skill_name" (the unique prefix can be the mod's name for example), while `def` is the definition table that can contain the following properties:
- **name** (string): the skill's name;
- **description** (string): the skill's description;
- **cooldown** (number): in seconds. It's the minimum amount of time to wait in order to cast the skill again;
- **cast(args)**: it contains the skill's logic. It'll return false if the player is offline or the cooldown is not finished;
- **loop_params** (table): when this is declared the skill will be looped: to cast a looped skill you need to use the `start(args)` function instead of `cast(args)`; what `start(args)` does is simply casting the function every `cast_rate` seconds (if cast_rate is declared, otherwise the skill's logic will never be executed; this is not affected by the `cooldown`);
    - **cast_rate** (number): in seconds. The rate at which the skill will be casted. Assigning 0 will loop it as fast as it can;
    - **duration** (number): in seconds. The amount of time after which the skill will stop;
- **passive** (boolean): false by default. If true the skill will start (calling the `on_start` callback) automatically once the player has unlocked it. It can be stopped calling `disable()` (this will call the `on_stop` callback) and restarted calling `enable()`;
- **sounds** (table; every sound is declared this way: `{name = "sound_name"[, to_player = true, object = true, other sound parameters/SoundSpec properties...]}`. If `to_player` or `object` are set to true their value will become the player's name):
    - **cast** (table): a sound that will be reproduced every time  the `cast(args)` function gets called;
    - **start** (table): a sound that is reproduced when the skill starts;
    - **stop** (table): a sound that is reproduced when the    skill stops;
    - **bgm** (table): a looped sound that is reproduced while the skill is being used;
- **hud** (`{{name = "hud_name", hud definition...}, ...}`): a list of hud elements that appear while the skill is being used. They are stored in the `data._hud` table (`{"hud_name" = hud_id}`);
- **attachments** (table):
    - **particles** (`{ParticleSpawner1, ...}`): a list of particle spawners that are created when the skill starts and destroyed when it stops. They're stored in the `data._particles` table and are always attached to the player - it's useful if you want to have something like a particles trail;
    - **entities** (`{{pos = {...}, name = "Entity1" [, bone = "bone", rotation = {...}, forced_visible = false]}, ...}`): a list of entities that are attacched to the player as long as the skill is being used. The staticdata passed to the entity's `on_activate` callback is the player name and it's automatically stored in the entity's `pl_name` property;
- **sky** (table): if declared, while the skill is being casted, the sky will be set according to [this parameters](https://github.com/minetest/minetest/blob/stable-5/doc/lua_api.txt#L7134);
- **clouds** (table): if declared, while the skill is being casted, the clouds will be set according to [this parameters](https://github.com/minetest/minetest/blob/stable-5/doc/lua_api.txt#L7248);
- **chat_warnings** (table):
    - **cooldown** (boolean): true by default. Whether to send a chat message warning the player that they have to wait `cooldown_timer` time before using the skill again;
    - **disabled** (boolean) true by default. Whether to send a chat message warning the player that the skill is disabled;
- **on_start()**: this is called when `start` is called;
- **on_stop()**: this is called when `stop` is called;
- **data** (table): use this to define custom per players' properties (beware: their names should never start with a "_"). These are stored in a database and won't be reset when the server shuts down unless you change the type of one of them in the registration table;
- **... any other properties you may need**: you can also define your own properties, just make sure that they don't exist already and remember that this are shared by all players.

Here some examples of how to register a skill:
<details>
<summary>click to expand...</summary>

```lua
skills.register_skill("example_mod:counter", {
    name = "Counter",
    description = "Counts. You can use it every 2s.",
    sounds = {
        cast = {name = "ding", pitch = 2}
    },
    cooldown = 2,
    data = {
        counter = 0
    },
    cast = function(self)
        self.data.counter = self.data.counter + 1
        print(self.pl_name .. " is counting: " .. self.data.counter)
    end
})
```

```lua
skills.register_skill("example_mod:heal_over_time", {
    name = "Heal Over Time",
    description = "Restores a heart every 3 seconds for 30 seconds.",
    loop_params = {
        cast_rate = 3,
        duration = 30
    },
    sounds = {
        cast = {name = "heart_added"},
        bgm = {name = "angelic_music"}
    },
    cast = function(self)
        local player = self.player
        player:set_hp(player:get_hp() + 2)
    end
})
```

```lua
skills.register_skill("example_mod:boost_for_3s", {
    name = "Boost for 3 seconds",
    description = "Boosts the speed for 3 seconds.",
    loop_params = {
        duration = 3
    },
    sounds = {
        start = {name = "speed_up"},
        stop = {name = "speed_down"}
    },
    on_start = function(self)
        boost_speed(self.player)
    end,
    on_stop = function(self)
        reset_speed(self.player)
    end
})
```

```lua
skills.register_skill("example_mod:increase_speed", {
    name = "Increase Speed",
    description = "Increase your speed.",
    passive = true,
    data = {
        original_speed = {}
    },
    on_start = function(self)
        local player = self.player
        self.data.original_speed = player:get_physics_override().speed

        player:set_physics_override({speed = self.data.original_speed + 1})
    end,
    on_stop = function(self)
        self.player:set_physics_override({speed = self.data.original_speed})
    end
})
```
</details>

If you want to reregister a skill you have to call this function again.

## Assigning a skill
To unlock or remove a skill from a player just use `skills.unlock_skill/remove_skill(pl_name, skill_name)` function. You can also use the shorter form:
```lua
local pl_name = "giov4"
pl_name:unlock_skill("example_mod:counter")
pl_name:remove_skill("example_mod:counter")
```

## Using a skill
To use a player's skill you can use the short method or the long one: for the short one use `pl_name:cast_skill/start_skill/stop_skill("skill_name"[, args])` (if the player can't use it, because they didn't unlock it, these will return false); 

for the long one, you first have to get the player's skill table, using `skills.get_skill(pl_name, "skill_name")` or `pl_name:get_skill("skill_name")` (as before, if the player can't use it, it'll return false).

The function will return the player's skill table, composed of the [definition properties](#registering) + the following new properties:
- **disable()**: to disable the skill: when disabled the `cast` and `start` functions won't work;
- **enable()**: to enable the skill;
- **data._enabled** (boolean): true if the skill is enabled;
- **internal_name** (string): the name used to refer to the skill in the code; 
- **cooldown_timer** (number): the time left until the end of the cooldown;
- **is_active** (boolean): true if the skill is active;
- **pl_name**: the name of the player using this skill;
- **player**: the ObjectRef of the player using this skill.

Once you have it, just call `skill_table:cast([args])` or `skill_table:start([args])` to cast the skill. To stop it use `skill_table:stop()`.


## Utils function
- **skills.register_on_unlock(function(skill_table), [prefix])**: this is called everytime a player unlocks a skill having the specified prefix; if the prefix isn't specified the function will be called everytime a player unlocks a skill;
- **skills.disable/enable_skill(pl_name, skill_name) / pl_name:enable/disable_skill(skill_name)**: short methods to enable or disable a skill; 
- **skills.get_skill_def(skill_name)**: returns the skill's definition table;
- **skills.does_skill_exist(skill_name)**: to check if a skill exists (not case-sensitive);
- **skills.get_registered_skills([prefix])**: returns the registered skills; if a prefix is specified, only the skills having that prefix will be listed (`{"prefix1:skill1" = {def}}`); 
- **skills.get_unlocked_skills(pl_name, [prefix]) / pl_name:get_unlocked_skills(prefix)**: returns the unlocked skills; if a prefix is specified, only the skills having that prefix will be listed (`{"prefix1:skill1" = {def}}`); 
- **skills.has_skill(pl_name, skill_name) / pl_name:has_skill(skill_name)**: returns true if the player has the skill.
