--[[
    {
        "player": {
            {"mod:skill1" = {...}},
            {"mod:skill2" = {...}}
        }
    }
--]]
skills.player_skills = {}

-- { "mod:skill1" = {...}, ...}
skills.registered_skills = {}
local T = minetest.get_translator("skills")

local function update_db() end
local function initialize_def() end
local function init_players_skill_def() end
local function update_player_skill_def() end
local function init_subtable() end
local function cast_passive_skills() end
local function remove_unregistered_skills_from_db() end
local function play_sound() end
local function remove_userdata() end
local function print_remaining_cooldown_seconds() end
local function update_pl_skill_data() end
local function start_cooldown() end
local function attach_expiring_entity() end

local string_metatable = getmetatable("")
local storage = minetest.get_mod_storage()
local deserialize = minetest.deserialize
local serialize = minetest.serialize
local after = minetest.after
local get_player_by_name = minetest.get_player_by_name
local table_copy = table.copy
local is_player = minetest.is_player

local on_unlocks = {
	globals = {},
	prefixes = {}
}
local skills_with_cooldowns = {}
local expire_entities = {} -- "entity" = true

skills.player_skills = deserialize(storage:get_string("player_skills")) or {}



minetest.register_on_mods_loaded(function()
	remove_unregistered_skills_from_db()
	update_db()
end)



minetest.register_on_joinplayer(function(player)
	cast_passive_skills(player:get_player_name())
end)



-- For each skill of each player decreases the cooldown_timer
minetest.register_globalstep(function(dtime)
	local recharged_skills = {}

	for skill_name, def in pairs(skills_with_cooldowns) do
		def.cooldown_timer = def.cooldown_timer - dtime

		if def.cooldown_timer < 0 then
			def.cooldown_timer = 0
			table.insert(recharged_skills, skill_name)
		end

	end

	for i, skill_name in ipairs(recharged_skills) do
		skills_with_cooldowns[skill_name] = nil
	end
end)



minetest.register_on_leaveplayer(function(player, timed_out)
	local pl_name = player:get_player_name()

	for skill_name, def in pairs(skills.player_skills[pl_name]) do
		def:stop()
	end
end)



--
--
-- PUBLIC FUNCTIONS
--
--
function skills.register_skill(internal_name, def)
	def = initialize_def(internal_name, def)

   skills.registered_skills[internal_name:lower()] = def

   init_players_skill_def(def)
end



function skills.register_on_unlock(func, prefix)
	if prefix then
		on_unlocks.prefixes[prefix] = func
	else
		on_unlocks.globals[#on_unlocks.globals] = func
	end
end



function skills.unlock_skill(pl_name, skill_name)
	skill_name = skill_name:lower()

	if not skills.does_skill_exist(skill_name) or pl_name:has_skill(skill_name) then
		return false
	end

	init_subtable(skills.player_skills, pl_name) -- init the skills' table
	init_subtable(skills.player_skills[pl_name], skill_name) -- init the single skill's table
	init_subtable(skills.player_skills[pl_name][skill_name], "data") -- init the skill's data table

	update_player_skill_def(pl_name, skill_name)
	local pl_skill = skills.player_skills[pl_name][skill_name]

	local skill_prefix = skill_name:split(":")[1]
	if on_unlocks.prefixes[skill_prefix] then on_unlocks.prefixes[skill_prefix](pl_skill) end
	for _, global_callback in pairs(on_unlocks.globals) do global_callback(pl_skill) end

	if pl_skill.passive then pl_skill:enable() end
end
string_metatable.__index["unlock_skill"] = skills.unlock_skill



function skills.remove_skill(pl_name, skill_name)
	skill_name = skill_name:lower()

	if not skills.does_skill_exist(skill_name) or not skills.player_skills[pl_name] then
		return false
	end

	pl_name:disable_skill(skill_name)
	skills.player_skills[pl_name][skill_name] = nil
end
string_metatable.__index["remove_skill"] = skills.remove_skill



function skills.get_skill(pl_name, skill_name)
   local pl_skills = skills.player_skills[pl_name]

	if
		not skills.does_skill_exist(skill_name)
		or
		pl_skills == nil or pl_skills[skill_name:lower()] == nil
	then
   	return false
	end

	local pl_skill_table = skills.player_skills[pl_name][skill_name:lower()]
	pl_skill_table.player = get_player_by_name(pl_name)

   return pl_skill_table
end
string_metatable.__index["get_skill"] = skills.get_skill



function skills.has_skill(pl_name, skill_name)
   return pl_name:get_skill(skill_name) ~= false
end
string_metatable.__index["has_skill"] = skills.has_skill



function skills.cast_skill(pl_name, skill_name, args)
   local skill = pl_name:get_skill(skill_name)

	if skill then
		return skill:cast(args)
	else
		return false
	end

end
string_metatable.__index["cast_skill"] = skills.cast_skill



function skills.start_skill(pl_name, skill_name, args)
   local skill = pl_name:get_skill(skill_name)

	if skill then
		return skill:start(args)
	else
		return false
	end

end
string_metatable.__index["start_skill"] = skills.start_skill



function skills.stop_skill(pl_name, skill_name)
   local skill = pl_name:get_skill(skill_name)

	if skill then
		return skill:stop()
	else
		return false
	end

end
string_metatable.__index["stop_skill"] = skills.stop_skill



function skills.enable_skill(pl_name, skill_name)
	local skill = pl_name:get_skill(skill_name)

	if not skill then return false end

	skill:enable()

	return true
end
string_metatable.__index["enable_skill"] = skills.enable_skill



function skills.disable_skill(pl_name, skill_name)
   local skill = pl_name:get_skill(skill_name)

	if not skill then return false end

	skill:disable()

	return true
end
string_metatable.__index["disable_skill"] = skills.disable_skill



function skills.get_skill_def(skill_name)
   if not skills.does_skill_exist(skill_name) then
   	return false
   end

   return skills.registered_skills[skill_name:lower()]
end



function skills.does_skill_exist(skill_name)
   return skill_name and skills.registered_skills[skill_name:lower()]
end



function skills.get_registered_skills(prefix)
	local registered_skills = {}

	for name, def in pairs(skills.registered_skills) do
		if prefix and name:match(prefix) then
			registered_skills[name] = def
		elseif prefix == nil then
			registered_skills[name] = def
		end
	end

   return registered_skills
end



function skills.get_unlocked_skills(pl_name, prefix)
	local pl_skills = skills.player_skills[pl_name]
	local unlocked_skills = {}

	if prefix then
		for name, def in pairs(pl_skills) do
			if name:match(prefix) then
				unlocked_skills[name] = def
			end
		end
	else
		unlocked_skills = pl_skills
	end

	return unlocked_skills
end
string_metatable.__index["get_unlocked_skills"] = skills.get_unlocked_skills



--
--
-- PRIVATE FUNCTIONS
--
--
function init_players_skill_def(def)
   for pl_name, skills in pairs(skills.player_skills) do
		update_player_skill_def(pl_name, def.internal_name)
   end
end



function update_player_skill_def(pl_name, skill_name)
	local def = skills.get_skill_def(skill_name)
	local pl_skills = skills.player_skills[pl_name]
	local skill = pl_skills[def.internal_name]

	if skill then
		skill = table_copy(def)
		skill.pl_name = pl_name
		skill.player = get_player_by_name(pl_name)
		skill.data = update_pl_skill_data(pl_name, skill.internal_name)

		pl_skills[def.internal_name] = skill
	end
end



function initialize_def(internal_name, def)
	local empty_func = function() end
	local cast = def.cast or empty_func
	def.internal_name = internal_name
	def.passive = def.passive or false
	def.cooldown = def.cooldown or 0
	def.description = def.description or "No description."
	def.sounds = def.sounds or {}
	def.attachments = def.attachments or {}
	def.cooldown_timer = 0
	def.is_active = false
	def.chat_warnings = def.chat_warnings or {}
	def.data = def.data or {}
	def.on_start = def.on_start or empty_func
	def.on_stop = def.on_stop or empty_func
	def.data = def.data or {}
	def.data._particles = {}
	def.data._enabled = true

	def.cast = function(self, args)
		if
			(not self.is_active and self.loop_params)
			or
			not get_player_by_name(self.pl_name)
		then
			return false
		end

		if self.cooldown_timer > 0 and not self.loop_params then
			print_remaining_cooldown_seconds(self)
			return false
		end

		if not self.data._enabled then
			if def.chat_warnings.enabled ~= false then
				skills.error(self.pl_name, T("You can't use this skill now!"))
			end

			return false
		end

		cast(self, args)
		play_sound(self, self.sounds.cast, true)

		if self.loop_params and self.loop_params.cast_rate then
			after(self.loop_params.cast_rate, function() self:cast(args) end)
		end

		if not self.loop_params then
			start_cooldown(self)
		end

		return true
	end

	def.start = function(self, args)
		if self.is_active or not get_player_by_name(self.pl_name) then
			return false
		end

		if not self.data._enabled then
			skills.error(self.pl_name, T("You can't use this skill now!"))
			return false
		end

		if self.cooldown_timer > 0 then
			print_remaining_cooldown_seconds(self)
			return false
		end

		self.is_active = true

		-- Create particle spawners
		if self.attachments.particles then
			self.data._particles = {}

			for i, spawner in ipairs(self.attachments.particles) do
				spawner.attached = self.player
				self.data._particles[i] = minetest.add_particlespawner(spawner)
			end
		end

		-- Create hud
		if self.hud then
			self.data._hud = {}

			for i, hud_element in ipairs(self.hud) do
				local name = hud_element.name
				self.data._hud[name] = self.player:hud_add(hud_element)
			end
		end

		-- Play sounds
		play_sound(self, self.sounds.start, true)
		if self.sounds.bgm then
			self.sounds.bgm.loop = true
			self.data._bgm = play_sound(self, self.sounds.bgm)
		end

		-- Stop function after duration
		if self.loop_params and self.loop_params.duration then
			after(self.loop_params.duration, function() self:stop() end)
		end

		-- Attach entities
		if self.attachments.entities then
			for i, entity_def in ipairs(self.attachments.entities) do
				attach_expiring_entity(self, entity_def)
			end
		end

		-- Change sky
		if self.sky then
			local pl = self.player
			self.data._sky = pl:get_sky(true)
			pl:set_sky(self.sky)
		end

		-- Change clouds
		if self.clouds then
			local pl = self.player
			self.data._clouds = pl:get_clouds()
			pl:set_clouds(self.clouds)
		end

		self:on_start()
		start_cooldown(self)

		if self.loop_params and not self.loop_params.cast_rate then
			return true
		else
			self:cast(args)
		end

		return true
   end

	def.stop = function(self)
		if not self.is_active then return false end
      self.is_active = false

		if not get_player_by_name(self.pl_name) then return false end

		play_sound(self, self.sounds.stop, true)

		-- I don't know. MT is weird or maybe my code is just bugged:
		-- without this after, if the skills ends very quickly the
		-- spawner and the sound simply... don't stop.
		after(0, function()
			-- Stop sound
			if self.data._bgm then minetest.sound_stop(self.data._bgm) end

			-- Remove particles
			if self.data._particles then
				for i, spawner_id in pairs(self.data._particles) do
					minetest.delete_particlespawner(spawner_id)
				end
			end

			-- Remove hud
			if self.data._hud then
				for name, id in pairs(self.data._hud) do
					self.player:hud_remove(id)
				end
			end
		end)

		-- Restore sky
		if self.sky then
			local pl = self.player
			pl:set_sky(self.data._sky)
			self.data._sky = {}
		end

		-- Restore clouds
		if self.clouds then
			local pl = self.player
			pl:set_clouds(self.data._clouds)
			self.data._clouds = {}
		end

		self:on_stop()

		return true
   end

	def.disable = function(self)
		self:stop()
		self.data._enabled = false
	end

	def.enable = function(self)
		self.data._enabled = true
		cast_passive_skills(self.pl_name)
	end

   return def
end



function init_subtable(table, subtable_name)
   table[subtable_name] = table[subtable_name] or {}
end



function update_db()
	local pl_skills_without_userdata = table_copy(skills.player_skills)
	remove_userdata(pl_skills_without_userdata)

   storage:set_string("player_skills", serialize(pl_skills_without_userdata))

	after(10, update_db)
end



function cast_passive_skills(pl_name)
	if not skills.player_skills[pl_name] then return false end

	for name, def in pairs(skills.player_skills[pl_name]) do
		if def.passive and def.data._enabled then
			def.player = get_player_by_name(pl_name)
			def:start()
		end
	end
end



function remove_unregistered_skills_from_db()
	for pl_name, skills in pairs(skills.player_skills) do
		for skill_name, def in pairs(skills) do
			if not skills.get_skill_def(skill_name) then skills[skill_name] = nil end
		end
	end
end



function play_sound(skill, sound, ephemeral)
	if not sound then return false end

	sound.pos = skill.player:get_pos()
	if sound.to_player then sound.to_player = skill.pl_name end
	if sound.object then sound.object = skill.player end

	return minetest.sound_play(sound, sound, ephemeral)
end



function remove_userdata(t)
	for key, value in pairs(t) do
		if type(value) == "table" then remove_userdata(value) end
		if is_player(value) or type(value) == "userdata" then t[key] = nil end
	end
end



function print_remaining_cooldown_seconds(skill)
	local remaining_seconds = math.floor(skill.cooldown_timer*10) / 10

	if skill.chat_warnings.cooldown ~= false then
		skills.error(skill.pl_name, T("You have to wait @1 seconds to use this again!", remaining_seconds))
	end
end



function update_pl_skill_data(pl_name, skill_name)
	local skill_def = skills.get_skill_def(skill_name)
	local pl_data = table_copy(skills.player_skills[pl_name][skill_name].data)

   -- adding any new data's properties declared in the def table
   -- to the already existing player's data table
   for key, def_value in pairs(skill_def.data) do
		if pl_data[key] == nil then pl_data[key] = def_value end

		-- if an old property's type changed, then reset it
		if type(pl_data[key]) ~= type(def_value) then pl_data[key] = def_value end
   end

	return pl_data
end



function start_cooldown(skill)
	skill.cooldown_timer = skill.cooldown
	skills_with_cooldowns[skill.pl_name .. ":" .. skill.internal_name] = skill
end



function attach_expiring_entity(skill, def)
	local pos = def.pos
	local name = def.name
	local bone = def.bone or ""
	local rotation = def.rotation or {x=0, y=0, z=0}
	local forced_visible = def.forced_visible or false

	local on_step = minetest.registered_entities[name].on_step or function() end
	local on_activate = minetest.registered_entities[name].on_activate or function() end

	-- Overriding the entity's function
	if not expire_entities[name] then
		minetest.registered_entities[name].pl_name = ""

		minetest.registered_entities[name].on_activate = function(self, staticdata, dtime_s)
			self.pl_name = staticdata

			if staticdata == "" then
				self.object:remove()
				return
			end

			on_activate(self, staticdata, dtime_s)
		end

		minetest.registered_entities[name].on_step = function(self, dtime, moveresult)
			local infotext = self.object:get_properties().infotext

			-- if it's a skill, remove the entity if the skill has stopped
			if #infotext:split("skill->") == 1 then
				local skill_name = infotext:split("skill->")[1]
				local skill = self.pl_name:get_skill(skill_name)

				if not skill or not skill.is_active then
					self.object:remove()
					return
				end
			end

			on_step(self, dtime, moveresult)
  		end
	else
		expire_entities[name] = true
	end

  local entity = minetest.add_entity(pos, name, skill.pl_name)
  entity:set_properties({infotext = "skill->"..skill.internal_name})
  entity:set_attach(skill.player, bone, pos, rotation, forced_visible)
end