skills.tests = {}

function log(msg)
   core.log(msg)
end



local S = core.get_translator("skills")
local test_prefix = "_skillz_test:"
local test_player_name = nil
local test_results = {passed = 0, failed = 0, skipped = 0, total = 0}
local test_states = {}

-- Register the expiring entity for the test at load time
local expiring_entity_name_for_test = "skills:test_expiring_entity"
skills.register_expiring_entity(expiring_entity_name_for_test, {
   initial_properties = {
      visual = "cube",
      visual_size = {x = 0.2, y = 0.2, z = 0.2},
      physical = false,
      collisionbox = {0, 0, 0, 0, 0, 0},
      static_save = false,
   },
})

-- Register the expiring entity for the on_remove test at load time
local expiring_entity_name_on_remove = "skills:test_expiring_entity_on_remove"
skills.register_expiring_entity(expiring_entity_name_on_remove, {
   description = "Test Expiring Entity for on_remove",
   initial_properties = {
      visual = "cube",
      visual_size = {x = 0.2, y = 0.2, z = 0.2},
      physical = false,
      collisionbox = {0, 0, 0, 0, 0, 0},
      static_save = false,
   },
   on_remove = function(self)
      -- This will be replaced in the test function
   end,
})

-- Helper function to update test state
local function update_test_state(test_name, new_state)
   local current_state = test_states[test_name]
   if current_state == "failed" then return end
   if current_state == "skipped" and (new_state == "passed" or new_state == "pending") then return end
   if current_state == "passed" and new_state == "pending" then return end
   test_states[test_name] = new_state
end

function skills.tests.log_test(msg)
   skills.log("action", "[TEST] " .. msg)
   if test_player_name then
      core.chat_send_player(test_player_name, "[SKILLS TEST] " .. msg)
   end
end



local function log_pass(test_name)
   skills.tests.log_test(test_name .. ": PASS")
   update_test_state(test_name, "passed")
end

local function log_fail(test_name, reason)
   skills.tests.log_test(test_name .. core.colorize("#e6482e", ": FAIL" .. (reason and (" - " .. reason) or "")))
   update_test_state(test_name, "failed")
end

local function log_skip(test_name, reason)
   skills.tests.log_test(test_name .. ": SKIP" .. (reason and (" - " .. reason) or ""))
   update_test_state(test_name, "skipped")
end



local function cleanup_test_skills()
   local test_name = "cleanup_test_skills"
   skills.tests.log_test("Cleaning up test skills...")
   local skills_to_remove = {}
   for name, _ in pairs(skills.registered_skills) do
      if string.sub(name, 1, #test_prefix) == test_prefix then
         table.insert(skills_to_remove, name)
      end
   end
   for _, name in ipairs(skills_to_remove) do
      skills.registered_skills[name] = nil
      skills.tests.log_test("Removed registered skill: " .. name)
   end

   if test_player_name then
      local player_ref = core.get_player_by_name(test_player_name)
      if player_ref and skills.player_skills[test_player_name] then
         local player_skills_to_remove = {}
         for name, _ in pairs(skills.player_skills[test_player_name]) do
            if string.sub(name, 1, #test_prefix) == test_prefix then
               table.insert(player_skills_to_remove, name)
            end
         end
         for _, name in ipairs(player_skills_to_remove) do
            skills.remove_skill(test_player_name, name)
            skills.tests.log_test("Removed player skill: " .. name .. " from " .. test_player_name)
         end
      end
   end
   skills.tests.log_test("Cleanup complete.")
   log_pass(test_name)
end



local function test_registration_simple(pl_name)
   local test_name = "test_registration_simple"
   local skill_name = test_prefix .. "simple"
   skills.register_skill(skill_name, {
      name = "Test Simple Skill",
      description = "A basic skill for testing.",
      cast = function(self)
         skills.tests.log_test(self.name .. " cast!")
         return true
      end,
   })
   assert(skills.does_skill_exist(skill_name), "Skill should exist after registration")
   local def = skills.get_skill_def(skill_name)
   assert(def and def.name == "Test Simple Skill", "Skill definition should be retrievable")
   assert(not skills.get_skill_def(test_prefix .. "nonexistent"), "Non-existent skill def should be nil")
   assert(not skills.does_skill_exist(test_prefix .. "nonexistent"), "Non-existent skill should not exist")
   log_pass(test_name)
end



local function test_unlock_remove(pl_name)
   local test_name = "test_unlock_remove"
   local skill_name = test_prefix .. "unlock_remove_target"

   skills.register_skill(skill_name, {name = "Unlock/Remove Test Skill"})
   assert(skills.does_skill_exist(skill_name), "Skill registration failed")

   local unlock_ok = skills.unlock_skill(pl_name, skill_name)
   assert(unlock_ok, "Unlocking skill should succeed")
   assert(skills.has_skill(pl_name, skill_name), "Player should have the skill after unlock")

   local unlocked_skills = skills.get_unlocked_skills(pl_name)
   assert(unlocked_skills[skill_name] ~= nil, "Unlocked skill should be in the list")

   local remove_ok = skills.remove_skill(pl_name, skill_name)
   assert(remove_ok, "Removing skill should succeed")
   assert(not skills.has_skill(pl_name, skill_name), "Player should not have the skill after remove")

   unlocked_skills = skills.get_unlocked_skills(pl_name)
   assert(not unlocked_skills[skill_name], "Removed skill should not be in the list")

   local remove_again_ok = skills.remove_skill(pl_name, skill_name)
   assert(not remove_again_ok, "Removing non-existent skill should fail")

   skills.unlock_skill(pl_name, skill_name)
   local unlock_again_ok = skills.unlock_skill(pl_name, skill_name)
   assert(not unlock_again_ok, "Unlocking already unlocked skill should fail")

   skills.remove_skill(pl_name, skill_name)
   log_pass(test_name)
end



local function test_cast_cooldown(pl_name)
   local test_name = "test_cast_cooldown"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "cooldown"
   local cooldown = 2.0
   local cast_count = 0

   skills.register_skill(skill_name, {
      name = "Test Cooldown Skill",
      cooldown = cooldown,
      cast = function(self)
         cast_count = cast_count + 1
         skills.tests.log_test(self.name .. " cast!")
         return true
      end
   })
   assert(skills.does_skill_exist(skill_name), "Cooldown skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Cooldown skill unlock failed")

   local cast1_ok = skills.cast_skill(pl_name, skill_name)
   assert(cast1_ok, "First cast should succeed")
   assert(cast_count == 1, "Cast count should be 1 after first cast")

   local cast2_ok = skills.cast_skill(pl_name, skill_name)
   assert(not cast2_ok, "Second cast (during cooldown) should fail")
   assert(cast_count == 1, "Cast count should still be 1")

   core.after(cooldown + 0.1, function()
      local cast3_ok = skills.cast_skill(pl_name, skill_name)
      if cast3_ok and cast_count == 2 then
         log_pass(test_name)
      else
         log_fail(test_name, "Cast after cooldown failed or count incorrect. Expected 2, got " .. cast_count)
      end
      skills.remove_skill(pl_name, skill_name)
      skills.tests.log_test(test_name .. ": Cleaned up skill " .. skill_name)
   end)
end



local function test_loop_skill(pl_name)
   local test_name = "test_loop_skill"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "loop"
   local cast_rate = 0.5
   local duration = 1.8
   local player = core.get_player_by_name(pl_name)
   assert(player, "Test player object not found")
   local cast_count = 0
   local stopped = false

   skills.register_skill(skill_name, {
      name = "Test Loop Skill",
      loop_params = {
         cast_rate = cast_rate,
         duration = duration,
      },
      cast = function(self)
         cast_count = cast_count + 1
         skills.tests.log_test(self.name .. " cast #" .. cast_count)
         return true
      end,
      on_stop = function(self)
         stopped = true
         skills.tests.log_test(self.name .. " stopped.")
      end
   })
   assert(skills.does_skill_exist(skill_name), "Loop skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Loop skill unlock failed")

   local start_ok = skills.start_skill(pl_name, skill_name)
   assert(start_ok, "Starting loop skill should succeed")

   local skill = skills.get_skill(pl_name, skill_name)
   assert(skill, "Failed to get skill table after starting")
   assert(skill.is_active, "Loop skill should be active after starting")

   core.after(duration + 0.2, function()
      local final_skill = skills.get_skill(pl_name, skill_name)
      local expected_casts = math.floor(duration / cast_rate) + 1

      if not final_skill or final_skill.is_active then
         log_fail(test_name, "Skill should be inactive after duration.")
      elseif not stopped then
         log_fail(test_name, "on_stop callback was not called.")
      elseif cast_count < expected_casts or cast_count > expected_casts + 1 then
         log_fail(test_name, "Incorrect number of casts. Expected ~" .. expected_casts .. ", got " .. cast_count)
      else
         log_pass(test_name)
      end
   end)
end



local function test_loop_manual_stop(pl_name)
   local test_name = "test_loop_manual_stop"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "loop_stop"
   local cast_rate = 0.5
   local stop_time = 1.2
   local cast_count = 0
   local stopped_flag = false

   skills.register_skill(skill_name, {
      name = "Manual Stop Loop Skill",
      loop_params = {cast_rate = cast_rate},
      cast = function(self)
         cast_count = cast_count + 1
         return true
      end,
      on_stop = function(self)
         stopped_flag = true
      end
   })
   assert(skills.does_skill_exist(skill_name), "Manual stop loop skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Manual stop loop skill unlock failed")

   local start_ok = skills.start_skill(pl_name, skill_name)
   assert(start_ok, "Starting manual stop loop skill should succeed")

   core.after(stop_time, function()
      local stop_ok = skills.stop_skill(pl_name, skill_name)
      if not stop_ok then
         log_fail(test_name, "skills.stop_skill failed unexpectedly")
         skills.remove_skill(pl_name, skill_name)
         skills.tests.log_test(test_name .. ": Cleaned up registered skill " .. skill_name)
         return
      end

      core.after(0.4, function()
         local skill = skills.get_skill(pl_name, skill_name)
         local expected_casts = math.floor(stop_time / cast_rate) + 1
         local test_passed = true
         local fail_reason = ""

         if skill and skill.is_active then
            test_passed = false
            fail_reason = "Skill should be inactive after manual stop."
         elseif not stopped_flag then
            test_passed = false
            fail_reason = "on_stop callback flag was not set after manual stop."
         end

         if test_passed and (cast_count < expected_casts or cast_count > expected_casts + 1) then
            test_passed = false
            fail_reason = "Incorrect number of casts before stop. Expected ~" .. expected_casts .. ", got " .. cast_count
         end

         if test_passed then
            log_pass(test_name)
         else
            log_fail(test_name, fail_reason)
         end

         skills.remove_skill(pl_name, skill_name)
         skills.tests.log_test(test_name .. ": Cleaned up registered skill " .. skill_name)
      end)
   end)
end



local function test_passive_skill(pl_name)
   local test_name = "test_passive_skill"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "passive"
   local cast_count = 0

   skills.register_skill(skill_name, {
      name = "Test Passive Skill",
      passive = true,
      loop_params = {cast_rate = 0.1},
      cast = function(self)
         cast_count = cast_count + 1
         return true
      end
   })
   assert(skills.does_skill_exist(skill_name), "Passive skill registration failed")

   local unlock_ok = skills.unlock_skill(pl_name, skill_name)
   assert(unlock_ok, "Unlocking passive skill should succeed")

   core.after(0.2, function()
      local skill = skills.get_skill(pl_name, skill_name)
      if not skill then
         log_fail(test_name, "Could not get skill table for passive skill.")
      elseif not skill.is_active then
         log_fail(test_name, "Passive skill should be active after unlock")
      elseif cast_count == 0 then
         log_fail(test_name, "Passive skill cast function not called.")
      else
         log_pass(test_name)
      end
      skills.remove_skill(pl_name, skill_name)
      skills.tests.log_test(test_name .. ": Cleaned up skill " .. skill_name)
   end)
end



local function test_enable_disable(pl_name)
   local test_name = "test_enable_disable"
   local skill_name = test_prefix .. "enable_disable"

   skills.register_skill(skill_name, {
      name = "Enable/Disable Test",
      cast = function() return true end,
      on_stop = function(self)
         self.stopped = true
      end,
      on_start = function(self)
         self.stopped = false
      end,
   })
   assert(skills.does_skill_exist(skill_name), "Enable/disable skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Enable/disable skill unlock failed")

   -- disabling
   local disable_ok = skills.disable_skill(pl_name, skill_name)
   assert(disable_ok, "Disabling skill should succeed")
   local skill = skills.get_skill(pl_name, skill_name)
   assert(skill and not skill.data._enabled, "Skill should be disabled")

   -- casting disabled
   local cast_disabled_ok = skills.start_skill(pl_name, skill_name)
   assert(not cast_disabled_ok, "Starting disabled skill should fail")

   -- enabling
   local enable_ok = skills.enable_skill(pl_name, skill_name)
   assert(enable_ok, "Enabling skill should succeed")
   assert(skill and skill.data._enabled, "Skill should be enabled")

   -- casting enabled
   local cast_enabled_ok = skills.start_skill(pl_name, skill_name)
   assert(cast_enabled_ok, "Casting enabled skill should succeed")
   assert(skill and (skill.stopped == false), "Skill.stopped should be false")

   -- disabling again and testing on_stop
   disable_ok = skills.disable_skill(pl_name, skill_name)
   assert(disable_ok, "Disabling skill should succeed")
   assert(skill and not skill.data._enabled, "Skill should be disabled")
   assert(skill and skill.stopped, "Skill.stopped should be true")

   skills.remove_skill(pl_name, skill_name)
   skills.tests.log_test(test_name .. ": Cleaned up skill " .. skill_name)
   log_pass(test_name)
end



local function test_physics(pl_name)
   local test_name = "test_physics"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "physics"
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for physics test")
   local original_physics = player:get_physics_override()
   local speed_multiplier = 2.0
   local duration = 0.5
   local physics_ok = true

   skills.register_skill(skill_name, {
      name = "Physics Test Skill",
      loop_params = {duration = duration},
      physics = {
         operation = "multiply",
         speed = speed_multiplier,
      }
   })
   assert(skills.does_skill_exist(skill_name), "Physics skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Physics skill unlock failed")

   local start_ok = skills.start_skill(pl_name, skill_name)
   assert(start_ok, "Starting physics skill failed")

   core.after(duration / 2, function()
      local current_physics = player:get_physics_override()
      local expected_speed = original_physics.speed * speed_multiplier
      if math.abs(current_physics.speed - expected_speed) < 0.01 then
         skills.tests.log_test(test_name .. ": Speed check while active PASSED")
      else
         skills.tests.log_test(test_name .. ": Speed check while active FAILED. Expected " .. expected_speed .. ", got " .. current_physics.speed)
         physics_ok = false
      end
   end)

   core.after(duration + 0.2, function()
      local restored_physics = player:get_physics_override()
      if math.abs(restored_physics.speed - original_physics.speed) < 0.01 and
          math.abs(restored_physics.jump - original_physics.jump) < 0.01 and
          math.abs(restored_physics.gravity - original_physics.gravity) < 0.01 then
         skills.tests.log_test(test_name .. " (restore check): PASS")
      else
         skills.tests.log_test(test_name .. " (restore check): FAIL - Physics not restored correctly.")
         skills.tests.log_test("Original: " .. dump(original_physics))
         skills.tests.log_test("Restored: " .. dump(restored_physics))
         physics_ok = false
      end

      if physics_ok then
         log_pass(test_name)
      else
         log_fail(test_name, "One or more physics checks failed.")
      end

      skills.remove_skill(pl_name, skill_name)
      skills.tests.log_test(test_name .. ": Cleaned up registered skill " .. skill_name)
   end)
end



local function test_monoids(pl_name)
   local test_name = "test_monoids"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")

   if not (player_monoids and player_monoids.jump) then
      log_skip(test_name, "player_monoids mod or player_monoids.jump not found. This test requires it.")
      return
   end

   local player = core.get_player_by_name(pl_name)
   if not player then
      log_fail(test_name, "Player object not found.")
      return
   end

   local original_jump = player_monoids.jump:value(player)
   skills.tests.log_test(test_name .. ": Original jump multiplier: " .. tostring(original_jump))

   local skill_simple = test_prefix .. "monoid_simple_jump"
   local skill_checkout = test_prefix .. "monoid_checkout_jump"
   local skill_branch = test_prefix .. "monoid_branch_jump"
   local duration = 0.6

   -- Scenario 1: Simple monoid application on main branch
   skills.register_skill(skill_simple, {
      name = "Monoid Simple Jump",
      loop_params = {duration = duration},
      monoids = {jump = {ref = "player_monoids.jump", value = 1.5}}
   })
   assert(skills.unlock_skill(pl_name, skill_simple), test_name .. ": Unlock failed for " .. skill_simple)
   assert(skills.start_skill(pl_name, skill_simple), test_name .. ": Start failed for " .. skill_simple)

   core.after(duration / 2, function()
      local val = player_monoids.jump:value(player)
      local expected = original_jump * 1.5
      assert(math.abs(val - expected) < 0.01, test_name .. ": Scenario 1: jump value during skill should be " .. expected .. ", got " .. val)
      local branch = player_monoids.jump:get_active_branch(player):get_name()
      assert(branch == "main", test_name .. ": Scenario 1: active branch should be 'main', got '" .. branch .. "'")
   end)

   core.after(duration + 0.2, function()
      local val = player_monoids.jump:value(player)
      assert(math.abs(val - original_jump) < 0.01, test_name .. ": Scenario 1: jump value after skill stop should be " .. original_jump .. ", got " .. val)
      skills.remove_skill(pl_name, skill_simple)

      -- Scenario 2: checkout_branch_while_active
      skills.register_skill(skill_checkout, {
         name = "Monoid Checkout Jump",
         loop_params = {duration = duration},
         monoids = {
            checkout_branch_while_active = true,
            jump = {ref = "player_monoids.jump", value = 2.0, branch = "test_jump_branch"}
         }
      })
      assert(skills.unlock_skill(pl_name, skill_checkout), test_name .. ": Unlock failed for " .. skill_checkout)
      assert(skills.start_skill(pl_name, skill_checkout), test_name .. ": Start failed for " .. skill_checkout)

      core.after(duration / 2, function()
         local branch = player_monoids.jump:get_active_branch(player):get_name()
         assert(branch == "test_jump_branch", test_name .. ": Scenario 2: active branch should be 'test_jump_branch', got '" .. branch .. "'")
         local val = player_monoids.jump:value(player)
         local expected = original_jump * 2.0
         assert(math.abs(val - expected) < 0.01, test_name .. ": Scenario 2: jump value on branch should be " .. expected .. ", got " .. val)
      end)

      core.after(duration + 0.2, function()
         local branch = player_monoids.jump:get_active_branch(player):get_name()
         assert(branch == "main", test_name .. ": Scenario 2: active branch after stop should revert to 'main', got '" .. branch .. "'")
         local val = player_monoids.jump:value(player)
         assert(math.abs(val - original_jump) < 0.01, test_name .. ": Scenario 2: jump value on 'main' after stop should be " .. original_jump .. ", got " .. val)
         skills.remove_skill(pl_name, skill_checkout)

         -- Scenario 3: Apply to specific branch, no checkout
         skills.register_skill(skill_branch, {
            name = "Monoid Branch Jump",
            loop_params = {duration = duration},
            monoids = {
               jump = {ref = "player_monoids.jump", value = 2.5, branch = "other_jump_branch"}
            }
         })
         assert(skills.unlock_skill(pl_name, skill_branch), test_name .. ": Unlock failed for " .. skill_branch)
         assert(skills.start_skill(pl_name, skill_branch), test_name .. ": Start failed for " .. skill_branch)

         core.after(duration / 2, function()
            local branch = player_monoids.jump:get_active_branch(player):get_name()
            assert(branch == "main", test_name .. ": Scenario 3: active branch should remain 'main', got '" .. branch .. "'")
            local val_main = player_monoids.jump:value(player)
            assert(math.abs(val_main - original_jump) < 0.01, test_name .. ": Scenario 3: jump value on 'main' should be " .. original_jump .. ", got " .. val_main)
            local val_other = player_monoids.jump:value(player, "other_jump_branch")
            local expected = original_jump * 2.5
            assert(math.abs(val_other - expected) < 0.01, test_name .. ": Scenario 3: jump value on 'other_jump_branch' should be " .. expected .. ", got " .. val_other)
         end)

         core.after(duration + 0.2, function()
            local branch = player_monoids.jump:get_active_branch(player):get_name()
            assert(branch == "main", test_name .. ": Scenario 3: active branch after stop should be 'main', got '" .. branch .. "'")
            local val_main = player_monoids.jump:value(player)
            assert(math.abs(val_main - original_jump) < 0.01, test_name .. ": Scenario 3: jump value on 'main' after stop should be " .. original_jump .. ", got " .. val_main)
            local val_other = player_monoids.jump:value(player, "other_jump_branch")
            assert(math.abs(val_other - original_jump) < 0.01, test_name .. ": Scenario 3: jump value on 'other_jump_branch' after stop should be " .. original_jump .. ", got " .. val_other)
            skills.remove_skill(pl_name, skill_branch)

            -- Scenario 4: Stacking/Priority of checkout_branch_while_active
            skills.tests.log_test(test_name .. ": Scenario 4 - Stacking/Priority of checkout_branch_while_active")
            local skill_a = test_prefix .. "monoid_stack_a"
            local skill_b = test_prefix .. "monoid_stack_b"
            local branch_a = "branch_A"
            local branch_b = "branch_B"
            local val_a = 1.7
            local val_b = 2.3

            skills.register_skill(skill_a, {
               name = "Monoid Stack A",
               loop_params = { duration = duration },
               monoids = {
                  checkout_branch_while_active = true,
                  jump = { ref = "player_monoids.jump", value = val_a, branch = branch_a }
               },
               on_stop = function(self)
                  core.log("warning", "Monoid Stack A stopped")
               end
            })
            skills.register_skill(skill_b, {
               name = "Monoid Stack B",
               loop_params = { duration = duration },
               monoids = {
                  checkout_branch_while_active = true,
                  jump = { ref = "player_monoids.jump", value = val_b, branch = branch_b }
               },
               on_stop = function(self)
                  core.log("warning", "Monoid Stack B stopped")
               end
            })

            assert(skills.unlock_skill(pl_name, skill_a), test_name .. ": Unlock failed for " .. skill_a)
            assert(skills.unlock_skill(pl_name, skill_b), test_name .. ": Unlock failed for " .. skill_b)
            assert(skills.start_skill(pl_name, skill_a), test_name .. ": Start failed for " .. skill_a)

            core.after(duration / 5, function()
               local active_branch = player_monoids.jump:get_active_branch(player):get_name()
               assert(active_branch == branch_a, test_name .. ": Scenario 4: After A, active branch should be '" .. branch_a .. "', got '" .. active_branch .. "'")
               local val = player_monoids.jump:value(player)
               local expected = original_jump * val_a
               assert(math.abs(val - expected) < 0.01, test_name .. ": Scenario 4: After A, jump value should be " .. expected .. ", got " .. val)
               
               -- Start B
               assert(skills.start_skill(pl_name, skill_b), test_name .. ": Start failed for " .. skill_b)
               core.after(duration / 5, function()
                  local active_branch2 = player_monoids.jump:get_active_branch(player):get_name()
                  assert(active_branch2 == branch_b, test_name .. ": Scenario 4: After B, active branch should be '" .. branch_b .. "', got '" .. active_branch2 .. "'")
                  local val2 = player_monoids.jump:value(player)
                  local expected2 = original_jump * val_b
                  assert(math.abs(val2 - expected2) < 0.01, test_name .. ": Scenario 4: After B, jump value should be " .. expected2 .. ", got " .. val2)
                 
                  -- Stop B
                  skills.remove_skill(pl_name, skill_b)
                  core.after(duration / 5, function()
                     local active_branch3 = player_monoids.jump:get_active_branch(player):get_name()
                     assert(active_branch3 == branch_a, test_name .. ": Scenario 4: After B stop, active branch should revert to '" .. branch_a .. "', got '" .. active_branch3 .. "'")
                     local val3 = player_monoids.jump:value(player)
                     local expected3 = original_jump * val_a
                     assert(math.abs(val3 - expected3) < 0.01, test_name .. ": Scenario 4: After B stop, jump value should be " .. expected3 .. ", got " .. val3)
                     
                     -- Stop A
                     skills.remove_skill(pl_name, skill_a)
                     core.after(duration / 4, function()
                        local active_branch4 = player_monoids.jump:get_active_branch(player):get_name()
                        assert(active_branch4 == "main", test_name .. ": Scenario 4: After A stop, active branch should revert to 'main', got '" .. active_branch4 .. "'")
                        local val4 = player_monoids.jump:value(player)
                        assert(math.abs(val4 - original_jump) < 0.01, test_name .. ": Scenario 4: After A stop, jump value should be " .. original_jump .. ", got " .. val4)
                        log_pass(test_name)
                     end)
                  end)
               end)
            end)
         end)
      end)
   end)
end



local function test_skill_based_on(pl_name)
   local test_name = "test_skill_based_on"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started")

   local prefix = test_prefix
   local layer_a = prefix .. "layer_a"
   local layer_b = prefix .. "layer_b"
   local skill_c = prefix .. "skill_c"

   -- Register prefix config with base_layers = {layer_a}
   skills.register_prefix_config(prefix, {base_layers = {layer_a}})

   -- Register Layer A
   skills.register_layer(layer_a, {
      data = {layer_a_val = 10, common_val = "from_a"},
      cooldown = 5,
      on_start = function(self)
         self.data.layer_a_started = true
         -- No logic in Layer A
      end,
   })

   -- Register Layer B
   skills.register_layer(layer_b, {
      data = {layer_b_val = 20, common_val = "from_b"},
      description = "Layer B Description",
      on_start = function(self)
         self.data.layer_b_started = true
         self:logic() -- Call logic (was cast)
      end,
      cast = function(self)
         self.data.layer_b_cast = true
         return true
      end,
   })

   -- Register Skill C based on Layer A and Layer B
   skills.register_skill_based_on({layer_a, layer_b}, skill_c, {
      data = {skill_c_val = 30, common_val = "from_c"},
      cooldown = 15,
      description = "Skill C Description",
      on_start = function(self)
         self.data.skill_c_started = true
         self:logic() -- Call logic (was cast)
      end,
      cast = function(self)
         self.data.skill_c_cast = true
         return true
      end,
   })

   -- Validation (Registration)
   local def = skills.get_skill_def(skill_c)
   if not def then
      log_fail(test_name, "Skill C definition not found")
      return
   end
   if def.data.layer_a_val ~= 10 then
      log_fail(test_name, "layer_a_val not inherited")
      return
   end
   if def.data.layer_b_val ~= 20 then
      log_fail(test_name, "layer_b_val not inherited")
      return
   end
   if def.data.common_val ~= "from_c" then
      log_fail(test_name, "common_val not overridden")
      return
   end
   if def.cooldown ~= 15 then
      log_fail(test_name, "cooldown not overridden")
      return
   end
   if def.description ~= "Skill C Description" then
      log_fail(test_name, "description not overridden")
      return
   end
   if def.data.skill_c_val ~= 30 then
      log_fail(test_name, "skill_c_val not set")
      return
   end
   if def.is_layer then
      log_fail(test_name, "Skill C should not be a layer")
      return
   end
   if skills.does_skill_exist(layer_a) then
      log_fail(test_name, "Layer A should not be a skill")
      return
   end
   if skills.does_skill_exist(layer_b) then
      log_fail(test_name, "Layer B should not be a skill")
      return
   end
   if skills.unlock_skill(pl_name, layer_a) then
      log_fail(test_name, "Should not be able to unlock Layer A")
      return
   end
   if skills.unlock_skill(pl_name, layer_b) then
      log_fail(test_name, "Should not be able to unlock Layer B")
      return
   end

   -- Validation (Runtime)
   assert(skills.unlock_skill(pl_name, skill_c), "Unlocking Skill C failed")
   local skill = skills.get_skill(pl_name, skill_c)
   if not skill then
      log_fail(test_name, "Skill C instance not found")
      return
   end
   if skill.data.layer_a_val ~= 10 then
      log_fail(test_name, "Instance: layer_a_val not inherited")
      return
   end
   if skill.data.layer_b_val ~= 20 then
      log_fail(test_name, "Instance: layer_b_val not inherited")
      return
   end
   if skill.data.skill_c_val ~= 30 then
      log_fail(test_name, "Instance: skill_c_val not set")
      return
   end
   if skill.data.common_val ~= "from_c" then
      log_fail(test_name, "Instance: common_val not overridden")
      return
   end
   if skill.cooldown ~= 15 then
      log_fail(test_name, "Instance: cooldown not overridden")
      return
   end
   if skill.description ~= "Skill C Description" then
      log_fail(test_name, "Instance: description not overridden")
      return
   end

   assert(skills.start_skill(pl_name, skill_c), "Starting Skill C failed")

   -- Check flags set by on_start and logic (cast) functions
   if not skill.data.layer_a_started then
      log_fail(test_name, "on_start chaining failed (Layer A)")
      return
   end
   if not skill.data.layer_b_started then
      log_fail(test_name, "on_start chaining failed (Layer B)")
      return
   end
   if not skill.data.skill_c_started then
      log_fail(test_name, "on_start chaining failed (Skill C)")
      return
   end
   if not skill.data.layer_b_cast then
      log_fail(test_name, "logic (cast) chaining failed (Layer B)")
      return
   end
   if not skill.data.skill_c_cast then
      log_fail(test_name, "logic (cast) chaining failed (Skill C)")
      return
   end

   -- Check cooldown was applied after start
   if skill.cooldown_timer <= 0 then
      log_fail(test_name, "Cooldown timer not set after start")
      return
   end

   -- Attempt to start again during cooldown (should fail)
   if skills.start_skill(pl_name, skill_c) then
      log_fail(test_name, "Starting skill during cooldown should fail")
      return
   end

   skills.stop_skill(pl_name, skill_c)
   skills.remove_skill(pl_name, skill_c)
   skills.registered_skills[skill_c] = nil
   skills.registered_skills[layer_b] = nil
   skills.registered_skills[layer_a] = nil
   skills.prefix_configs[prefix] = nil

   log_pass(test_name)
end



local function test_expiring_entity(pl_name)
   local test_name = "test_expiring_entity"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")

   local entity_name = expiring_entity_name_for_test
   local skill_name = test_prefix .. "expiring_entity_skill"
   local duration = 1.5
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for expiring entity test")

   local entity_exists_mid = false
   local entity_removed_after = false
   local spawned_entity_ref = nil

   skills.register_skill(skill_name, {
      name = "Expiring Entity Test Skill",
      loop_params = {duration = duration},
      on_start = function(self)
         local pos = self.player:get_pos()
         pos.y = pos.y + 1
         spawned_entity_ref = self:add_entity(pos, entity_name)
         if spawned_entity_ref then
            skills.tests.log_test(test_name .. ": Entity spawned by on_start.")
         else
            log_fail(test_name, "Failed to spawn entity using skill:add_entity.")
            self:stop("cancelled")
         end
         return spawned_entity_ref ~= nil
      end,
   })
   assert(skills.does_skill_exist(skill_name), "Expiring entity skill registration failed")
   skills.tests.log_test(test_name .. ": Registered skill " .. skill_name)

   assert(skills.unlock_skill(pl_name, skill_name), "Expiring entity skill unlock failed")
   local start_ok = skills.start_skill(pl_name, skill_name)
   assert(start_ok, "Starting expiring entity skill failed")
   skills.tests.log_test(test_name .. ": Skill started.")

   core.after(duration / 2, function()
      if not spawned_entity_ref then
         skills.tests.log_test(test_name .. ": Mid-duration check skipped (entity failed to spawn).")
         return
      end
      local luaentity = spawned_entity_ref:get_luaentity()
      if luaentity then
         skills.tests.log_test(test_name .. ": Mid-duration check PASSED (Entity exists).")
         entity_exists_mid = true
      else
         skills.tests.log_test(test_name .. ": Mid-duration check FAILED (Entity does not exist).")
         entity_exists_mid = false
      end
   end)

   core.after(duration + 0.5, function()
      if not spawned_entity_ref then
         skills.tests.log_test(test_name .. ": Post-duration check skipped (entity failed to spawn).")
         log_fail(test_name, "Entity failed to spawn initially.")
         skills.remove_skill(pl_name, skill_name)
         return
      end
      local luaentity = spawned_entity_ref:get_luaentity()
      if not luaentity then
         skills.tests.log_test(test_name .. ": Post-duration check PASSED (Entity removed).")
         entity_removed_after = true
      else
         skills.tests.log_test(test_name .. ": Post-duration check FAILED (Entity still exists).")
         entity_removed_after = false
         luaentity.object:remove()
      end
      if entity_exists_mid and entity_removed_after then
         log_pass(test_name)
      else
         local reason = ""
         if not entity_exists_mid then reason = reason .. "Entity did not exist mid-duration. " end
         if not entity_removed_after then reason = reason .. "Entity was not removed after skill stopped." end
         log_fail(test_name, reason)
      end
      skills.remove_skill(pl_name, skill_name)
      skills.tests.log_test(test_name .. ": Cleaned up skill " .. skill_name)
   end)
end



local function test_expiring_entity_on_remove(pl_name)
   local test_name = "test_expiring_entity_on_remove"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")

   local entity_name = expiring_entity_name_on_remove
   local skill_name = test_prefix .. "expiring_entity_on_remove_skill"
   local duration = 1.0
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for expiring entity on_remove test")

   local on_remove_called = false
   local spawned_entity_ref = nil

   -- Patch the on_remove callback for this test run
   core.registered_entities[entity_name].on_remove = function(self)
      on_remove_called = true
      skills.tests.log_test(test_name .. ": on_remove callback called.")
      return true -- prevent automatic removal
   end

   -- Register skill that spawns the entity
   skills.register_skill(skill_name, {
      name = "Expiring Entity OnRemove Test Skill",
      loop_params = {duration = duration},
      on_start = function(self)
         local pos = self.player:get_pos()
         pos.y = pos.y + 1
         spawned_entity_ref = self:add_entity(pos, entity_name)
         if spawned_entity_ref then
            skills.tests.log_test(test_name .. ": Entity spawned by on_start.")
         else
            log_fail(test_name, "Failed to spawn entity using skill:add_entity.")
            self:stop("cancelled")
         end
         return spawned_entity_ref ~= nil
      end,
   })
   assert(skills.does_skill_exist(skill_name), "Expiring entity on_remove skill registration failed")
   skills.tests.log_test(test_name .. ": Registered skill " .. skill_name)

   assert(skills.unlock_skill(pl_name, skill_name), "Expiring entity on_remove skill unlock failed")
   local start_ok = skills.start_skill(pl_name, skill_name)
   assert(start_ok, "Starting expiring entity on_remove skill failed")
   skills.tests.log_test(test_name .. ": Skill started.")

   core.after(duration + 0.5, function()
      if not spawned_entity_ref then
         log_fail(test_name, "Entity failed to spawn initially.")
         skills.remove_skill(pl_name, skill_name)
         return
      end
      local luaentity = spawned_entity_ref:get_luaentity()
      if not on_remove_called then
         log_fail(test_name, "on_remove callback was not called after skill stopped.")
         if luaentity then luaentity.object:remove() end
         skills.remove_skill(pl_name, skill_name)
         return
      end
      if not luaentity then
         log_fail(test_name, "Entity was removed even though on_remove returned true.")
         skills.remove_skill(pl_name, skill_name)
         return
      end
      skills.tests.log_test(test_name .. ": Entity still exists after on_remove returned true (expected). Now removing manually.")
      luaentity.object:remove()
      core.after(0.2, function()
         local luaentity2 = spawned_entity_ref:get_luaentity()
         if not luaentity2 then
            log_pass(test_name)
         else
            log_fail(test_name, "Entity was not removed after manual remove.")
         end
         skills.remove_skill(pl_name, skill_name)
      end)
   end)
end



local function test_can_cast(pl_name)
   local test_name = "test_can_cast"
   local skill_name = test_prefix .. "can_cast"
   local allow_cast = true
   local cast_count = 0

   skills.register_skill(skill_name, {
      name = "Can Cast Test Skill",
      can_cast = function(self)
         return allow_cast
      end,
      cast = function(self)
         cast_count = cast_count + 1
         return true
      end,
   })
   assert(skills.does_skill_exist(skill_name), "Skill registration failed")
   assert(skills.unlock_skill(pl_name, skill_name), "Skill unlock failed")

   allow_cast = true
   local ok1 = skills.cast_skill(pl_name, skill_name)
   assert(ok1 and cast_count == 1, "Cast should succeed when can_cast returns true")

   allow_cast = false
   local ok2 = skills.cast_skill(pl_name, skill_name)
   assert(not ok2 and cast_count == 1, "Cast should fail when can_cast returns false and not increment cast_count")

   allow_cast = true
   local ok3 = skills.cast_skill(pl_name, skill_name)
   assert(ok3 and cast_count == 2, "Cast should succeed again when can_cast returns true and increment cast_count")

   skills.remove_skill(pl_name, skill_name)
   log_pass(test_name)
end



local function test_stop_on_death(pl_name)
   local test_name = "test_stop_on_death"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_true = test_prefix .. "sod_true"
   local skill_false = test_prefix .. "sod_false"
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for stop_on_death test")

   local cast_count_true = 0
   local cast_count_false = 0
   local casts_after_death_true = 0
   local casts_after_death_false = 0
   local player_died_true = false
   local player_died_false = false

   local cast_rate = 0.2
   local check_delay = cast_rate * 3 -- Increased delay for checking

   -- Part 1: stop_on_death = true (default)
   skills.register_skill(skill_true, {
      name = "StopOnDeath True",
      loop_params = {cast_rate = cast_rate, duration = 5.0},
      cast = function(self)
         local current_hp = self.player:get_hp()
         skills.tests.log_test(string.format("%s cast called (HP: %d, player_died_true: %s)",
            self.internal_name, current_hp, tostring(player_died_true)))
         cast_count_true = cast_count_true + 1
         if player_died_true then
            casts_after_death_true = casts_after_death_true + 1
         end
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, skill_true), "Unlock failed for sod_true")
   assert(skills.start_skill(pl_name, skill_true), "Start failed for sod_true")

   core.after(cast_rate * 1.5, function()
      local initial_casts_true = cast_count_true
      assert(initial_casts_true > 0, "Skill sod_true did not cast initially")

      player:set_hp(0) -- Simulate death
      player_died_true = true
      skills.tests.log_test(test_name .. ": Player HP set to 0 for sod_true test.")

      core.after(check_delay, function()
         local part1_passed = (casts_after_death_true == 0)
         if part1_passed then
            skills.tests.log_test(test_name .. ": stop_on_death=true correctly blocked casts after death.")
         else
            log_fail(test_name, string.format("Skill with stop_on_death=true continued casting after death (%d casts).", casts_after_death_true))
         end

         player:set_hp(20)                      -- Restore HP
         skills.stop_skill(pl_name, skill_true) -- Manually stop

         -- Part 2: stop_on_death = false
         skills.register_skill(skill_false, {
            name = "StopOnDeath False",
            loop_params = {cast_rate = cast_rate, duration = 5.0},
            stop_on_death = false,
            cast = function(self)
               local current_hp = self.player:get_hp()
               skills.tests.log_test(string.format("%s cast called (HP: %d, player_died_false: %s)",
                  self.internal_name, current_hp, tostring(player_died_false)))
               cast_count_false = cast_count_false + 1
               if player_died_false then
                  casts_after_death_false = casts_after_death_false + 1
               end
               return true
            end,
         })
         assert(skills.unlock_skill(pl_name, skill_false), "Unlock failed for sod_false")
         assert(skills.start_skill(pl_name, skill_false), "Start failed for sod_false")

         core.after(cast_rate * 1.5, function()
            local initial_casts_false = cast_count_false
            assert(initial_casts_false > 0, "Skill sod_false did not cast initially")

            player:set_hp(0) -- Simulate death again
            player_died_false = true
            skills.tests.log_test(test_name .. ": Player HP set to 0 for sod_false test.")

            core.after(check_delay, function()
               local part2_passed = (casts_after_death_false > 0)
               if part2_passed then
                  skills.tests.log_test(test_name .. ": stop_on_death=false correctly continued casting after death.")
               else
                  log_fail(test_name, "Skill with stop_on_death=false did not continue casting after death.")
               end

               player:set_hp(20)                       -- Restore HP
               skills.stop_skill(pl_name, skill_false) -- Manually stop

               if part1_passed and part2_passed then
                  log_pass(test_name)
               else
                  update_test_state(test_name, "failed")
               end

               skills.remove_skill(pl_name, skill_true)
               skills.remove_skill(pl_name, skill_false)
               skills.registered_skills[skill_true] = nil
               skills.registered_skills[skill_false] = nil
            end)
         end)
      end)
   end)
end



local function test_skill_blocking(pl_name)
   local test_name = "test_skill_blocking"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local blocker = test_prefix .. "blocker"
   local blockable = test_prefix .. "blockable"
   local unblockable = test_prefix .. "unblockable"
   local blockable_passive = test_prefix .. "blockable_passive"
   local duration = 1.2
   local casted = {}

   -- Part 1: Active skills
   skills.register_skill(blocker, {
      name = "Blocker",
      loop_params = {duration = duration},
      blocks_other_skills = true,
      cast = function(self)
         casted.blocker = true
         return true
      end,
   })
   skills.register_skill(blockable, {
      name = "Blockable",
      loop_params = {duration = duration},
      cast = function(self)
         casted.blockable = true
         return true
      end,
   })
   skills.register_skill(unblockable, {
      name = "Unblockable",
      loop_params = {duration = duration},
      can_be_blocked_by_other_skills = false,
      cast = function(self)
         casted.unblockable = true
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, blocker), "Unlock failed for blocker")
   assert(skills.unlock_skill(pl_name, blockable), "Unlock failed for blockable")
   assert(skills.unlock_skill(pl_name, unblockable), "Unlock failed for unblockable")
   assert(skills.start_skill(pl_name, blockable), "Start failed for blockable")
   assert(skills.start_skill(pl_name, unblockable), "Start failed for unblockable")
   core.after(0.2, function()
      assert(skills.start_skill(pl_name, blocker), "Start failed for blocker")
      core.after(0.2, function()
         local b = skills.get_skill(pl_name, blocker)
         local ba = skills.get_skill(pl_name, blockable)
         local ub = skills.get_skill(pl_name, unblockable)
         if b and b.is_active and (not ba or not ba.is_active) and ub and ub.is_active then
            skills.tests.log_test(test_name .. ": blocking interaction correct (active skills)")
         else
            log_fail(test_name, "Blocking logic failed (active skills)")
         end
         skills.stop_skill(pl_name, blocker)
         skills.stop_skill(pl_name, unblockable)
         core.after(0.3, function()
            -- Part 2: Passive skill
            skills.register_skill(blockable_passive, {
               name = "Blockable Passive",
               passive = true,
               loop_params = {cast_rate = 0.2},
               cast = function(self)
                  casted.blockable_passive = true
                  return true
               end,
            })
            assert(skills.unlock_skill(pl_name, blockable_passive), "Unlock failed for blockable_passive")
            core.after(0.2, function()
               assert(skills.start_skill(pl_name, blocker), "Start failed for blocker (passive test)")
               core.after(0.2, function()
                  local b = skills.get_skill(pl_name, blocker)
                  local bp = skills.get_skill(pl_name, blockable_passive)
                  if b and b.is_active and (not bp or not bp.is_active) then
                     skills.tests.log_test(test_name .. ": blocking interaction correct (passive)")
                     skills.stop_skill(pl_name, blocker)
                     core.after(0.3, function()
                        local bp2 = skills.get_skill(pl_name, blockable_passive)
                        if bp2 and bp2.is_active then
                           log_pass(test_name)
                        else
                           log_fail(test_name, "Passive skill did not auto-restart after blocker stopped.")
                        end
                        skills.remove_skill(pl_name, blocker)
                        skills.remove_skill(pl_name, blockable)
                        skills.remove_skill(pl_name, unblockable)
                        skills.remove_skill(pl_name, blockable_passive)
                     end)
                  else
                     log_fail(test_name, "Blocking logic failed (passive skills)")
                     skills.remove_skill(pl_name, blocker)
                     skills.remove_skill(pl_name, blockable)
                     skills.remove_skill(pl_name, unblockable)
                     skills.remove_skill(pl_name, blockable_passive)
                  end
               end)
            end)
         end)
      end)
   end)
end



local function test_sounds(pl_name)
   local test_name = "test_sounds"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "sound_test"
   local duration = 0.8
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for sound test")

   skills.register_skill(skill_name, {
      name = "Sound Test Skill",
      loop_params = {duration = duration},
      sounds = {
         cast = {name = "default_place_node"},
         start = {name = "default_dig"},
         stop = {name = "default_dig"},
         bgm = {name = "default_footstep", loop = true},
      },
      cast = function(self)
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, skill_name), "Unlock failed for sound_test")
   skills.cast_skill(pl_name, skill_name)
   assert(skills.start_skill(pl_name, skill_name), "Start failed for sound_test")
   core.after(duration + 0.3, function()
      log_pass(test_name)
      skills.remove_skill(pl_name, skill_name)
      skills.registered_skills[skill_name] = nil
   end)
end



local function test_hud(pl_name)
   local test_name = "test_hud"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "hud_test"
   local duration = 1.0
   local player = core.get_player_by_name(pl_name)
   assert(player, "Player object not found for HUD test")
   local hud_name = "skillz_test_hud_element"
   local hud_id = nil

   skills.register_skill(skill_name, {
      name = "HUD Test Skill",
      loop_params = {duration = duration},
      hud = {
         {
            hud_elem_type = "text",
            position = {x = 0.5, y = 0.1},
            text = "HUD TEST ACTIVE",
            number = 0xFFFFFF,
            name = hud_name,
         }
      },
      on_start = function(self)
         core.after(0.1, function()
            if self._hud and self._hud[hud_name] then
               hud_id = self._hud[hud_name]
               skills.tests.log_test(test_name .. ": HUD element added with ID: " .. hud_id)
            end
         end)
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, skill_name), "Unlock failed for hud_test")
   assert(skills.start_skill(pl_name, skill_name), "Start failed for hud_test")
   core.after(duration / 2, function()
      if not hud_id then
         log_fail(test_name, "HUD ID was not set correctly.")
         return
      end
      local hud_elem = player:hud_get(hud_id)
      if not hud_elem then
         log_fail(test_name, "HUD element not found mid-duration (ID: " .. hud_id .. ").")
      else
         skills.tests.log_test(test_name .. ": HUD element exists mid-duration.")
      end
   end)
   core.after(duration + 0.3, function()
      if not hud_id then
         log_fail(test_name, "HUD ID was not set correctly (post-duration).")
         return
      end
      local hud_elem = player:hud_get(hud_id)
      if hud_elem then
         log_fail(test_name, "HUD element still exists after skill stopped (ID: " .. hud_id .. ").")
      else
         skills.tests.log_test(test_name .. ": HUD element correctly removed after stop.")
         log_pass(test_name)
      end
      skills.remove_skill(pl_name, skill_name)
      skills.registered_skills[skill_name] = nil
   end)
end



local function test_particles(pl_name)
   local test_name = "test_particles"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "particles_test"
   local duration = 1.0
   local spawner_ids = {}

   skills.register_skill(skill_name, {
      name = "Particles Test Skill",
      loop_params = {duration = duration},
      attachments = {
         particles = {
            {
               amount = 20,
               time = 0.1,
               minpos = {x = -0.1, y = 0, z = -0.1},
               maxpos = {x = 0.1, y = 0.2, z = 0.1},
               minvel = {x = -0.2, y = 0.5, z = -0.2},
               maxvel = {x = 0.2, y = 1, z = 0.2},
               minacc = {x = 0, y = -2, z = 0},
               maxacc = {x = 0, y = -1, z = 0},
               minexptime = 0.5,
               maxexptime = 1.5,
               minsize = 1,
               maxsize = 3,
               texture = "fire_basic_flame.png",
            }
         }
      },
      on_start = function(self)
         core.after(0.1, function()
            if self._particles and #self._particles > 0 then
               spawner_ids = table.copy(self._particles)
               skills.tests.log_test(test_name .. ": Particle spawners added with IDs: " .. table.concat(spawner_ids, ", "))
            end
         end)
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, skill_name), "Unlock failed for particles_test")
   assert(skills.start_skill(pl_name, skill_name), "Start failed for particles_test")
   core.after(duration / 2, function()
      if #spawner_ids == 0 then
         log_fail(test_name, "Spawner IDs were not set correctly.")
         return
      end
      skills.tests.log_test(test_name .. ": Particle spawners assumed to exist mid-duration (cannot check directly in API).")
   end)
   core.after(duration + 0.3, function()
      if #spawner_ids == 0 then
         log_fail(test_name, "Spawner IDs were not set correctly (post-duration).")
         return
      end
      skills.tests.log_test(test_name .. ": Particle spawners assumed to be removed after stop (cannot check directly in API).")
      log_pass(test_name)
      skills.remove_skill(pl_name, skill_name)
      skills.registered_skills[skill_name] = nil
   end)
end



local function test_attach_entity(pl_name)
   local test_name = "test_attach_entity"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "attach_entity_test"
   local duration = 1.0

   -- Ensure the test entity is registered
   if not minetest.registered_entities[expiring_entity_name_for_test] then
      minetest.register_entity(expiring_entity_name_for_test, {
         initial_properties = {
            physical = false,
            visual = "sprite",
            textures = {"default_apple.png"},
            pointable = false,
            collide_with_objects = false,
            is_visible = true,
            spritediv = {x = 1, y = 1},
            visual_size = {x = 1, y = 1},
         },
         get_staticdata = function() return "" end,
      })
   end

   skills.register_skill(skill_name, {
      name = "Attach Entity Test Skill",
      loop_params = {duration = duration},
      attachments = {
         entities = {
            {
               name = expiring_entity_name_for_test,
               bone = "Arm_R",
               pos = {x = 0, y = 0, z = 0},
               rotation = {x = 0, y = 0, z = 0},
               scale = {x = 1, y = 1, z = 1},
               duration = duration / 2,
            }
         }
      },
      on_start = function(self)
         return true
      end,
   })
   assert(skills.unlock_skill(pl_name, skill_name), "Unlock failed for attach_entity_test")
   assert(skills.start_skill(pl_name, skill_name), "Start failed for attach_entity_test")
   core.after(duration / 2, function()
      skills.tests.log_test(test_name .. ": Entity should be attached (mid-duration)")
   end)
   core.after(duration + 0.3, function()
      skills.tests.log_test(test_name .. ": Entity should be detached (post-duration)")
      log_pass(test_name)
      skills.remove_skill(pl_name, skill_name)
      skills.registered_skills[skill_name] = nil
   end)
end



local function test_data_persistence(pl_name)
   local test_name = "test_data_persistence"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started")
   local skill_name = test_prefix .. "data_persist"
   local def_data = {counter = 0, text = "initial"}
   local mod_data = {counter = 10, text = "modified"}
   local pass = true

   skills.register_skill(skill_name, {
      name = "Data Persistence Test Skill",
      data = table.copy(def_data),
   })
   skills.unlock_skill(pl_name, skill_name)
   local skill1 = skills.get_skill(pl_name, skill_name)
   if not (skill1 and skill1.data and skill1.data.counter == 0 and skill1.data.text == "initial") then
      log_fail(test_name, "Initial data not set correctly")
      return
   end
   skill1.data.counter = mod_data.counter
   skill1.data.text = mod_data.text
   skills.update_db("just_once", nil, 0, {[pl_name] = skills.player_skills[pl_name]})
   core.after(0.2, function()
      skills.player_skills[pl_name] = nil
      skills.import_player_from_db(pl_name)
      local skill2 = skills.get_skill(pl_name, skill_name)
      if not skill2 then
         pass = false
      elseif skill2.data.counter ~= mod_data.counter or skill2.data.text ~= mod_data.text then
         pass = false
      end
      if pass then
         log_pass(test_name)
      else
         log_fail(test_name, "Data did not persist")
      end
   end)
end



local function test_dynamic_properties(pl_name)
   local test_name = "test_dynamic_properties"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name = test_prefix .. "dynamic_hud"
   local hud_name = "dynamic_test_hud"
   local hud_id
   local pass = true

   skills.register_skill(skill_name, {
      name = "Dynamic HUD Test Skill",
      loop_params = {duration = 1},
      data = {dynamic_text = "Initial Text"},
      hud = {
         {
            name = hud_name,
            type = "text",
            text = skills.dynamic_value(function(skill) return skill.data.dynamic_text or "Error" end),
            position = {x = 0.5, y = 0.1},
         },
      },
      on_start = function(self) return true end,
      on_stop = function(self) return true end,
   })
   local unlock_ok = skills.unlock_skill(pl_name, skill_name)
   if not unlock_ok then
      log_fail(test_name, "Unlocking dynamic HUD skill failed")
      return
   end
   local skill = skills.get_skill(pl_name, skill_name)
   if not skill then
      log_fail(test_name, "Failed to get skill instance for dynamic HUD test")
      return
   end
   skills.start_skill(pl_name, skill_name)

   core.after(0.2, function()
      hud_id = skill._hud and skill._hud[hud_name]
      if not hud_id then
         log_fail(test_name, "Failed to get HUD ID after starting skill")
         pass = false
         return
      end

      local player = core.get_player_by_name(pl_name)
      if not player then
         log_fail(test_name, "Player object not found during check")
         pass = false
         return
      end

      local hud_def = player:hud_get(hud_id)
      if not hud_def or hud_def.text ~= "Initial Text" then
         log_fail(test_name, "Initial dynamic text value is incorrect. Got: " .. (hud_def and hud_def.text or "nil"))
         pass = false
      end

      -- Change the dynamic value source (in skill.data)
      skill.data.dynamic_text = "Updated Text"
      skills.tests.log_test(test_name .. ": Updated skill.data.dynamic_text to 'Updated Text'.")

      -- Wait before updating HUD to avoid immediate update crash
      core.after(0.2, function()
         local new_text = "Error: Failed to evaluate dynamic text"
         if skill.hud and skill.hud[1] then
            new_text = skill.hud[1].text
            skills.tests.log_test(test_name .. ": Evaluated skill.hud[1].text = '" .. tostring(new_text) .. "'")
         else
            log_fail(test_name, "Skill instance lost its HUD definition unexpectedly.")
            pass = false
         end

         if pass and player and hud_id then
            if new_text == "Updated Text" then
               player:hud_change(hud_id, "text", new_text)
               skills.tests.log_test(test_name .. ": Manually called hud_change with evaluated text after delay.")
            else
               log_fail(test_name, "Dynamic text evaluation did not return the updated value. Got: '" .. tostring(new_text) .. "'")
               pass = false
            end
         elseif not pass then
            -- If already failed, don't try hud_change
         else
            log_fail(test_name, "Player or HUD ID became invalid before hud_change.")
            pass = false
         end

         core.after(0.2, function()
            if not pass then return end -- Skip if already failed

            local player_inner = core.get_player_by_name(pl_name)
            if not player_inner then
               log_fail(test_name, "Player object not found during second check")
               pass = false
               return
            end
            local current_hud_id = skill._hud and skill._hud[hud_name]
            if not current_hud_id then
               log_fail(test_name, "HUD ID disappeared during second check")
               pass = false
               return
            end

            local updated_hud_def = player_inner:hud_get(current_hud_id)
            if not updated_hud_def or updated_hud_def.text ~= "Updated Text" then
               local actual_text = updated_hud_def and updated_hud_def.text or "nil"
               log_fail(test_name, "Updated dynamic text value is incorrect after hud_change. Expected 'Updated Text', Got: '" .. actual_text .. "'")
               pass = false
            end

            core.after(1.0, function()
               if pass then
                  log_pass(test_name)
               end
               skills.remove_skill(pl_name, skill_name)
               skills.registered_skills[skill_name] = nil
               skills.tests.log_test(test_name .. ": Cleaned up skill " .. skill_name)
            end)
         end)
      end)
   end)
end

local function test_register_on_unlock(pl_name)
   local test_name = "test_register_on_unlock"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_name1 = test_prefix .. "unlock_target1"
   local skill_name2 = "_skillz_test_other:unlock_target2"
   local global_cb_flag = false
   local specific_cb_flag = false

   skills.register_on_unlock(function(skill) global_cb_flag = true end)
   skills.register_on_unlock(function(skill)
      if skill.internal_name:sub(1, #test_prefix) == test_prefix then specific_cb_flag = true end
   end, test_prefix:sub(1, -2))

   skills.register_skill(skill_name1, {name = "Unlock Target 1"})
   skills.register_skill(skill_name2, {name = "Unlock Target 2"})

   global_cb_flag = false
   specific_cb_flag = false
   skills.unlock_skill(pl_name, skill_name1)
   core.after(0.1, function()
      if not (global_cb_flag and specific_cb_flag) then
         log_fail(test_name, "Callbacks not triggered for matching prefix")
         return
      end
      global_cb_flag = false
      specific_cb_flag = false
      skills.unlock_skill(pl_name, skill_name2)
      core.after(0.1, function()
         if not (global_cb_flag and not specific_cb_flag) then
            log_fail(test_name, "Specific callback triggered for non-matching prefix")
         else
            log_pass(test_name)
         end
      end)
   end)
end



local function test_get_active_skills(pl_name)
   local test_name = "test_get_active_skills"
   update_test_state(test_name, "pending")
   skills.tests.log_test(test_name .. ": Started (async)")
   local skill_active1 = test_prefix .. "active1"
   local skill_active2 = test_prefix .. "active2"
   local skill_inactive = test_prefix .. "inactive1"

   skills.register_skill(skill_active1, {name = "Active 1", loop_params = {duration = 2}})
   skills.register_skill(skill_active2, {name = "Active 2", loop_params = {duration = 2}})
   skills.register_skill(skill_inactive, {name = "Inactive 1", loop_params = {duration = 0.1}})

   skills.unlock_skill(pl_name, skill_active1)
   skills.unlock_skill(pl_name, skill_active2)
   skills.unlock_skill(pl_name, skill_inactive)
   skills.start_skill(pl_name, skill_active1)
   skills.start_skill(pl_name, skill_active2)
   core.after(0.2, function()
      local active = skills.get_active_skills(pl_name, string.gsub(test_prefix, ":", ""))
      if not (active[skill_active1] and active[skill_active2] and not active[skill_inactive]) then
         log_fail(test_name, "Active skills not correct after start")
         return
      end
      skills.stop_skill(pl_name, skill_active1)
      core.after(0.2, function()
         local active_after = skills.get_active_skills(pl_name, string.gsub(test_prefix, ":", ""))
         if not (not active_after[skill_active1] and active_after[skill_active2]) then
            log_fail(test_name, "Active skills not correct after stop")
         else
            log_pass(test_name)
         end
      end)
   end)
end



local function test_can_start(pl_name)
   local test_name = "test_can_start"
   local skill_name_loop = "_skillz_test:can_start_loop"
   local skill_name_single = "_skillz_test:can_start_single"
   local cast_count = 0
   local start_count = 0

   local function fail(reason)
      log_fail(test_name, reason)
      skills.remove_skill(pl_name, skill_name_loop)
      skills.remove_skill(pl_name, skill_name_single)
      return false
   end

   skills.register_skill(skill_name_loop, {
      name = "Can Start Loop Skill",
      loop_params = {cast_rate = 0.1, duration = 0.3},
      allow_start = true,
      can_start = function(self)
         return self.allow_start
      end,
      cast = function(self)
         cast_count = cast_count + 1
         return true
      end,
      on_start = function(self)
         start_count = start_count + 1
      end,
   })
   skills.register_skill(skill_name_single, {
      name = "Can Start Single Skill",
      allow_start = true,
      can_start = function(self)
         return self.allow_start
      end,
      cast = function(self)
         cast_count = cast_count + 1
         return true
      end,
   })
   skills.unlock_skill(pl_name, skill_name_loop)
   skills.unlock_skill(pl_name, skill_name_single)

   local pl_skill_loop = skills.get_skill(pl_name, skill_name_loop)
   local pl_skill_single = skills.get_skill(pl_name, skill_name_single)

   pl_skill_loop.allow_start = false
   pl_skill_single.allow_start = false
   if skills.start_skill(pl_name, skill_name_loop) then
      return fail("Looped skill should not start if can_start returns false")
   end
   if skills.cast_skill(pl_name, skill_name_single) then
      return fail("Single skill should not cast if can_start returns false")
   end

   pl_skill_single.allow_start = true
   pl_skill_loop.allow_start = true
   if not skills.start_skill(pl_name, skill_name_loop) then
      return fail("Looped skill should start if can_start returns true")
   end
   if not skills.cast_skill(pl_name, skill_name_single) then
      return fail("Single skill should cast if can_start returns true")
   end

   skills.remove_skill(pl_name, skill_name_loop)
   skills.remove_skill(pl_name, skill_name_single)
   log_pass(test_name)
end



local test_cases = {
   {name = "test_can_cast",                  test_func = test_can_cast,                  delay = 0.2},
   {name = "test_stop_on_death",             test_func = test_stop_on_death,             delay = 2.0},
   {name = "test_sounds",                    test_func = test_sounds,                    delay = 1.5},
   {name = "test_hud",                       test_func = test_hud,                       delay = 1.5},
   {name = "test_particles",                 test_func = test_particles,                 delay = 1.5},
   {name = "test_attach_entity",             test_func = test_attach_entity,             delay = 2.0},
   {name = "test_skill_blocking",            test_func = test_skill_blocking,            delay = 2.5},
   {name = "test_registration_simple",       test_func = test_registration_simple,       delay = 0.2},
   {name = "test_unlock_remove",             test_func = test_unlock_remove,             delay = 0.2},
   {name = "test_cast_cooldown",             test_func = test_cast_cooldown,             delay = 2.0},
   {name = "test_loop_skill",                test_func = test_loop_skill,                delay = 2.2},
   {name = "test_loop_manual_stop",          test_func = test_loop_manual_stop,          delay = 1.8},
   {name = "test_passive_skill",             test_func = test_passive_skill,             delay = 0.5},
   {name = "test_enable_disable",            test_func = test_enable_disable,            delay = 0.2},
   {name = "test_monoids",                   test_func = test_monoids,                   delay = 3.5},
   {name = "test_physics",                   test_func = test_physics,                   delay = 1.0},
   {name = "test_skill_based_on",            test_func = test_skill_based_on,            delay = 0.2},
   {name = "test_expiring_entity",           test_func = test_expiring_entity,           delay = 2.6},
   {name = "test_expiring_entity_on_remove", test_func = test_expiring_entity_on_remove, delay = 2.0},
   {name = "test_data_persistence",          test_func = test_data_persistence,          delay = 0.5},
   {name = "test_dynamic_properties",        test_func = test_dynamic_properties,        delay = 1.0},
   {name = "test_register_on_unlock",        test_func = test_register_on_unlock,        delay = 0.5},
   {name = "test_get_active_skills",         test_func = test_get_active_skills,         delay = 1.0},
   {name = "test_can_start",                 test_func = test_can_start,                 delay = 0.5},
   {name = "cleanup_test_skills",            test_func = cleanup_test_skills,            delay = 0.5},
}

local current_test_index = 1
local is_test_running = false

local function run_next_test()
   if not is_test_running then
      skills.tests.log_test("Test run aborted.")
      return
   end

   if current_test_index > #test_cases then
      core.after(0.5, function()
         local final_passed = 0
         local final_failed = 0
         local final_skipped = 0
         local final_total = #test_cases

         skills.tests.log_test("--- Final State Check ---")
         for i, tc in ipairs(test_cases) do
            local name = tc.name
            local state = test_states[name]
            skills.tests.log_test(string.format("Test %d: %s -> State: %s", i, name, state or "nil"))

            if state == "passed" then
               final_passed = final_passed + 1
            elseif state == "failed" then
               final_failed = final_failed + 1
            elseif state == "skipped" then
               final_skipped = final_skipped + 1
            elseif state == "pending" then
               log_fail(name, "Async test did not complete.")
               final_failed = final_failed + 1
            elseif state == nil then
               log_fail(name, "Test did not produce a result (PASS/FAIL/SKIP/PENDING).")
               final_failed = final_failed + 1
            end
         end
         skills.tests.log_test("--- End Final State Check ---")

         skills.tests.log_test("-----------------------------------------")
         skills.tests.log_test(string.format("All tests finished. Results: %d/%d Passed, %d Failed, %d Skipped",
            final_passed, final_total, final_failed, final_skipped))
         skills.tests.log_test("-----------------------------------------")

         test_player_name = nil
         current_test_index = 1
         test_states = {}
         test_results = {passed = 0, failed = 0, skipped = 0, total = 0}
         is_test_running = false
      end)
      return
   end

   local case = test_cases[current_test_index]
   local test_name = case.name

   skills.tests.log_test("-----------------------------------------")
   skills.tests.log_test("Running test " .. current_test_index .. "/" .. #test_cases .. ": " .. test_name)

   local success, err = pcall(case.test_func, test_player_name)

   if not success then
      log_fail(test_name, "Error during test execution: " .. tostring(err))
   end

   current_test_index = current_test_index + 1
   core.after(case.delay, run_next_test)
end



function skills.tests.run(pl_name)
   if is_test_running then
      skills.tests.log_test("Tests are already running. Please wait for the current run to finish.")
      return
   end
   is_test_running = true

   local player = core.get_player_by_name(pl_name)
   if not player then
      skills.log("error", "Skills tests cannot run: Invalid player name '" .. pl_name .. "'")
      core.chat_send_player(pl_name, "[SKILLS TEST] Error: Invalid player name.")
      is_test_running = false
      return
   end

   skills.tests.log_test("=========================================")
   skills.tests.log_test("Starting skill test suite for player: " .. pl_name)
   skills.tests.log_test("=========================================")
   test_player_name = pl_name
   current_test_index = 1
   test_results = {passed = 0, failed = 0, skipped = 0, total = #test_cases}
   test_states = {}

   cleanup_test_skills()

   core.after(1.0, run_next_test)
end
