#!/usr/bin/env bash
set -Eeuo pipefail
trap cleanup SIGINT SIGTERM EXIT

script_dir=$(cd "$(dirname "${BASH_SOURCE[0]}")" &>/dev/null && pwd -P)

usage() {
  cat <<EOF
Usage: $(basename "${BASH_SOURCE[0]}") [-h] [-f/--failfast] [-k/--keep-server] [-d/--debug]
                          [-p/--port port_server] [--dependencies mod_dependencies]
                          [-c/--clients nb] [-i/--client-image image]
                          [--docker-cmd docker|podman] [-t/--tests all|simple|players]
EOF
  exit
}

cleanup() {
  if [ -n "${docker_compose_file:-}" ] && [ -n "${docker_cmd:-}" ] && [ -f "$PWD/${docker_compose_file}" ]; then
    ${docker_cmd} compose -f "$PWD/${docker_compose_file}" down >/dev/null 2>&1 || true
  fi
  [ -n "${docker_compose_file:-}" ] && rm -f "$PWD/${docker_compose_file}" 2>/dev/null || true
}

# Colors for output
RED=$'\033[0;31m'; GREEN=$'\033[0;32m'; CYAN=$'\033[0;36m'; BOLD=$'\033[1m'; RESET=$'\033[0m'

exists_in_list() { local L="$1" D="$2" V="$3"; for x in $(echo "$L" | tr "$D" " "); do [[ "$x" == "$V" ]] && return 0; done; return 1; }
isuint() { [[ "$1" =~ ^[0-9]+$ ]] ;}

# ---------- PRE-PASS: Normalize CRLF to LF if dos2unix is available ----------
if command -v dos2unix >/dev/null 2>&1; then
  find . -maxdepth 2 -type f \( -name "*.sh" -o -name "*.yml" -o -name "*.yaml" -o -name "*.json" -o -name "*.conf" -o -name "Dockerfile" \) -print0 \
    | xargs -0 -I{} sh -c 'dos2unix -q "{}" >/dev/null 2>&1 || true'
else
  # Check if there are files with CRLF line endings
  crlf_files=$(find . -maxdepth 2 -type f \( -name "*.sh" -o -name "*.yml" -o -name "*.yaml" -o -name "*.json" -o -name "*.conf" -o -name "Dockerfile" \) -exec file {} \; | grep -i "CRLF" || true)
  if [ -n "$crlf_files" ]; then
    echo -e "${BOLD}${RED}WARNING:${RESET} ${CYAN}Windows-style line endings (CRLF) detected in some files.${RESET}"
    echo -e "${CYAN}This may cause issues when running in Docker/Linux environments.${RESET}"
    echo -e "${BOLD}Please install dos2unix:${RESET} ${GREEN}apk add dos2unix${RESET}"
    echo
  fi
fi
# ---------------------------------------------------------------------------

docker_cmd=
docker_build_cmd="build"
client_image=
client_nb=${CLIENT_NB:-1}
failfast="false"
stop_server="true"
server_port="30000"
show_debug="false"
run_tests="all"
dependencies="${DEPENDENCIES:-}"
docker_compose_file=""
debug=false

parse_params() {
  args=$(getopt -a -o fkhddc:p:t:i: --long failfast,keep-server,help,debug,clients:,client-image:,port:,docker-cmd:,tests:,dependencies: -- "$@") || usage
  eval set -- "$args"
  while [[ $# -gt 0 ]]; do
    case $1 in
      -f|--failfast) failfast="true"; shift;;
      -k|--keep-server) stop_server="false"; shift;;
      -h|--help) usage;;
      -d|--debug) debug=true; shift;;
      --docker-cmd) docker_cmd=$2; shift 2;;
      -p|--port) server_port=$2; shift 2;;
      -t|--tests) run_tests=$2; shift 2;;
      -i|--client-image) client_image=$2; shift 2;;
      -c|--clients) client_nb=$2; shift 2;;
      --dependencies) dependencies=$2; shift 2;;
      --) shift; break;;
      *) usage;;
    esac
  done
  [[ "$debug" == "true" ]] && set -x
  [[ -z "${docker_cmd:-}" ]] && docker_cmd="docker"
  [[ "$docker_cmd" == "docker" ]] && docker_build_cmd="buildx build"
  exists_in_list "docker podman" " " "$docker_cmd" || { echo "docker command must be in docker or podman"; exit 1; }
  exists_in_list "all simple players" " " "$run_tests" || { echo "run tests (-t) must be in (all,simple,players)"; exit 1; }
  isuint "$client_nb" || { echo "Number of clients must be an integer"; exit 1; }
}
parse_params "$@"

[ -f mod.conf ] || [ -f modpack.conf ] || { echo "Run this from the mod root folder."; exit 1; }

if [ "$docker_cmd" == "docker" ]; then
  command -v docker >/dev/null || { echo "Docker is not installed."; exit 1; }
else
  command -v podman >/dev/null || { echo "Podman is not installed."; exit 1; }
fi

: "${CLIENT_IMAGE:=${client_image:-registry.apps.education.fr/iri/minetest/docker_test_harness/client:latest}}"
$docker_cmd manifest inspect "$CLIENT_IMAGE" >/dev/null 2>&1 || { echo "CLIENT IMAGE $CLIENT_IMAGE does not exist"; exit 1; }

CURRENT_MOD=""
ADDITIONAL_MODS=""
ADDITIONAL_FILES="[]"
MINETEST_ADD_CONF_RAW=""
if [ -f "$PWD/.mod_env.json" ]; then
  dependencies="$(jq -cr '.dependencies // ""' "$PWD/.mod_env.json")"
  CURRENT_MOD="$(jq -cr '.current_mod // ""' "$PWD/.mod_env.json")"
  client_nb="$(jq -cr '.client_nb // "'"$client_nb"'"' "$PWD/.mod_env.json")"
  ADDITIONAL_MODS="$(jq -cr '.additional_mods // ""' "$PWD/.mod_env.json")"
  ADDITIONAL_FILES="$(jq -cr '.additional_files // []' "$PWD/.mod_env.json")"
  MINETEST_ADD_CONF_RAW="$(jq -cr '.minetest_add_conf // ""' "$PWD/.mod_env.json")"
fi

MTT_CONF="mtt_enable = true\nmtt_enable_benchmarks = false\n"
MINETEST_ADD_CONF="$(printf "%b%b" "$MINETEST_ADD_CONF_RAW" "$MTT_CONF")"

export SHOW_DEBUG=${show_debug:-false}
export RUN_TESTS=${run_tests}
export STOP_SERVER="${stop_server:-true}"
export FAILFAST="${failfast:-false}"
export DEPENDENCIES="${dependencies}"
export CLIENT_IMAGE="${CLIENT_IMAGE}"
export CURRENT_MOD="${CURRENT_MOD}"
export ADDITIONAL_MODS="${ADDITIONAL_MODS}"
export ADDITIONAL_FILES="${ADDITIONAL_FILES}"
export CLIENT_NB="${client_nb}"
export MINETEST_ADD_CONF="${MINETEST_ADD_CONF}"

mkdir -p .util
docker_compose_file=$(mktemp --suffix=.yaml docker-composeXXXXXX --tmpdir=.util)
cat "${script_dir}/docker-compose.yaml" > "$PWD/$docker_compose_file"

for (( clienti=1; clienti<=CLIENT_NB; clienti++ )); do
  cat <<EOF >> "$PWD/$docker_compose_file"
  client${clienti}:
    image: \${CLIENT_IMAGE}
    restart: on-failure:10
    depends_on:
      - "server"
    environment:
      SERVER: server
      PORT: 30000
      PLAYERNAME: player${clienti}
      RANDOM_INPUT: 0
      PASSWORD: test
EOF
done

set +e
${docker_cmd} compose -f "$PWD/${docker_compose_file}" down
${docker_cmd} compose -f "$PWD/${docker_compose_file}" pull
${docker_cmd} compose -f "$PWD/${docker_compose_file}" build --pull
${docker_cmd} compose -f "$PWD/${docker_compose_file}" up --force-recreate --exit-code-from server --abort-on-container-exit
exit_code=$?

all_logs="$(${docker_cmd} compose -f "$PWD/${docker_compose_file}" logs 2>/dev/null || true)"

if echo "$all_logs" | grep -Fqi "Access denied. Reason: success"; then
  exit_code=0
fi

echo
echo "=========================================================="
if [ "$exit_code" -eq 0 ]; then
  echo -e "${BOLD}${GREEN}TEST RESULT: PASS${RESET}  ${CYAN}(All tests completed successfully)${RESET}"
else
  echo -e "${BOLD}${RED}TEST RESULT: FAILED${RESET}  ${CYAN}(See server logs above for details)${RESET}"
fi
echo "=========================================================="
echo

exit "$exit_code"
