dynamic_values = {
	-- key: skill_name.subtable1.subtable2.etc.key, values: dynamic value
}



skills.dynamic_properties_metatable = {
	-- filtering access to the table, getting dynamic values from an external table
	__index = function(t, key)
		local id = rawget(t, "_id")
		local skill = rawget(t, "_skill")

		if not id then
			skills.log("error", "skills.dynamic_properties_metatable.__index called without _id on table " .. dump(t), true)
		elseif not skill then
			skills.log("error", "skills.dynamic_properties_metatable.__index called without _skill on table " .. dump(t), true)
		end

		-- Check for dynamic value using the full hierarchical ID
		if key then
			local full_id = id .. "." .. key
			if dynamic_values[full_id] then
				return skills.get_value(skill, dynamic_values[full_id])
			end
		end

		local value = rawget(t, key)
		if value ~= nil then
			return value
		end

		-- Fallback to original metatable for skill methods
		local original_metatable = rawget(t, "_original_metatable")
		if original_metatable and original_metatable.__index then
			if type(original_metatable.__index) == "function" then
				return original_metatable.__index(t, key)
			elseif type(original_metatable.__index) == "table" then
				return original_metatable.__index[key]
			end
		end

		return nil
	end,

	-- never storing dynamic values, but storing them in a separate table instead
	__newindex = function(t, key, value)
		if skills.is_dynamic_value(value) then
			local id = rawget(t, "_id")
			if key then
				local full_id = id .. "." .. key
				dynamic_values[full_id] = value
			end
			rawset(t, key, nil)
		else
			rawset(t, key, value)
		end
	end
}


---Check if a table contains any dynamic values (recursively)
---@param t table
---@return boolean
function skills.does_table_contain_dynamic_values(t)
	for key, val in pairs(t) do
		if skills.is_dynamic_value(val) then return true end
		if type(val) == "table" then
			if skills.does_table_contain_dynamic_values(val) then return true end
		end
	end
	return false
end



---Convert a table to use dynamic properties metatable for runtime evaluation
---@param skill PlayerSkill
---@param id string Property ID
---@param table table
---@param recursive? boolean
---@param prev_table? table
---@return table
function skills.make_dynamic_properties_table(skill, id, table, recursive, prev_table)
	if type(table) ~= "table" or not skills.does_table_contain_dynamic_values(table) then return table end

	table._skill = skill
	-- For root skill object, use just the skill name instead of adding another level
	if id == "skill" and not prev_table then
		table._id = skill.internal_name
		-- Store original metatable for fallback
		table._original_metatable = getmetatable(table)
	else
		local prev_id = (prev_table and prev_table._id) or skill.internal_name
		table._id = prev_id .. "." .. id
	end

	-- Track which keys have dynamic values before they get moved to external storage
	local dynamic_keys = {}
	for key, value in pairs(table) do
		if skills.is_dynamic_value(value) then
			dynamic_keys[key] = true
		end
	end
	rawset(table, "_dynamic_keys", dynamic_keys)

	if recursive == nil then recursive = true end
	if recursive then
		for key, value in pairs(table) do
			-- ASSUMING: there are no other circular references other than _skill
			if type(value) == "table" and key ~= "_skill" then
				skills.make_dynamic_properties_table(skill, key, value, true, table)
			end
		end
	end

	-- reconstructing table, storing dynamic values in a separate table
	for key, value in pairs(table) do
		skills.dynamic_properties_metatable.__newindex(table, key, value)
	end

	return setmetatable(table, skills.dynamic_properties_metatable)
end



---Create a dynamic value that will be evaluated at runtime
---@param func fun(skill: PlayerSkill, ...: any): any
---@return DynamicValue
function skills.dynamic_value(func)
	return {
		dynamic_value = true,
		get_value = func
	}
end



---Get value from dynamic value or return as-is
---@param skill PlayerSkill
---@param val any|DynamicValue
---@param ... any Varargs
---@return any
function skills.get_value(skill, val, ...)
	if val.get_value then
		-- Pass skill first, then the stored varargs from the skill
		local stored_varargs = skill.__varargs or {}
		return val.get_value(skill, unpack(stored_varargs))
	else
		return val
	end
end



---Check if a value is a dynamic value
---@param val any
---@return boolean
function skills.is_dynamic_value(val)
	return type(val) == "table" and val.dynamic_value
end



---Iterate over a dynamic properties table, resolving all dynamic values
---Use this instead of pairs() when you need to iterate and get resolved values
---@param tbl table Table with dynamic properties metatable
---@return table resolved A new table with all dynamic values resolved
function skills.iterate_dynamic_table(tbl)
	local id = rawget(tbl, "_id")
	local skill = rawget(tbl, "_skill")
	local dynamic_keys = rawget(tbl, "_dynamic_keys") or {}

	local result = {}

	-- Copy raw values (excluding internal fields)
	for key, value in pairs(tbl) do
		if key ~= "_id" and key ~= "_skill" and key ~= "_original_metatable" and key ~= "_dynamic_keys" then
			result[key] = value
		end
	end

	-- Resolve dynamic values
	for key in pairs(dynamic_keys) do
		local full_id = id .. "." .. key
		if dynamic_values[full_id] then
			result[key] = skills.get_value(skill, dynamic_values[full_id])
		end
	end

	return result
end



---Cleanup dynamic properties for a skill
---@param skill_internal_name SkillInternalName
function skills.cleanup_direct_dynamic_properties(skill_internal_name)
	if dynamic_values[skill_internal_name] then
		dynamic_values[skill_internal_name] = nil
	end
end
