local expire_entities = {} -- "entity" = true



---Register an entity that expires when skill stops
---@param name string Entity name
---@param def table Entity definition
function skills.register_expiring_entity(name, def)
	core.register_entity(name, def)
	skills.set_expiring_entity(name)
end



---Add entity attached to a skill (entity expires when skill stops)
---@param skill PlayerSkill
---@param pos vector
---@param name string Entity name
---@return ObjectRef|false
function skills.add_entity(skill, pos, name)
	if expire_entities[name] then
		return core.add_entity(pos, name, core.serialize({pl_name = skill.pl_name, skill_name = skill.internal_name}))
	else
		skills.log("error", "Tried to spawn an entity that isn't expiring: " .. name, true)
		return false
	end
end



---Mark an entity as expiring (will be removed when skill stops)
---@param name string Entity name
function skills.set_expiring_entity(name)
	local on_step = core.registered_entities[name].on_step or function(_, _, _) end
	local on_activate = core.registered_entities[name].on_activate or function(_, _, _) end

	local function remove(self)
		local handled = false
		if self.on_remove then handled = self:on_remove() end
		if not handled then self.object:remove() end
	end

	local function schedule_remove(self)
		-- Delay removal so any outer wrappers finish their teardown safely
		core.after(0, function()
			if self.object and self.object:get_pos() then
				remove(self)
			end
		end)
	end

	-- Overriding the entity's callbacks & properties
	if not expire_entities[name] then
		core.registered_entities[name].initial_properties = core.registered_entities[name].initial_properties or {}
		core.registered_entities[name].initial_properties.static_save = false
		core.registered_entities[name].pl_name = ""


		-----------------
		-- ON ACTIVATE --
		-----------------

		core.registered_entities[name].on_activate = function(self, staticdata, dtime_s)
			local decoded = core.deserialize(staticdata)
			local skill_data

			if type(decoded) == "table" and decoded.original_staticdata then
				-- Some mods (e.g. mtobjid ) re-wrap staticdata; restore the original payload if present
				skill_data = core.deserialize(decoded.original_staticdata) or {}
			else
				skill_data = decoded or {}
			end

			on_activate(self, staticdata, dtime_s)

			local pl_name = skill_data.pl_name
			local skill_name = skill_data.skill_name

			if type(pl_name) ~= "string" or pl_name == "" or type(skill_name) ~= "string" or skill_name == "" then
				schedule_remove(self)
				return
			end

			local skill = pl_name:get_skill(skill_name)
			if not skill or not skill.is_active then
				schedule_remove(self)
				return
			end

			self.pl_name = pl_name
			self.skill = skill
			self.player = skill.player
		end


		-------------
		-- ON STEP --
		-------------

		core.registered_entities[name].on_step = function(self, dtime, moveresult)
			-- Scheduled for removal
			if not self.pl_name or self.pl_name == "" then
				return
			end

			-- if spawned by a skill, remove the entity if the skill has stopped
			if self.skill and not self.skill.is_active then
				remove(self)
				return
			end

			on_step(self, dtime, moveresult)
		end

		expire_entities[name] = true
	end
end



---Attach an expiring entity to player using skill attachment config
---@param skill PlayerSkill
---@param def EntityAttachment
---@return boolean|nil
function skills.attach_expiring_entity(skill, def)
	local pos = def.pos
	local name = def.name
	local bone = def.bone or ""
	local rotation = def.rotation or {x = 0, y = 0, z = 0}
	local forced_visible = def.forced_visible or false

	local entity = skill:add_entity(pos, name)
	if not entity then
		skills.log("error", skill.internal_name .. " skill tried to attach an entity that isn't expiring: " .. name, true)
		return false
	end
	entity:set_attach(skill.player, bone, pos, rotation, forced_visible)
end
