---Send error message to player with sound
---@param pl_name string Player name
---@param msg string Error message
function skills.error(pl_name, msg)
	core.chat_send_player(pl_name, core.colorize("#f47e1b", "[!] " .. msg))
	core.sound_play("skills_error", {to_player = pl_name})
end



---Send message to player
---@param pl_name string Player name
---@param msg string Message
function skills.print(pl_name, msg)
	core.chat_send_player(pl_name, msg)
end



---@param level string none, error, warning, action, info (default), verbose
function skills.log(level, msg, stacktrace)
	if stacktrace then
		stacktrace = "\n" .. debug.traceback()
	else
		stacktrace = ""
	end

	if not msg then
		msg = level
		level = "info"
	end
	core.log(level, "[SKILLS] " .. msg .. stacktrace)
end



---Assert condition with skills prefix
---@param condition any
---@param msg string Error message
function skills.assert(condition, msg)
	assert(condition, "[SKILLS] " .. msg)
end



---Deep merge two tables, with new values overriding original (@@nil removes keys)
---@param original? table
---@param new table
---@return table Merged table
function skills.override_params(original, new)
	local output = table.copy(original or {})

	for key, new_value in pairs(new) do
		if new_value == "@@nil" then
			-- Directly set to nil, don't recurse
			output[key] = nil
		elseif type(new_value) == "table" and output[key] and type(output[key]) == "table" then
			output[key] = skills.override_params(output[key], new_value)
		elseif type(new_value) ~= "function" then
			output[key] = new_value
		elseif not original[key] then
			output[key] = new_value
		end
	end

	return output
end



---Normalize a value to a token list (retrocompatibility)
---@param value string|string[]|table|nil
---@return string[]
local function normalize_token_list(value)
	if not value then
		return {}
	end

	-- Single string -> wrap in array
	if type(value) == "string" then
		return {value}
	end

	-- Not a table -> empty array
	if type(value) ~= "table" then
		return {}
	end

	-- Try to copy as array (numeric keys)
	local tokens = {}
	for index, token in ipairs(value) do
		tokens[index] = token
	end

	-- If we got array elements, return them
	if #tokens > 0 then
		return tokens
	end

	-- Otherwise treat as dictionary: collect string keys or string values
	for key, val in pairs(value) do
		if type(key) == "string" and val then
			-- {["mod:skill"] = true} -> add key
			tokens[#tokens + 1] = key
		elseif type(val) == "string" then
			-- {foo = "mod:skill"} -> add value
			tokens[#tokens + 1] = val
		end
	end

	return tokens
end


---Check if token matches item name (prefix or full name)
---@param lowercase_item_name string
---@param item_prefix string|nil
---@param token string
---@return boolean
local function token_matches_item(lowercase_item_name, item_prefix, token)
	if type(token) ~= "string" then return false end

	local normalized_token = token:lower()

	if normalized_token:find(":", 1, true) then
		return lowercase_item_name == normalized_token
	end

	return item_prefix ~= nil and item_prefix == normalized_token
end


---Generic blocking logic for skills and states
---@param item_name SkillInternalName
---@param blocker_skill PlayerSkill
---@param blocker_field string "blocks_other_skills" or "blocks_other_states"
---@return boolean
function skills.should_logic_be_blocked(item_name, blocker_skill, blocker_field)
	local item_def = skills.get_def(item_name)
	if not item_def then return false end

	-- Get blocking definition from the blocker skill
	local blocker_def = blocker_skill[blocker_field]
	if not blocker_def or type(blocker_def) ~= "table" then
		return false
	end

	-- Normalize names for comparison
	item_name = item_name:lower()
	local item_prefix = item_name:match("^([^:]+):")

	local blocker_name = blocker_skill.internal_name
	if type(blocker_name) == "string" then
		blocker_name = blocker_name:lower()
	end

	-- Check if blocking self
	if blocker_name and item_name == blocker_name then
		return false
	end

	-- Determine blocking mode and act accordingly
	local mode = blocker_def.mode
	if type(mode) ~= "string" then
		return false
	end
	mode = mode:lower()

	if mode == "all" then
		return true
	elseif mode == "whitelist" then
		local allowed_tokens = normalize_token_list(blocker_def.allowed)

		if next(allowed_tokens) == nil then
			return true
		end

		for _, allowed_item in ipairs(allowed_tokens) do
			if token_matches_item(item_name, item_prefix, allowed_item) then
				return false -- item is allowed
			end
		end

		return true
	elseif mode == "blacklist" then
		local blocked_tokens = normalize_token_list(blocker_def.blocked)

		for _, blocked_item in ipairs(blocked_tokens) do
			if token_matches_item(item_name, item_prefix, blocked_item) then
				return true -- item is blocked
			end
		end

		return false
	end

	return false
end



---Check if a skill should be blocked by another skill
---@param skill_name SkillInternalName
---@param blocker_skill PlayerSkill
---@return boolean
function skills.should_skill_be_blocked(skill_name, blocker_skill)
	return skills.should_logic_be_blocked(skill_name, blocker_skill, "blocks_other_skills")
end



---Block and stop other skills based on blocking config
---@param skill PlayerSkill
function skills.block_other_skills(skill)
	local pl_skills = skills.player_skills[skill.pl_name]
	if not pl_skills then return end

	for skill_name, skill_instance in pairs(pl_skills) do
		if not skill_instance.is_state and skill_instance.is_active then
			if skills.should_skill_be_blocked(skill_name, skill) then
				skill.pl_name:stop_skill(skill_name)
			end
		end
	end
end



---Cast all passive skills for a player
---@param pl_name string Player name
function skills.cast_passive_skills(pl_name)
	for name, def in pairs(skills.get_unlocked_skills(pl_name)) do
		if def.passive and def.data.__enabled then
			pl_name:start_skill(name)
		end
	end
end



---Get active skills ordered by start timestamp (bottom to top)
---@param pl_name string Player name
---@param prefix? string Optional prefix filter
---@param filter_predicate? fun(skill: PlayerSkill): boolean Optional filter function
---@return PlayerSkill[] Ordered by start timestamp
function skills.get_active_skills_stack(pl_name, prefix, filter_predicate)
	local active_skills = skills.get_active_skills(pl_name, prefix)
	local final_skills = {}

	-- sort by skill.__last_start_timestamp [1: earliest, 2: latest]
	table.sort(active_skills, function(a, b)
		return a.__last_start_timestamp < b.__last_start_timestamp
	end)

	if filter_predicate then
		for name, skill in pairs(active_skills) do
			if filter_predicate(skill) then
				table.insert(final_skills, skill)
			end
		end
	else
		final_skills = active_skills
	end

	return final_skills
end



---Check if a state should be blocked by a skill
---@param state_name SkillInternalName
---@param blocker_skill PlayerSkill
---@return boolean
function skills.should_state_be_blocked(state_name, blocker_skill)
	return skills.should_logic_be_blocked(state_name, blocker_skill, "blocks_other_states")
end



---Block and stop other states based on blocking config
---@param skill PlayerSkill
function skills.block_other_states(skill)
	-- Get all active states for the player
	local pl_skills = skills.player_skills[skill.pl_name]
	if not pl_skills then return end

	for state_name, state_instance in pairs(pl_skills) do
		if state_instance.is_state and state_instance.is_active then
			if skills.should_state_be_blocked(state_name, skill) then
				-- Stop the state but don't remove it entirely
				state_instance:stop()
			end
		end
	end
end
