--- Main Flow GUI for the Skywars editor.
-- Provides tabbed interface for Treasures, Kits, and Templates management.
-- Per flow_docs.txt: use gui = flow.widgets, return gui.VBox from build function.

local gui = flow.widgets
local S = skywars.T

-- Tab constants
local TAB_TREASURES = "treasures"
local TAB_KITS = "kits"
local TAB_TEMPLATES = "templates"

-- Cache of all registered item names for search
local all_item_names = nil



--- Get all registered item names (cached after first call).
--- Includes nodes, craftitems, and tools.
local function get_all_item_names()
    if all_item_names then return all_item_names end

    all_item_names = {}
    -- core.registered_items includes all: nodes, craftitems, tools
    for name, def in pairs(core.registered_items) do
        -- Skip empty, internal, and hidden items
        if
            name ~= "" and name ~= "unknown" and name ~= "ignore"
            and not name:match("^:")
            and not (def.groups and def.groups.not_in_creative_inventory == 1)
        then
            table.insert(all_item_names, name)
        end
    end

    table.sort(all_item_names)

    return all_item_names
end



--- Search items by partial name match.
--- Prioritizes: 1) exact match after ':', 2) starts with query after ':', 3) contains query
---@param query string
---@return string[]
local function search_items(query)
    local q = query:lower()
    local exact = {}
    local starts = {}
    local contains = {}

    for _, name in ipairs(get_all_item_names()) do
        local lower_name = name:lower()
        local short_name = lower_name:match(":(.+)$") or lower_name

        if short_name == q then
            table.insert(exact, name)
        elseif short_name:find("^" .. q) then
            table.insert(starts, name)
        elseif lower_name:find(q, 1, true) then
            table.insert(contains, name)
        end
    end

    local results = {}
    for _, name in ipairs(exact) do
        table.insert(results, name)
    end
    for _, name in ipairs(starts) do
        table.insert(results, name)
    end
    for _, name in ipairs(contains) do
        table.insert(results, name)
    end

    return results
end



-- Default state values for each tab
local DEFAULTS = {
    treasure = {name = "", count = 1, preciousness = 1, rarity = 5.0},
    kit = {name = "", texture = ""},
    template = {name = "", min_p = 1, max_p = 3, min_t = 1, max_t = 5, color = "brown"},
}

-- Color options for dropdown
local COLOR_OPTIONS = {}
for color_key in pairs(skywars_settings.chest_placeholder_colors) do
    table.insert(COLOR_OPTIONS, color_key)
end
table.sort(COLOR_OPTIONS)


--- Runs all sanitization checks on editor open.
--- Fixes corrupted data and logs changes.
---@param arena table
---@param player_name string
local function sanitize_editor_data(arena, player_name)
    local messages = {}

    if skywars.sanitize_arena_treasures(arena) then
        arena_lib.change_arena_property(player_name, "skywars", arena.name, "treasures", arena.treasures)
        table.insert(messages, S("Treasures list was corrupted and has been repaired."))
    end

    if skywars.sanitize_all_kits() then
        table.insert(messages, S("Kits data was corrupted and has been repaired."))
    end

    if skywars.sanitize_arena_templates(arena) then
        arena_lib.change_arena_property(player_name, "skywars", arena.name, "chest_templates", arena.chest_templates)
        table.insert(messages, S("Chest templates were corrupted and have been repaired."))
    end

    if skywars.sanitize_arena_chests(arena) then
        arena_lib.change_arena_property(player_name, "skywars", arena.name, "chests", arena.chests)
        table.insert(messages, S("Chests had orphan template references and have been repaired."))
    end

    for _, msg in ipairs(messages) do
        skywars.print_msg(player_name, msg)
    end
end


--- Reset treasure edit state to defaults.
local function reset_treasure_state(ctx)
    ctx.selected_treasure_id = nil
    ctx.treasure_edit = table.copy(DEFAULTS.treasure)
    ctx.item_search_query = ""
    ctx.show_search_results = false
end

--- Reset kit edit state to defaults.
local function reset_kit_state(ctx)
    ctx.selected_kit = nil
    ctx.kit_edit = table.copy(DEFAULTS.kit)
    ctx.kit_show_settings = nil
    ctx.kit_icon_query = ""
    ctx.kit_item_query = ""
    ctx.kit_item_count = 1
end

--- Reset template edit state to defaults.
local function reset_template_state(ctx)
    ctx.selected_template_id = nil
    ctx.template_edit = table.copy(DEFAULTS.template)
    ctx.form.template_name = ""
    ctx.form.template_min_p = tostring(DEFAULTS.template.min_p)
    ctx.form.template_max_p = tostring(DEFAULTS.template.max_p)
    ctx.form.template_min_t = tostring(DEFAULTS.template.min_t)
    ctx.form.template_max_t = tostring(DEFAULTS.template.max_t)
end



--- Creates a tab button with active state styling.
local function tab_button(tab_id, label, ctx)
    local is_active = ctx.active_tab == tab_id
    return gui.Button {
        w = 2.5,
        label = label,
        name = "tab_" .. tab_id,
        style = is_active and {
            bgcolor = "#446688",
            textcolor = "#FFFFFF",
        } or nil,
        on_event = function(player, ctx)
            if ctx.active_tab ~= tab_id then
                ctx.active_tab = tab_id
                return true
            end
            return false
        end,
    }
end


--- Build the Treasures tab content.
local function build_treasures_tab(arena, ctx)
    local treasure_items = {}

    -- Sort by preciousness (low to high), then by rarity, then by name
    local sorted_treasures = {}
    for _, t in ipairs(arena.treasures or {}) do
        table.insert(sorted_treasures, t)
    end
    table.sort(sorted_treasures, function(a, b)
        if a.preciousness ~= b.preciousness then
            return a.preciousness < b.preciousness
        end
        if a.rarity ~= b.rarity then
            return a.rarity < b.rarity
        end
        return a.name < b.name
    end)

    for i, treasure in ipairs(sorted_treasures) do
        local t = treasure -- capture for closure
        local is_selected = ctx.selected_treasure_id == t.id
        local bg_color = is_selected and "#335577" or "#444444"
        local short_name = t.name:gsub("^[^:]+:", "")

        local item_exists = core.registered_items[t.name] ~= nil
        local display_name = item_exists and short_name or (S("[NOT FOUND] ") .. short_name)

        table.insert(treasure_items, gui.HBox {
            w = 5.3,
            spacing = 0.2,
            gui.Stack {
                min_w = 0.8,
                min_h = 0.8,
                gui.Box {w = 0.8, h = 0.8, color = item_exists and bg_color or "#883333"},
                item_exists and gui.ItemImage {w = 0.7, h = 0.7, item_name = t.name, padding = 0.05}
                or gui.Label {label = "?", w = 0.7, h = 0.7, align_h = "centre", align_v = "centre", style = {font = "bold", textcolor = "#FF6666"}},
            },
            gui.VBox {
                spacing = 0,
                w = 3.2,
                gui.Label {label = display_name, style = item_exists and {} or {textcolor = "#FF6666"}},
                gui.HBox {
                    spacing = 0.3,
                    gui.Label {label = "x" .. t.count, style = {font_size = "*0.9"}},
                    gui.Label {label = "P:" .. t.preciousness, style = {font_size = "*0.9"}},
                    gui.Label {label = "R:" .. string.format("%.1f", t.rarity), style = {font_size = "*0.9"}},
                },
            },
            gui.Button {
                w = 1,
                h = 0.8,
                label = is_selected and "*" or ">",
                name = "treasure_" .. t.id,
                style = is_selected and {bgcolor = "#446688"} or nil,
                on_event = function(player, ctx)
                    if ctx.selected_treasure_id == t.id then
                        reset_treasure_state(ctx)
                        return true
                    end
                    ctx.selected_treasure_id = t.id
                    ctx.treasure_edit = {
                        name = t.name,
                        count = t.count,
                        preciousness = t.preciousness,
                        rarity = t.rarity,
                    }
                    ctx.form.treasure_name = t.name
                    ctx.form.treasure_count = tostring(t.count)
                    ctx.form.treasure_preciousness = tostring(t.preciousness)
                    ctx.form.treasure_rarity = tostring(t.rarity)
                    return true
                end,
            },
        })
    end

    if #treasure_items == 0 then
        table.insert(treasure_items, gui.Label {label = S("No treasures defined.")})
    end

    -- Item search state
    local search_query = ctx.item_search_query or ""
    local show_search_results = ctx.show_search_results and search_query ~= ""
    local search_results = {}
    if show_search_results and #search_query >= 2 then
        search_results = search_items(search_query)
    end

    -- Build search result buttons
    local search_result_items = {}
    for _, item_name in ipairs(search_results) do
        local short_name = item_name:gsub("^[^:]+:", "")
        table.insert(search_result_items, gui.HBox {
            w = 7.1,
            spacing = 0.1,
            gui.ItemImage {w = 0.5, h = 0.5, item_name = item_name},
            gui.Button {
                w = 6.45,
                h = 0.55,
                label = #short_name > 28 and short_name:sub(1, 26) .. ".." or short_name,
                name = "search_select_" .. item_name:gsub(":", "_"),
                on_event = function(player, ctx)
                    ctx.treasure_edit.name = item_name
                    ctx.item_search_query = ""
                    ctx.show_search_results = false
                    return true
                end,
            },
        })
    end

    if show_search_results and #search_results == 0 and #search_query >= 2 then
        table.insert(search_result_items, gui.Label {label = S("No items found.")})
    end

    -- Field key suffix to force re-render on selection change
    local field_key = ctx.selected_treasure_id or "new"

    -- Truncate selected item name for display (remove mod prefix)
    local selected_display = S("(none)")
    if ctx.treasure_edit.name ~= "" then
        local name = ctx.treasure_edit.name:gsub("^[^:]+:", "")
        selected_display = #name > 28 and name:sub(1, 26) .. ".." or name
    end

    return gui.HBox {
        spacing = 1,
        -- Left: treasure list
        gui.VBox {
            w = 6,
            gui.Label {label = S("Treasures for: @1", arena.name), style = {font = "bold"}},
            gui.Box {w = 6, h = 0.02, color = "#555555"},
            gui.ScrollableVBox {
                name = "treasures_scroll",
                w = 5.5,
                h = 7,
                scroll_factor = 0,
                unpack(treasure_items),
            },
        },
        -- Right: edit form
        gui.VBox {
            w = 7.5,
            gui.Label {label = S("Edit Treasure"), style = {font = "bold"}},
            gui.Box {w = 7.5, h = 0.02, color = "#555555"},

            -- Item selection section
            gui.HBox {
                spacing = 0.2,
                gui.Field {
                    name = "item_search",
                    w = 5.3,
                    default = search_query,
                    on_event = function() return false end,
                },
                gui.Button {
                    w = 1.7,
                    label = S("Find"),
                    name = "search_btn",
                    on_event = function(player, ctx)
                        ctx.item_search_query = ctx.form.item_search or ""
                        ctx.show_search_results = true
                        return true
                    end,
                },
            },

            show_search_results and gui.VBox {
                gui.Box {w = 7.5, h = 0.02, color = "#444444"},
                gui.ScrollableVBox {
                    name = "search_results",
                    w = 7.5,
                    h = 3,
                    scroll_factor = 0,
                    unpack(search_result_items),
                },
                gui.Button {
                    w = 7.5,
                    h = 0.4,
                    label = S("Close"),
                    name = "close_search",
                    on_event = function(player, ctx)
                        ctx.show_search_results = false
                        return true
                    end,
                },
            } or gui.VBox {
                gui.Box {w = 7.5, h = 0.02, color = "#333333"},
                ctx.treasure_edit.name ~= "" and gui.HBox {
                    spacing = 0.2,
                    gui.Stack {
                        min_w = 1,
                        min_h = 1,
                        gui.ItemImage {w = 1, h = 1, item_name = ctx.treasure_edit.name},
                    },
                    gui.VBox {
                        gui.Label {label = S("Item:"), style = {font_size = "*0.85"}},
                        gui.Label {label = selected_display},
                    },
                } or gui.Label {label = S("Search for an item above")},
            },

            gui.Box {w = 7.5, h = 0.02, color = "#333333"},

            gui.HBox {
                spacing = 0.3,
                gui.VBox {
                    w = 2.1,
                    gui.Label {label = S("Count") .. ":", style = {font_size = "*0.85"}},
                    gui.Field {
                        name = "treasure_count_" .. field_key,
                        w = 2.1,
                        default = tostring(ctx.treasure_edit.count or 1),
                        on_event = function() return false end,
                    },
                },
                gui.VBox {
                    w = 2.1,
                    gui.Label {label = S("Preciousness") .. " (0-10):", style = {font_size = "*0.85"}},
                    gui.Field {
                        name = "treasure_prec_" .. field_key,
                        w = 2.1,
                        default = tostring(ctx.treasure_edit.preciousness or 1),
                        on_event = function() return false end,
                    },
                },
                gui.VBox {
                    w = 2.1,
                    gui.Label {label = S("Rarity") .. " (0.1-10):", style = {font_size = "*0.85"}},
                    gui.Field {
                        name = "treasure_rarity_" .. field_key,
                        w = 2.1,
                        default = tostring(ctx.treasure_edit.rarity or 5.0),
                        on_event = function() return false end,
                    },
                },
            },

            gui.Box {w = 7.5, h = 0.02, color = "#333333"},

            gui.HBox {
                gui.Button {
                    w = 2,
                    label = S("Save"),
                    name = "treasure_save",
                    on_event = function(player, ctx)
                        local pl_name = player:get_player_name()
                        local name = ctx.treasure_edit.name or ""
                        local field_key = ctx.selected_treasure_id or "new"
                        local count = tonumber(ctx.form["treasure_count_" .. field_key]) or 1
                        local prec = tonumber(ctx.form["treasure_prec_" .. field_key]) or 1
                        local rarity = tonumber(ctx.form["treasure_rarity_" .. field_key]) or 5.0

                        if name == "" then
                            skywars.print_msg(pl_name, S("Item name required."))
                            return false
                        end

                        if not core.registered_items[name] then
                            skywars.print_msg(pl_name, S("Unknown item: @1", name))
                            return false
                        end

                        -- Validate ranges
                        if count < 1 or count > 65535 then
                            skywars.print_msg(pl_name, S("Count must be between 1 and 65535."))
                            return false
                        end
                        if prec < 0 or prec > 10 then
                            skywars.print_msg(pl_name, S("Preciousness must be between 0 and 10."))
                            return false
                        end
                        if rarity < 0.1 or rarity > 10.0 then
                            skywars.print_msg(pl_name, S("Rarity must be between 0.1 and 10."))
                            return false
                        end

                        if ctx.selected_treasure_id then
                            -- Update existing
                            for _, t in ipairs(arena.treasures) do
                                if t.id == ctx.selected_treasure_id then
                                    t.name = name
                                    t.count = count
                                    t.preciousness = prec
                                    t.rarity = rarity
                                    break
                                end
                            end
                        else
                            -- Add new
                            local new_id = skywars.generate_treasure_id(arena)
                            table.insert(arena.treasures, {
                                id = new_id,
                                name = name,
                                count = count,
                                preciousness = prec,
                                rarity = rarity,
                            })
                        end

                        skywars.reorder_treasures(arena)
                        arena_lib.change_arena_property(player:get_player_name(), "skywars", arena.name, "treasures", arena.treasures)

                        reset_treasure_state(ctx)
                        return true
                    end,
                },
                ctx.selected_treasure_id and gui.Button {
                    w = 2,
                    label = S("Delete"),
                    name = "treasure_delete",
                    on_event = function(player, ctx)
                        if not ctx.selected_treasure_id then return false end

                        for i, t in ipairs(arena.treasures) do
                            if t.id == ctx.selected_treasure_id then
                                table.remove(arena.treasures, i)
                                break
                            end
                        end

                        arena_lib.change_arena_property(player:get_player_name(), "skywars", arena.name, "treasures", arena.treasures)

                        reset_treasure_state(ctx)
                        return true
                    end,
                } or gui.Button {
                    w = 2,
                    label = S("Clear"),
                    name = "treasure_clear",
                    on_event = function(player, ctx)
                        reset_treasure_state(ctx)
                        return true
                    end,
                },
            },
        },
    }
end


--- Build the Kits tab content.
local function build_kits_tab(arena, ctx)
    local all_kits = skywars.load_table("kits") or {}

    -- Kit list (left side)
    local kit_items = {}
    local kit_names = {}
    for kit_name in pairs(all_kits) do
        table.insert(kit_names, kit_name)
    end
    table.sort(kit_names)

    for _, kit_name in ipairs(kit_names) do
        local kname, kdata = kit_name, all_kits[kit_name]
        local is_selected = ctx.selected_kit == kname
        local bg_color = is_selected and "#335577" or "#444444"
        local has_texture = kdata.texture and kdata.texture ~= ""

        table.insert(kit_items, gui.HBox {
            w = 6,
            spacing = 0.2,
            gui.Stack {
                min_w = 0.7,
                min_h = 0.7,
                gui.Box {w = 0.7, h = 0.7, color = bg_color},
                has_texture and gui.Image {w = 0.6, h = 0.6, texture_name = kdata.texture, padding = 0.05}
                or gui.Label {label = "K", align_h = "centre", align_v = "centre"},
            },
            gui.Label {label = kname, w = 4},
            gui.Button {
                w = 0.7,
                h = 0.7,
                label = is_selected and "*" or ">",
                name = "kit_" .. kname,
                style = is_selected and {bgcolor = "#446688"} or nil,
                on_event = function(player, ctx)
                    if ctx.selected_kit == kname then
                        reset_kit_state(ctx)
                        return true
                    end
                    ctx.selected_kit = kname
                    ctx.kit_edit = {name = kname, texture = kdata.texture or ""}
                    return true
                end,
            },
        })
    end

    if #kit_items == 0 then
        table.insert(kit_items, gui.Label {label = S("No kits defined.")})
    end

    -- Arena kit checkboxes
    local arena_kit_checks = {}
    local arena_kit_names = {}
    for kit_name in pairs(all_kits) do
        table.insert(arena_kit_names, kit_name)
    end
    table.sort(arena_kit_names)

    for _, kit_name in ipairs(arena_kit_names) do
        local in_arena = false
        for _, ak in ipairs(arena.kits or {}) do
            if ak == kit_name then
                in_arena = true; break
            end
        end

        table.insert(arena_kit_checks, gui.Checkbox {
            name = "arena_kit_" .. kit_name,
            label = kit_name,
            selected = in_arena,
            on_event = function(player, ctx)
                local checked = ctx.form["arena_kit_" .. kit_name]
                if checked then
                    local found = false
                    for _, ak in ipairs(arena.kits) do
                        if ak == kit_name then
                            found = true; break
                        end
                    end
                    if not found then table.insert(arena.kits, kit_name) end
                else
                    for i, ak in ipairs(arena.kits) do
                        if ak == kit_name then
                            table.remove(arena.kits, i); break
                        end
                    end
                end
                arena_lib.change_arena_property(player:get_player_name(), "skywars", arena.name, "kits", arena.kits)
                return true
            end,
        })
    end

    -- Right panel: kit details
    local right_content = {}

    if ctx.selected_kit and all_kits[ctx.selected_kit] then
        local kit = all_kits[ctx.selected_kit]

        -- Sync kit_edit with current kit data
        if not ctx.kit_edit.name or ctx.kit_edit.name == "" then
            ctx.kit_edit = {
                name = ctx.selected_kit,
                texture = kit.texture or "",
            }
        end

        -- Header with name and settings
        table.insert(right_content, gui.HBox {
            gui.Label {label = ctx.selected_kit, style = {font = "bold"}, w = 5.5},
            gui.Button {
                w = 0.8, h = 0.5,
                label = "...",
                name = "kit_toggle_settings",
                on_event = function(player, ctx)
                    ctx.kit_show_settings = not ctx.kit_show_settings
                    if ctx.kit_show_settings then
                        ctx.kit_edit = {
                            name = ctx.selected_kit,
                            texture = kit.texture or "",
                        }
                    end
                    return true
                end,
            },
        })

        -- Inline settings (if expanded)
        if ctx.kit_show_settings then
            -- Field key suffix to force re-render on kit change
            local kit_field_key = ctx.selected_kit or "new"

            table.insert(right_content, gui.Box {w = 6.5, h = 0.02, color = "#555555"})
            table.insert(right_content, gui.HBox {
                gui.Label {label = S("Name:"), w = 1.3},
                gui.Field {
                    name = "kit_edit_name_" .. kit_field_key,
                    w = 5,
                    default = ctx.kit_edit.name or ctx.selected_kit,
                    on_event = function() return false end,
                },
            })
            table.insert(right_content, gui.HBox {
                spacing = 0.15,
                gui.Label {label = S("Icon:"), w = 1.3},
                ctx.kit_edit.texture ~= "" and gui.Image {w = 0.5, h = 0.5, texture_name = ctx.kit_edit.texture} or gui.Nil {},
                gui.Field {
                    name = "kit_icon_search_" .. kit_field_key,
                    w = 2.7,
                    default = ctx.kit_icon_query or "",
                    on_event = function() return false end,
                },
                gui.Button {
                    w = 1.5,
                    label = S("Find"),
                    name = "kit_icon_find_" .. kit_field_key,
                    on_event = function(player, ctx)
                        ctx.kit_icon_query = ctx.form["kit_icon_search_" .. kit_field_key] or ""
                        return true
                    end,
                },
            })

            -- Icon search results (inline)
            local icon_query = ctx.kit_icon_query or ""
            if #icon_query >= 2 then
                local icon_results = search_items(icon_query)
                for i, item_name in ipairs(icon_results) do
                    local iname = item_name
                    table.insert(right_content, gui.HBox {
                        spacing = 0.15,
                        gui.ItemImage {w = 0.4, h = 0.4, item_name = iname},
                        gui.Button {
                            w = 5.9, h = 0.45,
                            label = iname:gsub("^[^:]+:", ""),
                            name = "kit_icon_" .. iname:gsub(":", "_"),
                            on_event = function(player, ctx)
                                local texture = skywars.get_texture_from_item(iname)
                                if texture ~= "" then
                                    ctx.kit_edit.texture = texture
                                    ctx.kit_icon_query = ""
                                end
                                return true
                            end,
                        },
                    })
                end
            end

            table.insert(right_content, gui.HBox {
                gui.Button {
                    w = 3.2, label = S("Save"),
                    name = "kit_save_settings",
                    on_event = function(player, ctx)
                        local pl_name = player:get_player_name()
                        local old_name = ctx.selected_kit
                        local new_name = ctx.form["kit_edit_name_" .. old_name] or ""
                        local texture = ctx.kit_edit.texture or ""

                        if new_name == "" then
                            skywars.print_msg(pl_name, S("Kit name required."))
                            return false
                        end

                        local kits = skywars.load_table("kits")
                        if not kits[old_name] then return false end

                        if new_name ~= old_name and kits[new_name] then
                            skywars.print_msg(pl_name, S("Kit '@1' already exists.", new_name))
                            return false
                        end

                        if new_name ~= old_name then
                            kits[new_name] = kits[old_name]
                            kits[old_name] = nil
                            local mod_ref = arena_lib.mods["skywars"]
                            if mod_ref then
                                for _, ar in pairs(mod_ref.arenas) do
                                    for i, ak in ipairs(ar.kits or {}) do
                                        if ak == old_name then ar.kits[i] = new_name end
                                    end
                                end
                            end
                        end

                        kits[new_name].texture = texture
                        skywars.store_table("kits", kits)
                        ctx.selected_kit = new_name
                        ctx.kit_edit = {name = new_name, texture = texture}
                        ctx.kit_show_settings = false
                        ctx.kit_icon_query = ""
                        return true
                    end,
                },
                gui.Button {
                    w = 3, label = S("Delete"),
                    name = "kit_delete",
                    style = {bgcolor = "#AA3333"},
                    on_event = function(player, ctx)
                        local kits = skywars.load_table("kits")
                        kits[ctx.selected_kit] = nil
                        skywars.store_table("kits", kits)
                        for i, ak in ipairs(arena.kits) do
                            if ak == ctx.selected_kit then
                                table.remove(arena.kits, i); break
                            end
                        end
                        arena_lib.change_arena_property(player:get_player_name(), "skywars", arena.name, "kits", arena.kits)
                        reset_kit_state(ctx)
                        return true
                    end,
                },
            })
        end

        table.insert(right_content, gui.Box {w = 6.5, h = 0.02, color = "#333333"})

        -- Items section
        table.insert(right_content, gui.Label {label = S("Items:")})

        -- Use kit name in field names to force re-render on kit change
        local kit_key = ctx.selected_kit:gsub("[^%w]", "_")

        for i, item in ipairs(kit.items or {}) do
            local idx = i
            -- Check if item exists in registered items
            local item_exists = core.registered_items[item.name] ~= nil
            local short_name = item.name:gsub("^[^:]+:", "")
            local display_name = item_exists and short_name or (S("[NOT FOUND] ") .. short_name)

            table.insert(right_content, gui.HBox {
                spacing = 0.15,
                item_exists and gui.ItemImage {w = 1, h = 1, item_name = item.name}
                or gui.Stack {
                    min_w = 1, min_h = 1,
                    gui.Box {w = 1, h = 1, color = "#883333"},
                    gui.Label {label = "?", w = 1, h = 1, align_h = "centre", align_v = "centre", style = {font = "bold", textcolor = "#FF6666"}},
                },
                gui.Label {label = display_name, w = 3.5, style = item_exists and {} or {textcolor = "#FF6666"}},
                gui.Label {label = "x", w = 0.3},
                gui.Field {
                    name = "kit_item_count_" .. kit_key .. "_" .. idx,
                    w = 0.7,
                    default = tostring(item.count),
                    on_event = function(player, ctx)
                        local field_name = "kit_item_count_" .. kit_key .. "_" .. idx
                        local new_count = tonumber(ctx.form[field_name]) or 1
                        new_count = math.max(1, math.min(99, new_count))
                        if new_count == item.count then return false end
                        local kits = skywars.load_table("kits")
                        if kits[ctx.selected_kit] and kits[ctx.selected_kit].items[idx] then
                            kits[ctx.selected_kit].items[idx].count = new_count
                            skywars.store_table("kits", kits)
                        end
                        return false
                    end,
                },
                gui.Button {
                    label = "X", w = 0.5, h = 0.45,
                    name = "kit_rm_" .. kit_key .. "_" .. idx,
                    on_event = function(player, ctx)
                        local kits = skywars.load_table("kits")
                        if kits[ctx.selected_kit] then
                            table.remove(kits[ctx.selected_kit].items, idx)
                            skywars.store_table("kits", kits)
                        end
                        return true
                    end,
                },
            })
        end

        if #(kit.items or {}) == 0 then
            table.insert(right_content, gui.Label {label = S("No items in kit.")})
        end

        -- Add item (inline)
        table.insert(right_content, gui.Box {w = 6.5, h = 0.02, color = "#333333"})
        table.insert(right_content, gui.HBox {
            spacing = 0.15,
            gui.Field {
                name = "kit_item_search",
                w = 5.3,
                default = ctx.kit_item_query or "",
                on_event = function() return false end,
            },
            gui.Button {
                w = 1.4,
                label = S("Find"),
                name = "kit_item_find",
                on_event = function(player, ctx)
                    ctx.kit_item_query = ctx.form.kit_item_search or ""
                    return true
                end,
            },
        })

        -- Item search results (inline)
        local item_query = ctx.kit_item_query or ""
        if #item_query >= 2 then
            local item_results = search_items(item_query)
            for i, item_name in ipairs(item_results) do
                local iname = item_name
                table.insert(right_content, gui.HBox {
                    spacing = 0.15,
                    gui.ItemImage {w = 0.4, h = 0.4, item_name = iname},
                    gui.Button {
                        w = 5.9, h = 0.45,
                        label = iname:gsub("^[^:]+:", ""),
                        name = "kit_add_" .. iname:gsub(":", "_"),
                        on_event = function(player, ctx)
                            local kits = skywars.load_table("kits")
                            if kits[ctx.selected_kit] then
                                kits[ctx.selected_kit].items = kits[ctx.selected_kit].items or {}
                                table.insert(kits[ctx.selected_kit].items, {name = iname, count = 1})
                                skywars.store_table("kits", kits)
                                ctx.kit_item_query = ""
                            end
                            return true
                        end,
                    },
                })
            end
        end
    else
        -- No kit selected: show creation form
        table.insert(right_content, gui.Label {label = S("New Kit"), style = {font = "bold"}})
        table.insert(right_content, gui.Box {w = 6.5, h = 0.02, color = "#555555"})

        table.insert(right_content, gui.HBox {
            gui.Label {label = S("Name:"), w = 1.3},
            gui.Field {
                name = "kit_new_name",
                w = 5,
                default = ctx.kit_edit.name or "",
                on_event = function() return false end,
            },
        })

        table.insert(right_content, gui.HBox {
            spacing = 0.2,
            gui.Label {label = S("Icon:"), w = 1.3},
            ctx.kit_edit.texture ~= "" and gui.Image {w = 0.5, h = 0.5, texture_name = ctx.kit_edit.texture} or gui.Nil {},
            gui.Field {
                name = "kit_new_icon_search",
                w = 3,
                default = ctx.kit_icon_query or "",
                on_event = function() return false end,
            },
            gui.Button {
                w = 1.4,
                label = S("Find"),
                name = "kit_new_icon_find",
                on_event = function(player, ctx)
                    ctx.kit_icon_query = ctx.form.kit_new_icon_search or ""
                    return true
                end,
            },
        })

        -- Icon search results for new kit
        local icon_query = ctx.kit_icon_query or ""
        if #icon_query >= 2 then
            local icon_results = search_items(icon_query)
            for i, item_name in ipairs(icon_results) do
                local iname = item_name
                table.insert(right_content, gui.HBox {
                    spacing = 0.15,
                    gui.ItemImage {w = 0.4, h = 0.4, item_name = iname},
                    gui.Button {
                        w = 5.9, h = 0.45,
                        label = iname:gsub("^[^:]+:", ""),
                        name = "kit_new_icon_" .. iname:gsub(":", "_"),
                        on_event = function(player, ctx)
                            local texture = skywars.get_texture_from_item(iname)
                            if texture ~= "" then
                                ctx.kit_edit.texture = texture
                                ctx.kit_icon_query = ""
                            end
                            return true
                        end,
                    },
                })
            end
        end

        table.insert(right_content, gui.Box {w = 6.5, h = 0.02, color = "#333333"})
        table.insert(right_content, gui.Button {
            w = 6.5,
            label = S("Create"),
            name = "kit_create",
            on_event = function(player, ctx)
                local pl_name = player:get_player_name()
                local name = ctx.form.kit_new_name or ""
                local texture = ctx.kit_edit.texture or ""

                if name == "" then
                    skywars.print_msg(pl_name, S("Kit name required."))
                    return false
                end

                local kits = skywars.load_table("kits")
                if kits[name] then
                    skywars.print_msg(pl_name, S("Kit already exists."))
                    return false
                end

                kits[name] = {texture = texture, items = {}}
                skywars.store_table("kits", kits)
                ctx.selected_kit = name
                ctx.kit_edit = {name = name, texture = texture}
                ctx.kit_icon_query = ""
                return true
            end,
        })
    end

    return gui.HBox {
        spacing = 1.2,
        -- Left: kit list + arena assignment
        gui.VBox {
            w = 6,
            gui.Label {label = S("Global Kits"), style = {font = "bold"}},
            gui.Box {w = 6, h = 0.02, color = "#555555"},
            gui.ScrollableVBox {
                name = "kits_scroll",
                w = 6,
                h = 4,
                scroll_factor = 0,
                unpack(kit_items),
            },
            gui.Box {w = 6, h = 0.02, color = "#333333"},
            gui.Label {label = S("In Arena:")},
            gui.ScrollableVBox {
                name = "arena_kits_scroll",
                w = 6,
                h = 2.5,
                scroll_factor = 0,
                unpack(arena_kit_checks),
            },
        },
        -- Right: details (scrollable)
        gui.ScrollableVBox {
            name = "kit_details_scroll",
            w = 7,
            h = 7.5,
            scroll_factor = 0,
            unpack(right_content),
        },
    }
end


--- Build the Templates tab content.
---@param arena table
---@param ctx table
local function build_templates_tab(arena, ctx)
    local templates = arena.chest_templates or {}
    local template_items = {}

    -- Sort templates by max preciousness, then by min preciousness, then alphabetically
    local sorted_ids = {}
    for id in pairs(templates) do
        table.insert(sorted_ids, id)
    end
    table.sort(sorted_ids, function(a, b)
        local max_a = templates[a].max_preciousness
        local max_b = templates[b].max_preciousness
        if max_a == max_b then
            local min_a = templates[a].min_preciousness
            local min_b = templates[b].min_preciousness
            if min_a == min_b then
                return templates[a].name < templates[b].name
            end
            return min_a < min_b
        end
        return max_a < max_b
    end)

    for _, id in ipairs(sorted_ids) do
        local tpl_id, tpl_data = id, templates[id] -- capture for closure
        local is_selected = ctx.selected_template_id == tpl_id
        local bg_color = is_selected and "#335577" or "#444444"
        local tpl_color = tpl_data.color or "brown"
        local tpl_hex = skywars_settings.chest_placeholder_colors[tpl_color] or "#8B4513"

        table.insert(template_items, gui.HBox {
            w = 7,
            spacing = 0.2,
            gui.Box {w = 0.3, h = 0.8, color = tpl_hex},
            gui.Stack {
                min_w = 0.8,
                min_h = 0.8,
                gui.Box {w = 0.8, h = 0.8, color = bg_color},
                gui.ItemImage {w = 0.7, h = 0.7, item_name = "default:chest", padding = 0.05},
            },
            gui.VBox {
                spacing = 0,
                w = 4.5,
                gui.Label {label = tpl_data.name, style = {font = "bold"}},
                gui.HBox {
                    spacing = 0.3,
                    gui.Label {label = "P:" .. tpl_data.min_preciousness .. "-" .. tpl_data.max_preciousness, style = {font_size = "*0.9"}},
                    gui.Label {label = "T:" .. tpl_data.min_treasures .. "-" .. tpl_data.max_treasures, style = {font_size = "*0.9"}},
                },
            },
            gui.Button {
                w = 0.8,
                h = 0.8,
                label = is_selected and "*" or ">",
                name = "template_" .. tpl_id,
                style = is_selected and {bgcolor = "#446688"} or nil,
                on_event = function(player, ctx)
                    if ctx.selected_template_id == tpl_id then
                        reset_template_state(ctx)
                        return true
                    end
                    local color_idx = 1
                    local color_key = tpl_data.color or "brown"
                    for i, c in ipairs(COLOR_OPTIONS) do
                        if c == color_key then
                            color_idx = i
                            break
                        end
                    end
                    ctx.selected_template_id = tpl_id
                    ctx.template_edit = {
                        name = tpl_data.name,
                        min_p = tpl_data.min_preciousness,
                        max_p = tpl_data.max_preciousness,
                        min_t = tpl_data.min_treasures,
                        max_t = tpl_data.max_treasures,
                        color = color_key,
                    }
                    ctx.form.template_name = tpl_data.name
                    ctx.form.template_min_p = tostring(tpl_data.min_preciousness)
                    ctx.form.template_max_p = tostring(tpl_data.max_preciousness)
                    ctx.form.template_min_t = tostring(tpl_data.min_treasures)
                    ctx.form.template_max_t = tostring(tpl_data.max_treasures)
                    ctx.form.template_color = color_idx
                    return true
                end,
            },
        })
    end

    if #template_items == 0 then
        table.insert(template_items, gui.Label {label = S("No templates defined.")})
    end

    return gui.HBox {
        spacing = 1,
        -- Left: templates list
        gui.VBox {
            w = 7,
            gui.Label {label = S("Chests for: @1", arena.name), style = {font = "bold"}},
            gui.Box {w = 7, h = 0.02, color = "#555555"},
            gui.ScrollableVBox {
                name = "templates_scroll",
                w = 7,
                h = 6,
                scroll_factor = 0,
                unpack(template_items),
            },
        },
        -- Right: edit form
        gui.VBox {
            w = 7,
            gui.Label {
                label = ctx.selected_template_id and S("Edit Template") or S("New Template"),
                style = {font = "bold"},
            },
            gui.Box {w = 7, h = 0.02, color = "#555555"},

            gui.Label {label = S("Name:")},
            gui.Field {
                name = "template_name",
                w = 7,
                default = ctx.template_edit.name or "",
                on_event = function() return false end,
            },

            gui.HBox {
                gui.Label {label = S("Preciousness") .. " (0-10):", w = 2.5, style = {font_size = "*0.9"}},
                gui.Field {
                    name = "template_min_p",
                    w = 1.5,
                    default = tostring(ctx.template_edit.min_p or 1),
                    on_event = function() return false end,
                },
                gui.Label {label = "-", w = 0.3},
                gui.Field {
                    name = "template_max_p",
                    w = 1.5,
                    default = tostring(ctx.template_edit.max_p or 3),
                    on_event = function() return false end,
                },
            },

            gui.HBox {
                gui.Label {label = S("Treasures") .. " (1-20):", w = 2.5, style = {font_size = "*0.9"}},
                gui.Field {
                    name = "template_min_t",
                    w = 1.5,
                    default = tostring(ctx.template_edit.min_t or 1),
                    on_event = function() return false end,
                },
                gui.Label {label = "-", w = 0.3},
                gui.Field {
                    name = "template_max_t",
                    w = 1.5,
                    default = tostring(ctx.template_edit.max_t or 5),
                    on_event = function() return false end,
                },
            },

            gui.HBox {
                gui.Label {label = S("Color:"), w = 2.5},
                gui.Box {
                    w = 0.6, h = 0.6,
                    color = skywars_settings.chest_placeholder_colors[ctx.template_edit.color or "brown"] or "#8B4513",
                },
                gui.Dropdown {
                    name = "template_color",
                    w = 3.5,
                    items = COLOR_OPTIONS,
                    selected_idx = (function()
                        local cur = ctx.template_edit.color or "brown"
                        for i, c in ipairs(COLOR_OPTIONS) do
                            if c == cur then return i end
                        end
                        return 1
                    end)(),
                    index_event = true,
                    on_event = function(player, ctx)
                        local new_color = COLOR_OPTIONS[ctx.form.template_color]
                        if new_color and new_color ~= ctx.template_edit.color then
                            ctx.template_edit.color = new_color
                            return true
                        end
                        return false
                    end,
                },
            },

            gui.Box {w = 7, h = 0.02, color = "#333333"},

            gui.HBox {
                gui.Button {
                    w = 2,
                    label = S("Create"),
                    name = "template_create",
                    on_event = function(player, ctx)
                        local pl_name = player:get_player_name()
                        local _, arena = arena_lib.get_arena_by_name("skywars", ctx.arena_name)
                        if not arena then return false end

                        local name = ctx.form.template_name or ""
                        local min_p = tonumber(ctx.form.template_min_p) or 1
                        local max_p = tonumber(ctx.form.template_max_p) or 3
                        local min_t = tonumber(ctx.form.template_min_t) or 1
                        local max_t = tonumber(ctx.form.template_max_t) or 5

                        if name == "" then
                            skywars.print_msg(pl_name, S("Template name required."))
                            return false
                        end

                        -- Validate ranges
                        if min_p < 0 or min_p > 10 then
                            skywars.print_msg(pl_name, S("Preciousness must be between 0 and 10."))
                            return false
                        end
                        if max_p < 0 or max_p > 10 then
                            skywars.print_msg(pl_name, S("Preciousness must be between 0 and 10."))
                            return false
                        end
                        if max_p < min_p then
                            skywars.print_msg(pl_name, S("Max preciousness must be greater than or equal to min."))
                            return false
                        end
                        if min_t < 1 or min_t > 20 then
                            skywars.print_msg(pl_name, S("Treasures must be between 1 and 20."))
                            return false
                        end
                        if max_t < 1 or max_t > 20 then
                            skywars.print_msg(pl_name, S("Treasures must be between 1 and 20."))
                            return false
                        end
                        if max_t < min_t then
                            skywars.print_msg(pl_name, S("Max treasures must be greater than or equal to min."))
                            return false
                        end

                        local color = ctx.template_edit.color or "brown"
                        local tpl = skywars.create_chest_template(arena, pl_name, name, min_p, max_p, min_t, max_t, color)

                        skywars.print_msg(pl_name, S("Template '@1' created.", name))
                        skywars.refresh_editor_hotbar(player, arena)

                        ctx.selected_template_id = tpl.id
                        ctx.template_edit = {
                            name = tpl.name,
                            min_p = tpl.min_preciousness,
                            max_p = tpl.max_preciousness,
                            min_t = tpl.min_treasures,
                            max_t = tpl.max_treasures,
                            color = tpl.color,
                        }
                        return true
                    end,
                },
                ctx.selected_template_id and gui.Button {
                    w = 2,
                    label = S("Update"),
                    name = "template_update",
                    on_event = function(player, ctx)
                        if not ctx.selected_template_id then return false end

                        local pl_name = player:get_player_name()
                        local _, arena = arena_lib.get_arena_by_name("skywars", ctx.arena_name)
                        if not arena then return false end

                        local name = ctx.form.template_name or ""
                        local min_p = tonumber(ctx.form.template_min_p) or 1
                        local max_p = tonumber(ctx.form.template_max_p) or 3
                        local min_t = tonumber(ctx.form.template_min_t) or 1
                        local max_t = tonumber(ctx.form.template_max_t) or 5

                        if name == "" then
                            skywars.print_msg(pl_name, S("Template name required."))
                            return false
                        end

                        -- Validate ranges
                        if min_p < 0 or min_p > 10 then
                            skywars.print_msg(pl_name, S("Preciousness must be between 0 and 10."))
                            return false
                        end
                        if max_p < 0 or max_p > 10 then
                            skywars.print_msg(pl_name, S("Preciousness must be between 0 and 10."))
                            return false
                        end
                        if max_p < min_p then
                            skywars.print_msg(pl_name, S("Max preciousness must be greater than or equal to min."))
                            return false
                        end
                        if min_t < 1 or min_t > 20 then
                            skywars.print_msg(pl_name, S("Treasures must be between 1 and 20."))
                            return false
                        end
                        if max_t < 1 or max_t > 20 then
                            skywars.print_msg(pl_name, S("Treasures must be between 1 and 20."))
                            return false
                        end
                        if max_t < min_t then
                            skywars.print_msg(pl_name, S("Max treasures must be greater than or equal to min."))
                            return false
                        end

                        local color = ctx.template_edit.color or "brown"
                        skywars.update_chest_template(arena, pl_name, ctx.selected_template_id, name, min_p, max_p, min_t, max_t, color)
                        skywars.print_msg(pl_name, S("Template updated."))
                        skywars.refresh_editor_hotbar(player, arena)

                        ctx.template_edit = {name = name, min_p = min_p, max_p = max_p, min_t = min_t, max_t = max_t, color = color}
                        return true
                    end,
                } or gui.Nil {},
                ctx.selected_template_id and gui.Button {
                    w = 2,
                    label = S("Delete"),
                    name = "template_delete",
                    on_event = function(player, ctx)
                        if not ctx.selected_template_id then return false end

                        local pl_name = player:get_player_name()
                        local _, arena = arena_lib.get_arena_by_name("skywars", ctx.arena_name)
                        if not arena then return false end

                        skywars.delete_chest_template(arena, ctx.selected_template_id, pl_name)
                        skywars.print_msg(pl_name, S("Template deleted."))
                        skywars.refresh_editor_hotbar(player, arena)

                        reset_template_state(ctx)
                        return true
                    end,
                } or gui.Nil {},
            },
        },
    }
end


--- Main editor GUI form.
local editor_form = flow.make_gui(function(player, ctx)
    -- Initialize context defaults using DEFAULTS as single source of truth
    ctx.active_tab = ctx.active_tab or TAB_TREASURES
    ctx.treasure_edit = ctx.treasure_edit or table.copy(DEFAULTS.treasure)
    ctx.kit_edit = ctx.kit_edit or table.copy(DEFAULTS.kit)
    ctx.template_edit = ctx.template_edit or table.copy(DEFAULTS.template)

    -- Get arena from context
    local arena_name = ctx.arena_name
    if not arena_name then
        return gui.VBox {
            padding = 0.5,
            gui.Label {label = S("Error: No arena selected.")},
        }
    end

    local _, arena = arena_lib.get_arena_by_name("skywars", arena_name)
    if not arena then
        return gui.VBox {
            padding = 0.5,
            gui.Label {label = S("Error: Arena '@1' not found.", arena_name)},
        }
    end

    -- Run sanitization only once when form is first opened
    if not ctx.sanitized then
        ctx.sanitized = true
        sanitize_editor_data(arena, player:get_player_name())
    end

    -- Build active tab content
    local tab_content
    if ctx.active_tab == TAB_TREASURES then
        tab_content = build_treasures_tab(arena, ctx)
    elseif ctx.active_tab == TAB_KITS then
        tab_content = build_kits_tab(arena, ctx)
    elseif ctx.active_tab == TAB_TEMPLATES then
        tab_content = build_templates_tab(arena, ctx)
    end

    return gui.VBox {
        padding = 0.4,
        spacing = 0.3,
        w = 15,

        -- Tab bar
        gui.HBox {
            spacing = 0.1,
            tab_button(TAB_TREASURES, S("Treasures"), ctx),
            tab_button(TAB_KITS, S("Kits"), ctx),
            tab_button(TAB_TEMPLATES, S("Chests"), ctx),
        },

        gui.Box {w = 15, h = 0.03, color = "#666666"},

        -- Tab content
        tab_content,
    }
end)



--- Shows the editor GUI to a player.
---@param player ObjectRef
---@param arena_name string
function skywars.show_editor_gui(player, arena_name)
    editor_form:show(player, {arena_name = arena_name})
end
