--- Chest Templates storage and CRUD operations.
-- Templates are stored per-arena as arena properties (arena.chest_templates).
-- Each template defines a preset chest configuration (preciousness/treasure ranges).


---@alias ChestTemplate {id: string, name: string, min_preciousness: integer, max_preciousness: integer, min_treasures: integer, max_treasures: integer, color: string?}


--- Saves all chest templates for an arena.
---@param arena table
---@param templates table<string, ChestTemplate>
---@param pl_name string
function skywars.save_arena_templates(arena, templates, pl_name)
	arena.chest_templates = templates
	arena_lib.change_arena_property(pl_name, "skywars", arena.name, "chest_templates", templates)
end



--- Generates a unique template ID for an arena.
---@param arena table
---@return string
function skywars.generate_template_id(arena)
	local templates = arena.chest_templates or {}
	local max_id = 0
	for id, _ in pairs(templates) do
		local num = tonumber(id)
		if num and num > max_id then
			max_id = num
		end
	end
	return tostring(max_id + 1)
end



--- Creates a new chest template for an arena.
-- After creation, assigns template_id to matching chests in the arena.
---@param arena table
---@param pl_name string
---@param name string
---@param min_p integer
---@param max_p integer
---@param min_t integer
---@param max_t integer
---@param color? string
---@return ChestTemplate
function skywars.create_chest_template(arena, pl_name, name, min_p, max_p, min_t, max_t, color)
	local templates = arena.chest_templates or {}
	local id = skywars.generate_template_id(arena)

	local template = {
		id = id,
		name = name,
		min_preciousness = min_p,
		max_preciousness = max_p,
		min_treasures = min_t,
		max_treasures = max_t,
		color = color,
	}

	templates[id] = template
	skywars.save_arena_templates(arena, templates, pl_name)

	-- Assign template_id to matching chests in this arena
	skywars.apply_template_to_matching_chests(arena, template, pl_name)

	return template
end



--- Updates an existing chest template in an arena.
-- Updates all chests that reference this template and their placeholder nodes.
---@param arena table
---@param pl_name string
---@param id string
---@param name string
---@param min_p integer
---@param max_p integer
---@param min_t integer
---@param max_t integer
---@param color? string
---@return ChestTemplate|nil
function skywars.update_chest_template(arena, pl_name, id, name, min_p, max_p, min_t, max_t, color)
	local templates = arena.chest_templates or {}
	if not templates[id] then return end

	templates[id] = {
		id = id,
		name = name,
		min_preciousness = min_p,
		max_preciousness = max_p,
		min_treasures = min_t,
		max_treasures = max_t,
		color = color,
	}

	skywars.save_arena_templates(arena, templates, pl_name)

	-- Update placeholder nodes for this template
	skywars.update_template_placeholder_nodes(arena, id, color)

	-- Update chests referencing this template
	local changed = false
	for _, chest in ipairs(arena.chests or {}) do
		-- Compare as strings to handle type mismatches
		if tostring(chest.template_id) == tostring(id) then
			chest.min_preciousness = min_p
			chest.max_preciousness = max_p
			chest.min_treasures = min_t
			chest.max_treasures = max_t
			changed = true
		end
	end
	if changed then
		arena_lib.change_arena_property(pl_name, "skywars", arena.name, "chests", arena.chests)
	end

	-- Apply to matching chests without template
	skywars.apply_template_to_matching_chests(arena, templates[id], pl_name)

	return templates[id]
end



--- Deletes a chest template from an arena and clears template_id from referencing chests.
---@param arena table
---@param id string
---@param pl_name string
function skywars.delete_chest_template(arena, id, pl_name)
	local templates = arena.chest_templates or {}
	if not templates[id] then return end

	templates[id] = nil
	skywars.save_arena_templates(arena, templates, pl_name)

	-- Clear template_id from chests referencing this template
	local changed = false
	for _, chest in ipairs(arena.chests or {}) do
		if chest.template_id == id then
			chest.template_id = nil
			changed = true
		end
	end
	if changed then
		arena_lib.change_arena_property(pl_name, "skywars", arena.name, "chests", arena.chests)
	end
end



--- Finds chests in an arena with exact matching params and assigns the template_id.
-- Also updates the placeholder nodes to the template's color.
-- Forces node update even if current node is not a placeholder.
---@param arena table
---@param template ChestTemplate
---@param pl_name string
function skywars.apply_template_to_matching_chests(arena, template, pl_name)
	local placeholder_node = skywars.get_placeholder_node_name(template.color)

	local changed = false
	for _, chest in ipairs(arena.chests or {}) do
		if not chest.template_id
			and chest.min_preciousness == template.min_preciousness
			and chest.max_preciousness == template.max_preciousness
			and chest.min_treasures == template.min_treasures
			and chest.max_treasures == template.max_treasures
		then
			chest.template_id = template.id
			changed = true

			-- Update placeholder node to template color
			if chest.pos then
				-- Ensure the area is loaded before modifying
				core.load_area(chest.pos, chest.pos)

				local current_node = core.get_node(chest.pos)
				local param2 = current_node.param2 or 0

				-- Mark position as updating to prevent on_destruct from removing chest data
				skywars.mark_updating_pos(chest.pos)

				-- Use swap_node for placeholder->placeholder (avoids on_destruct anyway)
				-- Use set_node for other nodes
				if skywars.is_chest_placeholder(current_node.name) then
					core.swap_node(chest.pos, {name = placeholder_node, param2 = param2})
				else
					core.set_node(chest.pos, {name = placeholder_node, param2 = 0})
				end

				skywars.unmark_updating_pos(chest.pos)
			end
		end
	end
	if changed then
		arena_lib.change_arena_property(pl_name, "skywars", arena.name, "chests", arena.chests)
	end
end



--- Gets a template by ID from an arena.
---@param arena table
---@param id string
---@return ChestTemplate|nil
function skywars.get_arena_template(arena, id)
	local templates = arena.chest_templates or {}
	return templates[id]
end



--- Counts how many chests in an arena match a template's params but have no template assigned.
---@param arena table
---@param template ChestTemplate
---@return integer
function skywars.count_matching_chests(arena, template)
	local count = 0
	for _, chest in ipairs(arena.chests or {}) do
		if not chest.template_id
			and chest.min_preciousness == template.min_preciousness
			and chest.max_preciousness == template.max_preciousness
			and chest.min_treasures == template.min_treasures
			and chest.max_treasures == template.max_treasures
		then
			count = count + 1
		end
	end
	return count
end



--- Counts how many chests in an arena are linked to a specific template.
---@param arena table
---@param template_id string
---@return integer
function skywars.count_linked_chests(arena, template_id)
	local count = 0
	for _, chest in ipairs(arena.chests or {}) do
		if chest.template_id == template_id then
			count = count + 1
		end
	end
	return count
end



--- Builds a short description string for a template (for HUD/item display).
---@param template ChestTemplate
---@return string
function skywars.get_template_description(template)
	return string.format("%s | P:%d-%d | T:%d-%d",
		template.name,
		template.min_preciousness,
		template.max_preciousness,
		template.min_treasures,
		template.max_treasures
	)
end
