--- Utilities for data sanitization and validation.
-- Provides helpers to ensure tables are ipairs-compatible arrays
-- and to sanitize corrupted data structures.



--- Checks if a table is a valid ipairs-compatible array.
--- A valid array has only numeric contiguous keys starting from 1.
---@param tbl table
---@return boolean
function skywars.is_valid_ipairs_array(tbl)
    if type(tbl) ~= "table" then return false end

    local max_index = 0
    for k, v in pairs(tbl) do
        if type(k) ~= "number" or k < 1 or k % 1 ~= 0 then
            return false
        end
        if k > max_index then max_index = k end
    end

    -- Check for holes in array
    if max_index > 0 then
        for i = 1, max_index do
            if tbl[i] == nil then
                return false
            end
        end
    end

    return true
end



--- Ensures a table is ipairs-compatible by rebuilding it if necessary.
--- Returns the original table if valid, or a new array with same values.
---@param tbl table
---@return table
function skywars.ensure_ipairs_array(tbl)
    if type(tbl) ~= "table" then return {} end
    if skywars.is_valid_ipairs_array(tbl) then return tbl end

    local result = {}
    for _, v in pairs(tbl) do
        table.insert(result, v)
    end
    return result
end



--- Sanitizes a single treasure entry, fixing missing or invalid fields.
---@param treasure table
---@param idx number Index used to generate fallback id
---@return table|nil Sanitized treasure or nil if unrecoverable
function skywars.sanitize_treasure(treasure, idx)
    if type(treasure) ~= "table" then return nil end

    if type(treasure.name) ~= "string" or treasure.name == "" then
        return nil
    end

    return {
        id = tonumber(treasure.id) or idx,
        name = treasure.name,
        count = math.max(1, math.min(65535, tonumber(treasure.count) or 1)),
        preciousness = math.max(0, math.min(10, tonumber(treasure.preciousness) or 1)),
        rarity = math.max(0.1, math.min(10, tonumber(treasure.rarity) or 5.0)),
    }
end



--- Sanitizes arena treasures list, removing invalid entries and fixing fields.
---@param arena table
---@return boolean changed Whether the list was modified
function skywars.sanitize_arena_treasures(arena)
    if type(arena.treasures) ~= "table" then
        arena.treasures = {}
        return true
    end

    local was_invalid = not skywars.is_valid_ipairs_array(arena.treasures)
    arena.treasures = skywars.ensure_ipairs_array(arena.treasures)

    local sanitized = {}
    local changed = was_invalid

    for i, treasure in ipairs(arena.treasures) do
        local clean = skywars.sanitize_treasure(treasure, i)
        if clean then
            table.insert(sanitized, clean)
            if clean.id ~= treasure.id or clean.count ~= treasure.count
                or clean.preciousness ~= treasure.preciousness or clean.rarity ~= treasure.rarity then
                changed = true
            end
        else
            changed = true
        end
    end

    if changed then
        arena.treasures = sanitized
    end

    return changed
end



--- Sanitizes a single kit entry, fixing missing or invalid fields.
---@param kit table
---@return table Sanitized kit
function skywars.sanitize_kit(kit)
    if type(kit) ~= "table" then
        return {texture = "", items = {}}
    end

    local sanitized = {
        texture = type(kit.texture) == "string" and kit.texture or "",
        items = {},
    }

    if type(kit.items) == "table" then
        local items_to_check = skywars.ensure_ipairs_array(kit.items)

        for _, item in ipairs(items_to_check) do
            if type(item) == "table" and type(item.name) == "string" and item.name ~= "" then
                table.insert(sanitized.items, {
                    name = item.name,
                    count = math.max(1, math.min(65535, tonumber(item.count) or 1)),
                })
            end
        end
    end

    return sanitized
end



--- Sanitizes all kits, removing invalid entries and fixing fields.
---@return boolean changed Whether any kit was modified
function skywars.sanitize_all_kits()
    local kits = skywars.load_table("kits")
    if type(kits) ~= "table" then
        skywars.store_table("kits", {})
        return true
    end

    local changed = false
    local sanitized = {}

    for name, kit in pairs(kits) do
        if type(name) == "string" and name ~= "" then
            local clean = skywars.sanitize_kit(kit)
            sanitized[name] = clean
            if type(kit.items) ~= "table" or #clean.items ~= #(kit.items or {}) then
                changed = true
            end
        else
            changed = true
        end
    end

    if changed then
        skywars.store_table("kits", sanitized)
    end
    return changed
end



--- Sanitizes a single template entry, fixing missing or invalid fields.
---@param template table
---@param id string
---@return table|nil Sanitized template or nil if unrecoverable
function skywars.sanitize_template(template, id)
    if type(template) ~= "table" then return nil end

    if type(template.name) ~= "string" or template.name == "" then
        return nil
    end

    local min_p = math.max(0, math.min(10, tonumber(template.min_preciousness) or 1))
    local max_p = math.max(min_p, math.min(10, tonumber(template.max_preciousness) or 3))
    local min_t = math.max(1, math.min(20, tonumber(template.min_treasures) or 1))
    local max_t = math.max(min_t, math.min(20, tonumber(template.max_treasures) or 5))
    local color = template.color
    if type(color) ~= "string" or not skywars_settings.chest_placeholder_colors[color] then
        color = "brown"
    end

    return {
        id = id,
        name = template.name,
        min_preciousness = min_p,
        max_preciousness = max_p,
        min_treasures = min_t,
        max_treasures = max_t,
        color = color,
    }
end



--- Sanitizes arena templates, removing invalid entries and fixing fields.
---@param arena table
---@return boolean changed Whether any template was modified
function skywars.sanitize_arena_templates(arena)
    if type(arena.chest_templates) ~= "table" then
        arena.chest_templates = {}
        return true
    end

    local changed = false
    local sanitized = {}

    for id, template in pairs(arena.chest_templates) do
        local clean = skywars.sanitize_template(template, id)
        if clean then
            sanitized[id] = clean
        else
            changed = true
        end
    end

    if changed then
        arena.chest_templates = sanitized
    end
    return changed
end



--- Sanitizes arena chests, removing orphan template_id references.
-- If a chest has a template_id that doesn't exist in arena.chest_templates,
-- the template_id is removed from that chest.
---@param arena table
---@return boolean changed Whether any chest was modified
function skywars.sanitize_arena_chests(arena)
    if type(arena.chests) ~= "table" then
        return false
    end

    local templates = arena.chest_templates or {}
    local changed = false

    for _, chest in ipairs(arena.chests) do
        if chest.template_id and not templates[chest.template_id] then
            chest.template_id = nil
            changed = true
        end
    end

    return changed
end
