--- Chest placeholder nodes (per-color) and template setter tool.
-- A placeholder node is registered for each color in skywars_settings.chest_placeholder_colors.
-- The setter tool stamps metadata identifying the template and places the colored placeholder.

local S = skywars.T

local SETTER_INVENTORY_IMAGE = core.inventorycube("default_chest_top.png", "default_chest_front.png", "default_chest_side.png")

local BASE_CHEST_TOP = "default_chest_top.png"
local BASE_CHEST_FRONT = "default_chest_front.png"
local BASE_CHEST_SIDE = "default_chest_side.png"

-- Positions being updated (to avoid on_destruct removing chest during color change)
-- Format: {["x,y,z"] = true}
local updating_positions = {}


--- Marks a position as being updated (prevents on_destruct from removing chest data).
---@param pos table
function skywars.mark_updating_pos(pos)
    updating_positions[core.pos_to_string(pos)] = true
end



--- Unmarks a position as being updated.
---@param pos table
function skywars.unmark_updating_pos(pos)
    updating_positions[core.pos_to_string(pos)] = nil
end



--- Checks if a position is currently being updated.
---@param pos table
---@return boolean
local function is_updating(pos)
    return updating_positions[core.pos_to_string(pos)] == true
end


--- on_destruct callback shared by all placeholder nodes.
local function placeholder_on_destruct(pos)
    -- A color update, not a real destruction
    if is_updating(pos) then return end

    local mod_data = arena_lib.mods["skywars"]

    for _, arena in pairs(mod_data.arenas) do
        if arena.enabled or arena.in_queue or arena.in_loading or arena.in_game or arena.in_celebration then
            goto continue
        end

        -- Remove node from arena chests
        for i, chest in ipairs(arena.chests or {}) do
            if chest.pos and vector.equals(chest.pos, pos) then
                table.remove(arena.chests, i)
                local editor_pl = arena_lib.get_player_in_edit_mode(arena.name)
                if editor_pl then
                    arena_lib.change_arena_property(editor_pl, "skywars", arena.name, "chests", arena.chests)
                end
                return
            end
        end

        ::continue::
    end
end


--- Registers a chest placeholder node for a specific color.
---@param color_key string
---@param hex_color string
local function register_placeholder_node(color_key, hex_color)
    local node_name = "skywars:chest_placeholder_" .. color_key
    local colorize_mod = "^[colorize:" .. hex_color .. ":170"

    core.register_node(node_name, {
        description = S("Chest Placeholder (@1)", color_key),
        tiles = {
            BASE_CHEST_TOP .. colorize_mod,
            BASE_CHEST_TOP .. colorize_mod,
            BASE_CHEST_SIDE .. colorize_mod,
            BASE_CHEST_SIDE .. colorize_mod,
            BASE_CHEST_SIDE .. colorize_mod,
            BASE_CHEST_FRONT .. colorize_mod,
        },
        groups = {dig_immediate = 3, not_in_creative_inventory = 1},
        drop = "",
        paramtype2 = "facedir",
        on_destruct = placeholder_on_destruct,
    })
end

-- Register a placeholder node for each color in settings
for color_key, hex_color in pairs(skywars_settings.chest_placeholder_colors) do
    register_placeholder_node(color_key, hex_color)
end



--- Gets the placeholder node name for a color key.
---@param color_key? string
---@return string
function skywars.get_placeholder_node_name(color_key)
    if color_key and skywars_settings.chest_placeholder_colors[color_key] then
        return "skywars:chest_placeholder_" .. color_key
    end
    return "skywars:chest_placeholder"
end



--- Checks if a node name is any chest placeholder.
---@param node_name string
---@return boolean
function skywars.is_chest_placeholder(node_name)
    if node_name:match("^skywars:chest_placeholder_") then return true end
    return false
end



core.register_craftitem("skywars:chest_template_setter", {
    description = S("Chest Template Setter"),
    inventory_image = SETTER_INVENTORY_IMAGE,
    stack_max = 1,

    on_drop = function()
        return nil
    end,

    on_place = function(itemstack, player, pointed_thing)
        if not player or pointed_thing.type ~= "node" then
            return itemstack
        end

        local pl_name = player:get_player_name()
        local p_meta = player:get_meta()

        local editor_mod = p_meta:get_string("arena_lib_editor.mod")
        local editor_arena = p_meta:get_string("arena_lib_editor.arena")

        if editor_mod ~= "skywars" or editor_arena == "" then
            skywars.print_msg(pl_name, S("You must be in arena editor mode to use this."))
            return itemstack
        end

        local arena_id, arena = arena_lib.get_arena_by_name("skywars", editor_arena)
        if not arena then
            skywars.print_msg(pl_name, S("Arena not found."))
            return itemstack
        end

        if not arena.chests then
            arena.chests = {}
        end

        local item_meta = itemstack:get_meta()
        local template_id = item_meta:get_string("template_id")

        if template_id == "" then
            skywars.print_msg(pl_name, S("This setter has no template assigned."))
            return itemstack
        end

        local template = skywars.get_arena_template(arena, template_id)
        if not template then
            skywars.print_msg(pl_name, S("Template not found."))
            return itemstack
        end

        local pos = pointed_thing.above

        local node_at_pos = core.get_node(pos)
        if node_at_pos.name ~= "air" then
            skywars.print_msg(pl_name, S("Position is not empty."))
            return itemstack
        end

        -- Place colored placeholder node
        local placeholder_node = skywars.get_placeholder_node_name(template.color)
        core.set_node(pos, {name = placeholder_node})

        local chest_id = skywars.generate_chest_id(arena)
        local new_chest = {
            id = chest_id,
            pos = vector.copy(pos),
            min_preciousness = template.min_preciousness,
            max_preciousness = template.max_preciousness,
            min_treasures = template.min_treasures,
            max_treasures = template.max_treasures,
            template_id = template_id,
        }

        table.insert(arena.chests, new_chest)
        arena_lib.change_arena_property(pl_name, "skywars", arena.name, "chests", arena.chests)

        return itemstack
    end,

    on_secondary_use = function(itemstack, player, pointed_thing)
        return itemstack
    end,
})


--- Creates a setter item with a specific template stamped in metadata.
---@param arena table
---@param template_id string
---@return ItemStack|nil
function skywars.create_template_setter_item(arena, template_id)
    local template = skywars.get_arena_template(arena, template_id)
    if not template then return end

    local item = ItemStack("skywars:chest_template_setter")
    local meta = item:get_meta()
    meta:set_string("template_id", template_id)
    meta:set_string("description", skywars.get_template_description(template))

    local color_key = template.color or "brown"
    local hex_color = skywars_settings.chest_placeholder_colors[color_key] or "#8B4513"
    local colorize_mod = "^[colorize:" .. hex_color .. ":128"
    local colorized_inv_image = core.inventorycube(
        BASE_CHEST_TOP .. colorize_mod,
        BASE_CHEST_FRONT .. colorize_mod,
        BASE_CHEST_SIDE .. colorize_mod
    )
    meta:set_string("inventory_image", colorized_inv_image)

    return item
end



--- Updates all chest placeholder nodes in an arena for a template to use its color.
-- Called when template color changes or on creation with auto-apply.
-- Forces node update even if current node is not a placeholder.
---@param arena table
---@param template_id string
---@param color_key? string
function skywars.update_template_placeholder_nodes(arena, template_id, color_key)
    local placeholder_node = skywars.get_placeholder_node_name(color_key)

    for _, chest in ipairs(arena.chests or {}) do
        -- Compare as strings to handle type mismatches
        if tostring(chest.template_id) == tostring(template_id) and chest.pos then
            -- Ensure the area is loaded before modifying
            core.load_area(chest.pos, chest.pos)

            local current_node = core.get_node(chest.pos)
            local param2 = current_node.param2 or 0

            -- Mark position as updating to prevent on_destruct from removing chest data
            skywars.mark_updating_pos(chest.pos)

            core.set_node(chest.pos, {name = placeholder_node, param2 = param2})

            skywars.unmark_updating_pos(chest.pos)
        end
    end
end
