--- Arena editor section registration for arena_lib.
-- Adds a "Skywars" section to the arena_lib editor hotbar (slot 6).
-- Provides tool items for opening the Flow GUI and placing template chests.

local S = skywars.T


--- Builds the list of editor items (editor tool + template setters) for an arena.
---@param arena table
---@return table
local function get_editor_items(arena)
	local items = {
		"skywars:editor_tool",
	}

	-- Sort templates by max preciousness for consistent order
	local templates = arena.chest_templates or {}
	local sorted_ids = {}
	for id in pairs(templates) do
		table.insert(sorted_ids, id)
	end
	table.sort(sorted_ids, function(a, b)
		local max_a = templates[a].max_preciousness
		local max_b = templates[b].max_preciousness
		if max_a == max_b then
			return templates[a].name < templates[b].name
		end
		return max_a < max_b
	end)

	for _, id in ipairs(sorted_ids) do
		local setter = skywars.create_template_setter_item(arena, id)
		if setter then
			table.insert(items, setter)
		end
		if #items >= 6 then break end
	end

	return items
end


--- Refreshes the editor hotbar for a player in editor mode.
-- Updates hotbar slots 1-6 with current editor items.
---@param player ObjectRef
---@param arena? table Optional arena to use, otherwise fetched from player meta
function skywars.refresh_editor_hotbar(player, arena)
	if not player then return end

	local p_meta = player:get_meta()
	local editor_mod = p_meta:get_string("arena_lib_editor.mod")
	if editor_mod ~= "skywars" then return end

	if not arena then
		local editor_arena = p_meta:get_string("arena_lib_editor.arena")
		if editor_arena == "" then return end
		local _, a = arena_lib.get_arena_by_name("skywars", editor_arena)
		arena = a
	end
	if not arena then return end

	local inv = player:get_inventory()
	local items = get_editor_items(arena)

	-- Update hotbar slots (first 7 slots)
	for i = 1, 7 do
		local item = items[i]
		if item then
			inv:set_stack("main", i, ItemStack(item))
		else
			inv:set_stack("main", i, ItemStack(""))
		end
	end
end



arena_lib.register_editor_section("skywars", {
	name = "Skywars",
	icon = "skywars_icon.png",

	give_items = function(itemstack, user, arena)
		return get_editor_items(arena)
	end
})


--- Tool item that opens the main Skywars editor Flow GUI.
core.register_craftitem("skywars:editor_tool", {
	description = S("Skywars Editor"),
	inventory_image = "skywars_icon.png",
	stack_max = 1,

	on_drop = function()
		return nil
	end,

	on_use = function(itemstack, player, pointed_thing)
		if not player then return end

		local pl_name = player:get_player_name()
		local p_meta = player:get_meta()

		local editor_mod = p_meta:get_string("arena_lib_editor.mod")
		local editor_arena = p_meta:get_string("arena_lib_editor.arena")

		if editor_mod ~= "skywars" or editor_arena == "" then
			skywars.print_msg(pl_name, S("You must be in arena editor mode."))
			return itemstack
		end

		skywars.show_editor_gui(player, editor_arena)
		return itemstack
	end,

	on_secondary_use = function(itemstack, player, pointed_thing)
		if not player then return end

		local pl_name = player:get_player_name()
		local p_meta = player:get_meta()
		local editor_mod = p_meta:get_string("arena_lib_editor.mod")
		local editor_arena = p_meta:get_string("arena_lib_editor.arena")

		if editor_mod ~= "skywars" or editor_arena == "" then
			skywars.print_msg(pl_name, S("You must be in arena editor mode."))
			return itemstack
		end

		skywars.show_editor_gui(player, editor_arena)
		return itemstack
	end,
})
