local function starts_with_mod(name, modname)
    return type(name) == "string" and name:sub(1, #modname + 1) == (modname .. ":")
end

local function count_by_keyprefix(reg_table, modname)
    local n = 0
    for k, _ in pairs(reg_table or {}) do
        if starts_with_mod(k, modname) then n = n + 1 end
    end
    return n
end

local function count_abm_like(reg_table, modname)
    local n = 0
    for _, def in ipairs(reg_table or {}) do
        -- Prefer engine-provided origin hint if present
        if def.mod_origin == modname then
            n = n + 1
        else
            -- Heuristics fallback
            local hit = false
            if type(def.label) == "string" and def.label:find("^" .. modname .. "[:%-]") then
                hit = true
            else
                local function list_has_mod(list)
                    if type(list) ~= "table" then return false end
                    for _, nn in ipairs(list) do
                        if starts_with_mod(nn, modname) then return true end
                    end
                    return false
                end
                if list_has_mod(def.nodenames) or list_has_mod(def.targets) then
                    hit = true
                end
            end
            if hit then n = n + 1 end
        end
    end
    return n
end

local function get_counts(modname)
    -- Basic existence check
    local exists = minetest.get_modpath(modname) ~= nil

    local nodes      = count_by_keyprefix(minetest.registered_nodes, modname)
    local craftitems = count_by_keyprefix(minetest.registered_craftitems, modname)
    local tools      = count_by_keyprefix(minetest.registered_tools, modname)
    local entities   = count_by_keyprefix(minetest.registered_entities, modname)

    -- ABMs / LBMs
    local abms = count_abm_like(minetest.registered_abms, modname)
    local lbms = count_abm_like(minetest.registered_lbms, modname)

    return {
        exists = exists,
        nodes = nodes,
        craftitems = craftitems,
        tools = tools,
        entities = entities,
        abms = abms,
        lbms = lbms,
    }
end

local function format_report(modname, c)
    local hdr = (c.exists and "" or "[warning: mod not found on server] ")
    return string.format(
        "%sFeatures for '%s': nodes=%d, craftitems=%d, tools=%d, entities=%d, abms=%d, lbms=%d",
        hdr, modname, c.nodes, c.craftitems, c.tools, c.entities, c.abms, c.lbms
    )
end

minetest.register_chatcommand("detailedfeatures", {
    params = "<modname>",
    description = "Show counts of nodes, craftitems, tools, entities, ABMs, LBMs for a mod",
    privs = {server = false},
    func = function(name, param)
        local modname = (param or ""):match("^%s*(.-)%s*$")
        if modname == "" then
            return false, "Usage: /detailedfeatures <modname>"
        end
        local c = get_counts(modname)
        local msg = format_report(modname, c)
        minetest.chat_send_player(name, msg)
    end,
})

-- Alias
minetest.register_chatcommand("df", {
    params = "<modname>",
    description = "Alias of /detailedfeatures",
    func = function(name, param)
        return minetest.registered_chatcommands["detailedfeatures"].func(name, param)
    end,
})

