local MOD = minetest.get_current_modname()
local created = 0
local map_orig_to_fake = {}

local function tcopy(v)
    if table.copy then return table.copy(v) end
    if type(v) ~= "table" then return v end
    local r = {}
    for k, x in pairs(v) do r[k] = (type(x) == "table") and tcopy(x) or x end
    return r
end

-- fake_solids:<srcmod>__<basename>_fake
local function fakename(fullname)
    local mod, base = fullname:match("^([^:]+):(.+)$")
    if not mod or not base then return MOD .. ":invalid_fake" end
    base = base:gsub(":", "_")
    return ("%s:%s__%s_fake"):format(MOD, mod, base)
end

local function is_solid(def, name)
    if name == "ignore" then return false end
    if def.liquidtype and def.liquidtype ~= "none" then return false end
    if def.drawtype == "airlike" then return false end
    if def.walkable == false then return false end
    return true
end

local function clone_def(src, srcname)
    local def = tcopy(src)
    def.description = (src.description or srcname) .. " (Fake)"
    def.tiles = tcopy(src.tiles)
    def.overlay_tiles = tcopy(src.overlay_tiles)
    def.special_tiles = tcopy(src.special_tiles)
    def.drawtype = src.drawtype
    def.paramtype = src.paramtype
    def.paramtype2 = src.paramtype2
    def.selection_box = src.selection_box
    def.node_box = src.node_box
    def.mesh = src.mesh
    def.palette = src.palette
    def.use_texture_alpha = src.use_texture_alpha
    def.visual_scale = src.visual_scale
    def.connected_sides = src.connected_sides
    def.connects_to = src.connects_to
    def.waving = src.waving
    def.light_source = src.light_source
    def.sounds = src.sounds

    def.walkable = false
    def.climbable = false
    def.pointable = true
    def.diggable = true
    def.buildable_to = src.buildable_to or false
    def.sunlight_propagates = true
    def.drop = nil

    def.groups = tcopy(src.groups or {})
    def.groups.falling_node = nil
    def.groups.not_in_creative_inventory = nil

    def.on_construct = nil
    def.after_construct = function(pos)
        minetest.get_meta(pos):set_string("infotext", "Fake of " .. srcname)
    end
    def.on_destruct = nil
    def.after_destruct = nil
    def.on_receive_fields = nil
    def.on_timer = nil
    def.on_rightclick = nil
    def.on_place = nil
    def.on_punch = nil
    def.on_use = nil
    def.on_blast = nil
    def.on_rotate = nil
    def.on_dig = nil
    def.on_burn = nil
    def.after_dig_node = nil

    return def
end

-- snapshot nodes from mods that loaded before us
local snapshot = {}
for name, def in pairs(minetest.registered_nodes) do
    snapshot[name] = def
end

-- register fake nodes
for name, def in pairs(snapshot) do
    if name:find(":")
    and not name:find("^" .. MOD .. ":")
    and not name:find("_fake$")
    and is_solid(def, name) then
        local newname = fakename(name)
        if not minetest.registered_nodes[newname] then
            minetest.register_node(newname, clone_def(def, name))
            map_orig_to_fake[name] = newname
            created = created + 1
        end
    end
end

-- shapeless crafts: original + any leaves -> fake
for orig, fake in pairs(map_orig_to_fake) do
    minetest.register_craft({
        type = "shapeless",
        output = fake,
        recipe = { orig, "group:leaves" },
    })
end

minetest.log("action", ("[%s] created %d fake nodes and crafts."):format(MOD, created))

minetest.register_chatcommand("fake_of", {
    params = "<mod:item>",
    description = "Show the fake_solids node name for a real node",
    func = function(_, param)
        local real = (param or ""):match("^%s*(.-)%s*$")
        if real == "" then return false, "Usage: /fake_of <mod:item>" end
        local fake = map_orig_to_fake[real]
        if not fake then return false, "No fake registered for " .. real end
        return true, fake
    end,
})

minetest.register_chatcommand("fakecount", {
    description = "Show number of fake nodes created",
    func = function() return true, "fake_solids created " .. created .. " nodes." end,
})
