--[[ Defines all of the stats for monsters. Monsters have base stats, 
per level stats, and personality boosts. The per level stat is 
multiplied by the monster level, added to the base stat, multiplied by 
the personality boost, and then rounded down to the nearest 
whole number.

Monster stats follow a few guidelines to help balance them. The max 
level for any monster is 100. The max health, attack, defense, or 
agility at level 100 should never exceed 500 (exluding personality
boost). The max energy at level 100 should never exceed 20.

"EXP per level" is used to determine both how much EXP is needed
for each level up and how much EXP to give when defeated. A larger
number means it takes more EXP for each level up and more EXP is
given when that monster is defeated. A value from 75 to 99 is
considered low. A value of 100 is considered average. A value from
100 to 125 is considered high.]]--

-- Local namespace
local monsters = {}

-- Import files
local mod_path = minetest.get_modpath("zoonami")
local group = dofile(mod_path .. "/lua/group.lua")
local move_stats = dofile(mod_path .. "/lua/move_stats.lua")

-- Monster personalities
monsters.personalities = {"Bulky", "Fierce", "Sturdy", "Hasty", "Noble", "Robust", "Joyful", "Mighty", "Nimble", "Stealthy", "Balanced"}

-- Monster personality boost
monsters.personality_boost = {
	bulky = {health = 1.1, attack = 1, defense = 1, agility = 1},
	fierce = {health = 1, attack = 1.1, defense = 1, agility = 1},
	sturdy = {health = 1, attack = 1, defense = 1.1, agility = 1},
	hasty = {health = 1, attack = 1, defense = 1, agility = 1.1},
	noble = {health = 1.06, attack = 1.06, defense = 1, agility = 1},
	robust = {health = 1.06, attack = 1, defense = 1.06, agility = 1},
	joyful = {health = 1.06, attack = 1, defense = 1, agility = 1.06},
	mighty = {health = 1, attack = 1.06, defense = 1.06, agility = 1},
	nimble = {health = 1, attack = 1.06, defense = 1, agility = 1.06},
	stealthy = {health = 1, attack = 1, defense = 1.06, agility = 1.06},
	balanced = {health = 1.04, attack = 1.04, defense = 1.04, agility = 1.04},
}

-- Generates a new monster's stats
function monsters.generate(asset_name, level, presets)
	local base_stats = monsters.stats[asset_name]
	local personality = presets.personality or monsters.personalities[math.random(#monsters.personalities)]
	local stat_boost = monsters.personality_boost[string.lower(personality)]
	local generated_monster = {
		nickname = presets.nickname,
		asset_name = base_stats.asset_name,
		prisma_id = presets.prisma_id,
		level = level,
		health = math.floor((base_stats.health + base_stats.health_per_level * level) * stat_boost.health),
		energy = math.min(math.ceil((level + 1) / 10) * 2, base_stats.energy_cap),
		exp_total = level * level * base_stats.exp_per_level,
		personality = personality,
		move_pool = {},
		moves = {},
	}
	for key, value in pairs(base_stats.level_up_moves) do
		if level >= key then
			table.insert(generated_monster.move_pool, value)
		end
	end
	if presets.taught_moves then
		for key, value in pairs(base_stats.taught_moves) do
			local learn_move_chance = math.random(6)
			if generated_monster.energy >= move_stats[value].energy and learn_move_chance == 1 then
				table.insert(generated_monster.move_pool, value)
			end
		end
	end
	generated_monster.moves = table.copy(generated_monster.move_pool)
	table.shuffle(generated_monster.moves)
	generated_monster.moves = {unpack(generated_monster.moves, 1, 4)}
	return generated_monster
end

-- Loads a monster's stats and sanity checks existing stats
function monsters.load_stats(monster)
	monster.level = monster.level or 0
	monster.health = monster.health or 0
	monster.energy = monster.energy or 0
	monster.exp_total = monster.exp_total or 0
	monster.personality = monsters.personality_boost[string.lower(monster.personality or "")] and monster.personality or "Balanced"
	monster.base = monsters.stats[monster.asset_name]
	monster.boost = monsters.personality_boost[string.lower(monster.personality)]
	monster.name = monster.base.name
	monster.type = monster.base.type
	monster.max_health = math.floor((monster.base.health + monster.base.health_per_level * monster.level) * monster.boost.health)
	monster.max_energy = math.min(math.ceil((monster.level + 1) / 10) * 2, monster.base.energy_cap)
	monster.attack = math.floor((monster.base.attack + monster.base.attack_per_level * monster.level) * monster.boost.attack)
	monster.defense = math.floor((monster.base.defense + monster.base.defense_per_level * monster.level) * monster.boost.defense)
	monster.agility = math.floor((monster.base.agility + monster.base.agility_per_level * monster.level) * monster.boost.agility)
	return monster
end

-- Saves a monster's stats
function monsters.save_stats(monster)
	local saved_stats = {
		asset_name = monster.asset_name,
		nickname = monster.nickname,
		starter_monster = monster.starter_monster,
		prisma_id = monster.prisma_id,
		level = monster.level,
		health = monster.health,
		energy = monster.energy,
		exp_total = monster.exp_total,
		personality = monster.personality,
		move_pool = monster.move_pool,
		moves = monster.moves,
	}
	return saved_stats
end

-- Monsters
monsters.stats = {}

monsters.stats.armite = {
	name = "Armite",
	asset_name = "armite",
	type = "Warrior",
	color = "#EBDF6EFF",
	tier = 1,
	energy_cap = 12,
	health = 12,
	health_per_level = 4.6,
	attack = 10,
	attack_per_level = 1.6,
	defense = 15,
	defense_per_level = 4.7,
	agility = 11,
	agility_per_level = 1.1,
	exp_per_level = 125,
	level_up_moves = {[1] = "punch", [5] = "crusher", [13] = "dropkick", [17] = "iron_fist", [25] = "bulldoze"},
	taught_moves = {"barrier", "ground_pound", "guard", "rage"},
	rewards = {"zoonami:move_book_guard 2", "zoonami:move_book_rage 2", "zoonami:mystery_move_book 5"},
	rewards_chance = 1,
	spawn_min_level = 65,
	spawn_max_level = 75,
	spawn_on = {group.cracky},
	spawn_chance = "legendary",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 3.5, y = 3.5},
	spawn_collisionbox = {-0.85, -1.65, -0.85, 0.85, 1.1, 0.85},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
}

monsters.stats.bebop = {
	name = "Bebop",
	asset_name = "bebop",
	type = "Robotic",
	color = "#B6B6B6FF",
	tier = 4,
	energy_cap = 8,
	health = 11,
	health_per_level = 2.1,
	attack = 15,
	attack_per_level = 2.5,
	defense = 16,
	defense_per_level = 2.7,
	agility = 15,
	agility_per_level = 2.4,
	exp_per_level = 102,
	level_up_moves = {[1] = "electrocute", [14] = "power_surge", [16] = "gear_shift"},
	taught_moves = {"force_field", "split", "refresh", "ultrasonic"},
	morph_level = 25,
	morphs_into = "mechabyte",
	rewards = {"zoonami:orange_berry 1"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.stone},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground"},
	spawn_visual_size = {x = 1.25, y = 1.25},
	spawn_collisionbox = {-0.35, -0.55, -0.35, 0.35, 0.1, 0.35},
	spawn_walk_velocity = 1,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.bellglum = {
	name = "Bellglum",
	asset_name = "bellglum",
	type = "Plant",
	color = "#48B5D6FF",
	tier = 4,
	energy_cap = 9,
	health = 12,
	health_per_level = 2.2,
	attack = 11,
	attack_per_level = 2.0,
	defense = 17,
	defense_per_level = 2.8,
	agility = 15,
	agility_per_level = 2.9,
	exp_per_level = 98,
	level_up_moves = {[1] = "vine_wrap", [8] = "pellet", [12] = "downpour", [17] = "shadow_orb", [23] = "thissle_missle", [29] = "fissure"},
	taught_moves = {"barrier", "restore", "void"},
	rewards = {"zoonami:blue_berry 2", "zoonami:red_berry 2", "zoonami:blue_tulip 2"},
	rewards_chance = 3,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.crumbly},
	spawn_by = {group.flora},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.35, -0.85, -0.35, 0.35, 0.2, 0.35},
	spawn_walk_velocity = 1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.brilly = {
	name = "Brilly",
	asset_name = "brilly",
	type = "Plant",
	color = "#2BB988FF",
	tier = 4,
	energy_cap = 9,
	health = 15,
	health_per_level = 3.2,
	attack = 14,
	attack_per_level = 2.5,
	defense = 12,
	defense_per_level = 2.2,
	agility = 12,
	agility_per_level = 2.0,
	exp_per_level = 83,
	level_up_moves = {[1] = "spore_storm", [5] = "bubble_stream", [9] = "purify", [13] = "grass_blade", [17] = "downpour"},
	taught_moves = {"boil", "swirl", "thorns"},
	rewards = {"zoonami:green_berry 2"},
	rewards_chance = 3,
	spawn_min_level = 8,
	spawn_max_level = 20,
	spawn_on = {group.water},
	spawn_by = {group.flora},
	spawn_chance = "common",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.75, y = 1.75},
	spawn_collisionbox = {-0.35, -0.7, -0.35, 0.35, 0.1, 0.35},
	spawn_walk_velocity = 0.3,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.brontore = {
	name = "Brontore",
	asset_name = "brontore",
	type = "Reptile",
	color = "#23B6C9FF",
	tier = 3,
	energy_cap = 11,
	health = 15,
	health_per_level = 2.4,
	attack = 13,
	attack_per_level = 2.1,
	defense = 14,
	defense_per_level = 2.9,
	agility = 11,
	agility_per_level = 1.7,
	exp_per_level = 120,
	level_up_moves = {[1] = "tail_swipe", [8] = "roar", [14] = "spikes", [24] = "fissure"},
	taught_moves = {"ground_pound", "guard"},
	rewards = {"zoonami:green_berry 1"},
	rewards_chance = 3,
	spawn_min_level = 25,
	spawn_max_level = 40,
	spawn_on = {group.snowy},
	spawn_chance = "mythical",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 3, y = 3},
	spawn_collisionbox = {-0.85, -1.35, -0.85, 0.85, 1.1, 0.85},
	spawn_walk_velocity = 0.6,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.bubblebee = {
	name = "Bubblebee",
	asset_name = "bubblebee",
	type = "Insect",
	color = "#EAE356FF",
	tier = 4,
	energy_cap = 9,
	health = 13,
	health_per_level = 2.3,
	attack = 13,
	attack_per_level = 2.7,
	defense = 14,
	defense_per_level = 2.4,
	agility = 16,
	agility_per_level = 2.7,
	exp_per_level = 94,
	level_up_moves = {[1] = "poison_sting", [11] = "life_drain", [23] = "bug_bite"},
	taught_moves = {"barrier", "rage", "refresh", "toxin"},
	rewards = {"zoonami:green_berry 1", "zoonami:yellow_rug 2"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.grass},
	spawn_by = {group.flower},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1, y = 1},
	spawn_collisionbox = {-0.3, -0.5, -0.3, 0.3, 0.2, 0.3},
	spawn_walk_velocity = 1,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.buderfli = {
	name = "Buderfli",
	asset_name = "buderfli",
	type = "Insect",
	color = "#3AD864FF",
	tier = 4,
	energy_cap = 10,
	health = 13,
	health_per_level = 2.6,
	attack = 12,
	attack_per_level = 2.2,
	defense = 14,
	defense_per_level = 2.6,
	agility = 17,
	agility_per_level = 2.7,
	exp_per_level = 108,
	level_up_moves = {[1] = "poison_sting", [9] = "spore_storm", [16] = "grass_blade", [25] = "bug_bite"},
	taught_moves = {"restore", "thorns", "whirlwind"},
	rewards = {"zoonami:green_berry 1", "zoonami:daisy 2"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 20,
	spawn_on = {group.grass},
	spawn_by = {group.leaves},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.25, y = 1.25},
	spawn_collisionbox = {-0.3, -0.7, -0.3, 0.3, 0.15, 0.3},
	spawn_walk_velocity = 1,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.burrlock = {
	name = "Burrlock",
	asset_name = "burrlock",
	type = "Plant",
	color = "#30D35CFF",
	tier = 4,
	energy_cap = 7,
	health = 10,
	health_per_level = 2.4,
	attack = 18,
	attack_per_level = 2.6,
	defense = 16,
	defense_per_level = 2.3,
	agility = 13,
	agility_per_level = 2.5,
	exp_per_level = 90,
	level_up_moves = {[1] = "prickle", [8] = "snare", [14] = "grass_blade"},
	taught_moves = {"guard", "pinpoint", "refresh", "thorns"},
	morph_level = 20,
	morphs_into = "kackaburr",
	rewards = {"zoonami:green_berry 1", "zoonami:zinnia 2"},
	rewards_chance = 3,
	spawn_min_level = 1,
	spawn_max_level = 10,
	spawn_on = {group.grass},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.6, y = 1.6},
	spawn_collisionbox = {-0.35, -0.65, -0.35, 0.35, 0.3, 0.35},
	spawn_walk_velocity = 1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.cheerum = {
	name = "Cheerum",
	asset_name = "cheerum",
	type = "Spirit",
	color = "#FFF8BFFF",
	tier = 3,
	energy_cap = 11,
	health = 10,
	health_per_level = 2.0,
	attack = 16,
	attack_per_level = 2.6,
	defense = 14,
	defense_per_level = 2.6,
	agility = 12,
	agility_per_level = 2.2,
	exp_per_level = 105,
	level_up_moves = {[1] = "shine", [14] = "burnout", [23] = "sword_swipe", [27] = "harmony"},
	taught_moves = {"guard", "restore", "whirlwind"},
	rewards = {"zoonami:blue_berry 2", "zoonami:blue_tulip 4", "zoonami:daisy 4"},
	rewards_chance = 3,
	spawn_min_level = 40,
	spawn_max_level = 55,
	spawn_on = {group.grass, group.snowy},
	spawn_chance = "rare",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"sky"},
	spawn_visual_size = {x = 2.2, y = 2.2},
	spawn_collisionbox = {-0.6, -0.91, -0.6, 0.6, 0.6, 0.6},
	spawn_walk_velocity = 0.7,
	spawn_float = true,
}

monsters.stats.chickadee = {
	name = "Chickadee",
	asset_name = "chickadee",
	type = "Avian",
	color = "#FFF496FF",
	tier = 5,
	energy_cap = 6,
	health = 17,
	health_per_level = 3,
	attack = 10,
	attack_per_level = 2,
	defense = 22,
	defense_per_level = 3.3,
	agility = 5,
	agility_per_level = 1.9,
	exp_per_level = 100,
	level_up_moves = {[1] = "sing", [12] = "peck", [18] = "quill_drill", [26] = "harmony"},
	taught_moves = {"barrier", "guard", "refresh", "rest"},
	morph_level = 15,
	morphs_into = "spaero",
	rewards = {"zoonami:blue_berry 1", "zoonami:daisy 2", "zoonami:sunflower 2"},
	rewards_chance = 3,
	spawn_min_level = 1,
	spawn_max_level = 10,
	spawn_on = {group.grass},
	spawn_by = {group.tree},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1, y = 1},
	spawn_collisionbox = {-0.2, -0.4, -0.2, 0.2, 0.1, 0.2},
	spawn_walk_velocity = 0.8,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.dracken = {
	name = "Dracken",
	asset_name = "dracken",
	type = "Aquatic",
	color = "#D28694FF",
	tier = 3,
	energy_cap = 10,
	health = 11,
	health_per_level = 2.1,
	attack = 17,
	attack_per_level = 2.8,
	defense = 11,
	defense_per_level = 1.5,
	agility = 13,
	agility_per_level = 2.4,
	exp_per_level = 107,
	level_up_moves = {[1] = "constrict", [9] = "electrocute", [15] = "geyser", [26] = "venom_fangs"},
	taught_moves = {"guard", "rage"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 2,
	spawn_min_level = 35,
	spawn_max_level = 55,
	spawn_on = {group.water},
	spawn_chance = "rare",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.65, -0.9, -0.65, 0.65, 0.8, 0.65},
	spawn_walk_velocity = 1,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.dromite = {
	name = "Dromite",
	asset_name = "dromite",
	type = "Robotic",
	color = "#844997FF",
	tier = 3,
	energy_cap = 12,
	health = 11,
	health_per_level = 1.8,
	attack = 14,
	attack_per_level = 2.3,
	defense = 20,
	defense_per_level = 3.0,
	agility = 10,
	agility_per_level = 1.7,
	exp_per_level = 100,
	level_up_moves = {[1] = "crusher", [5] = "illusion", [13] = "iron_fist", [25] = "laser_beam"},
	taught_moves = {"force_field", "guard"},
	rewards = {"zoonami:orange_berry 1"},
	rewards_chance = 3,
	spawn_min_level = 20,
	spawn_max_level = 35,
	spawn_on = {group.stone},
	spawn_by = {group.crumbly},
	spawn_chance = "uncommon",
	spawn_time = {"night"},
	spawn_light = {"medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.6, -0.7, -0.6, 0.6, 0.4, 0.6},
	spawn_walk_velocity = 0.7,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.ekko = {
	name = "Ekko",
	asset_name = "ekko",
	type = "Avian",
	color = "#1C9678FF",
	tier = 4,
	energy_cap = 10,
	health = 11,
	health_per_level = 2.4,
	attack = 15,
	attack_per_level = 2.2,
	defense = 17,
	defense_per_level = 2.4,
	agility = 17,
	agility_per_level = 2.8,
	exp_per_level = 95,
	level_up_moves = {[1] = "gust", [13] = "life_drain", [17] = "shadow_orb", [27] = "twister"},
	taught_moves = {"force_field", "rest", "ultrasonic", "void"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 20,
	spawn_on = {group.stone},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground", "bedrock"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.4, -0.8, -0.4, 0.4, 0.3, 0.4},
	spawn_walk_velocity = 1.2,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.firefli = {
	name = "Firefli",
	asset_name = "firefli",
	type = "Fire",
	color = "#DB4343FF",
	tier = 4,
	energy_cap = 7,
	health = 14,
	health_per_level = 2.2,
	attack = 14,
	attack_per_level = 2.5,
	defense = 12,
	defense_per_level = 2.1,
	agility = 18,
	agility_per_level = 2.9,
	exp_per_level = 95,
	level_up_moves = {[1] = "scorch", [8] = "infestation", [15] = "burnout", [23] = "bug_bite"},
	taught_moves = {"flash_fire", "guard", "slice", "whirlwind"},
	rewards = {"zoonami:red_berry 1", "zoonami:light 2", "zoonami:zinnia 2"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.grass},
	spawn_chance = "common",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.1, y = 1.1},
	spawn_collisionbox = {-0.35, -0.7, -0.35, 0.35, 0.3, 0.35},
	spawn_walk_velocity = 0.5,
	spawn_drown = true,
	spawn_float = true,
}

monsters.stats.fleace = {
	name = "Fleace",
	asset_name = "fleace",
	type = "Spirit",
	color = "#FFFFFFFF",
	tier = 5,
	energy_cap = 11,
	health = 14,
	health_per_level = 2.3,
	attack = 14,
	attack_per_level = 2.3,
	defense = 14,
	defense_per_level = 2.3,
	agility = 14,
	agility_per_level = 2.3,
	exp_per_level = 100,
	level_up_moves = {[1] = "purify", [15] = "cleanse", [25] = "harmony"},
	taught_moves = {"guard", "rest", "restore"},
	rewards = {"zoonami:blue_berry 1", "zoonami:daisy 2", "zoonami:cloth 5"},
	rewards_chance = 3,
	spawn_min_level = 15,
	spawn_max_level = 30,
	spawn_on = {group.grass, group.snowy},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"sky"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.6, -0.85, -0.6, 0.6, 0.35, 0.6},
	spawn_walk_velocity = 0.7,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.foliedge = {
	name = "Foliedge",
	asset_name = "foliedge",
	type = "Insect",
	color = "#3F993BFF",
	tier = 4,
	energy_cap = 8,
	health = 15,
	health_per_level = 2.5,
	attack = 13,
	attack_per_level = 2.4,
	defense = 15,
	defense_per_level = 2.5,
	agility = 13,
	agility_per_level = 2.2,
	exp_per_level = 85,
	level_up_moves = {[1] = "pincer", [4] = "snare", [12] = "life_drain", [18] = "grass_blade"},
	taught_moves = {"refresh", "slice"},
	morph_level = 10,
	morphs_into = "buderfli",
	rewards = {"zoonami:green_berry 2", "zoonami:paper 3"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.grass},
	spawn_by = {group.tree},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1, y = 1},
	spawn_collisionbox = {-0.2, -0.4, -0.2, 0.2, 0.1, 0.2},
	spawn_walk_velocity = 1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.foxtail = {
	name = "Foxtail",
	asset_name = "foxtail",
	type = "Beast",
	color = "#FFA55AFF",
	tier = 5,
	energy_cap = 12,
	health = 11,
	health_per_level = 1.8,
	attack = 13,
	attack_per_level = 2.3,
	defense = 12,
	defense_per_level = 1.9,
	agility = 17,
	agility_per_level = 3.4,
	exp_per_level = 105,
	level_up_moves = {[1] = "roar", [6] = "prickle", [12] = "slash", [19] = "grass_blade", [23] = "razor_fang"},
	taught_moves = {"barrier", "void"},
	rewards = {"zoonami:red_berry 2", "zoonami:zinnia 4"},
	rewards_chance = 3,
	spawn_min_level = 20,
	spawn_max_level = 35,
	spawn_on = {group.sand},
	spawn_by = {group.tall_grass},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.7, y = 1.7},
	spawn_collisionbox = {-0.5, -0.77, -0.5, 0.5, 0.35, 0.5},
	spawn_walk_velocity = 1.0,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.freazel = {
	name = "Freazel",
	asset_name = "freazel",
	type = "Rodent",
	color = "#21BFFFFF",
	tier = 4,
	energy_cap = 8,
	health = 15,
	health_per_level = 2.5,
	attack = 15,
	attack_per_level = 2.7,
	defense = 13,
	defense_per_level = 2.3,
	agility = 12,
	agility_per_level = 2.0,
	exp_per_level = 101,
	level_up_moves = {[1] = "bite", [12] = "slash", [15] = "claw", [18] = "downpour", [23] = "razor_fang"},
	taught_moves = {"barrier", "swirl"},
	rewards = {"zoonami:orange_berry 1", "zoonami:blue_tulip 2", "zoonami:daisy 2"},
	rewards_chance = 3,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.snowy},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.35, -0.6, -0.35, 0.35, 0.3, 0.35},
	spawn_walk_velocity = 0.7,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.fuzall = {
	name = "Fuzall",
	asset_name = "fuzall",
	type = "Mutant",
	color = "#4F89FFFF",
	tier = 4,
	energy_cap = 6,
	health = 16,
	health_per_level = 2.4,
	attack = 12,
	attack_per_level = 2.6,
	defense = 11,
	defense_per_level = 2.3,
	agility = 16,
	agility_per_level = 2.7,
	exp_per_level = 95,
	level_up_moves = {[1] = "smog", [16] = "shadow_orb", [19] = "poison", [28] = "nightmare"},
	taught_moves = {"rest", "toxin", "void"},
	rewards = {"zoonami:red_berry 1", "zoonami:blue_rug 2", "zoonami:blue_tulip 2"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.snowy},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.9, y = 1.9},
	spawn_collisionbox = {-0.35, -0.8, -0.35, 0.35, 0.08, 0.35},
	spawn_walk_velocity = 1.1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.goopo = {
	name = "Goopo",
	asset_name = "goopo",
	type = "Aquatic",
	color = "#7C956EFF",
	tier = 5,
	energy_cap = 10,
	health = 10,
	health_per_level = 1.9,
	attack = 14,
	attack_per_level = 2.4,
	defense = 15,
	defense_per_level = 3.2,
	agility = 4,
	agility_per_level = 1.0,
	exp_per_level = 98,
	level_up_moves = {[1] = "electrocute", [10] = "downpour", [13] = "mudslide", [23] = "vortex"},
	taught_moves = {"barrier", "guard", "swirl"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 2,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.water},
	spawn_chance = "common",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground", "underground"},
	spawn_visual_size = {x = 1.3, y = 1.3},
	spawn_collisionbox = {-0.35, -0.6, -0.35, 0.35, 0.15, 0.35},
	spawn_walk_velocity = 0.5,
	spawn_sink = true,
	spawn_burn = true,
}

monsters.stats.grimlit = {
	name = "Grimlit",
	asset_name = "grimlit",
	type = "Fire",
	color = "#EFEBC2FF",
	tier = 4,
	energy_cap = 10,
	health = 14,
	health_per_level = 2.5,
	attack = 10,
	attack_per_level = 2.3,
	defense = 17,
	defense_per_level = 2.7,
	agility = 5,
	agility_per_level = 1.5,
	exp_per_level = 105,
	level_up_moves = {[1] = "fireball", [13] = "burnout", [21] = "man_melter"},
	taught_moves = {"flash_fire", "refresh", "void"},
	rewards = {"zoonami:red_berry 1", "zoonami:white_brick 5", "zoonami:zinnia 2"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.stone},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"dark"},
	spawn_height = {"underground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.3, -0.65, -0.3, 0.3, 0.3, 0.3},
	spawn_walk_velocity = 0.5,
	spawn_drown = true,
	spawn_float = true,
}

monsters.stats.groudile = {
	name = "Groudile",
	asset_name = "groudile",
	type = "Rock",
	color = "#DAB072FF",
	tier = 4,
	energy_cap = 7,
	health = 12,
	health_per_level = 2.0,
	attack = 14,
	attack_per_level = 2.5,
	defense = 12,
	defense_per_level = 2.1,
	agility = 18,
	agility_per_level = 2.8,
	exp_per_level = 107,
	level_up_moves = {[1] = "rockburst", [8] = "tail_swipe", [15] = "mudslide", [19] = "slash", [23] = "fissure"},
	taught_moves = {"diced", "refresh"},
	morph_level = 40,
	morphs_into = "rockadile",
	rewards = {"zoonami:orange_berry 2", "zoonami:orange_tile 5", "zoonami:tiger_lily 3"},
	rewards_chance = 2,
	spawn_min_level = 30,
	spawn_max_level = 40,
	spawn_on = {group.cracky},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"bedrock"},
	spawn_visual_size = {x = 1.6, y = 1.6},
	spawn_collisionbox = {-0.3, -0.7, -0.3, 0.3, 0.3, 0.3},
	spawn_walk_velocity = 0.8,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.howler = {
	name = "Howler",
	asset_name = "howler",
	type = "Avian",
	color = "#A5733DFF",
	tier = 3,
	energy_cap = 7,
	health = 15,
	health_per_level = 2.9,
	attack = 18,
	attack_per_level = 3,
	defense = 11,
	defense_per_level = 2.4,
	agility = 12,
	agility_per_level = 2.1,
	exp_per_level = 120,
	level_up_moves = {[1] = "swoop", [9] = "illusion", [15] = "quill_drill", [25] = "twister"},
	taught_moves = {"barrier", "split"},
	rewards = {"zoonami:blue_berry 1", "zoonami:green_berry 1", "zoonami:tiger_lily 2"},
	rewards_chance = 3,
	spawn_min_level = 15,
	spawn_max_level = 30,
	spawn_on = {group.snowy},
	spawn_by = {group.leaves},
	spawn_chance = "uncommon",
	spawn_time = {"night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.6, -0.91, -0.6, 0.6, 0.6, 0.6},
	spawn_walk_velocity = 0.5,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.hydrode = {
	name = "Hydrode",
	asset_name = "hydrode",
	type = "Aquatic",
	color = "#2A4955FF",
	tier = 4,
	energy_cap = 6,
	health = 13,
	health_per_level = 2.6,
	attack = 11,
	attack_per_level = 2.4,
	defense = 14,
	defense_per_level = 2.6,
	agility = 11,
	agility_per_level = 2.0,
	exp_per_level = 103,
	level_up_moves = {[1] = "aqua_jet", [8] = "rockburst", [15] = "power_surge", [27] = "vortex"},
	taught_moves = {"force_field", "ultrasonic"},
	rewards = {"zoonami:blue_berry 2"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 30,
	spawn_on = {group.water},
	spawn_chance = "uncommon",
	spawn_time = {"night"},
	spawn_light = {"dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.4, -0.75, -0.4, 0.4, 0.3, 0.4},
	spawn_walk_velocity = 0.3,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.hypnotile = {
	name = "Hypnotile",
	asset_name = "hypnotile",
	type = "Reptile",
	color = "#11AE39FF",
	tier = 3,
	energy_cap = 8,
	health = 12,
	health_per_level = 2.2,
	attack = 14,
	attack_per_level = 2.7,
	defense = 12,
	defense_per_level = 1.9,
	agility = 11,
	agility_per_level = 2.1,
	exp_per_level = 108,
	level_up_moves = {[1] = "constrict", [5] = "illusion", [15] = "spikes", [24] = "nightmare"},
	taught_moves = {"rest", "swirl", "toxin", "ultrasonic"},
	rewards = {"zoonami:green_berry 3"},
	rewards_chance = 3,
	spawn_min_level = 25,
	spawn_max_level = 40,
	spawn_on = {group.sand},
	spawn_chance = "rare",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.7, y = 1.7},
	spawn_collisionbox = {-0.5, -0.8, -0.5, 0.5, 0.7, 0.5},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.jetsuna = {
	name = "Jetsuna",
	asset_name = "jetsuna",
	type = "Aquatic",
	color = "#7EE8FAFF",
	tier = 5,
	energy_cap = 11,
	health = 12,
	health_per_level = 2.0,
	attack = 11,
	attack_per_level = 2.4,
	defense = 12,
	defense_per_level = 2.0,
	agility = 13,
	agility_per_level = 2.7,
	exp_per_level = 107,
	level_up_moves = {[1] = "pierce", [11] = "downpour", [14] = "geyser", [23] = "sword_swipe"},
	taught_moves = {"barrier", "pinpoint", "swirl"},
	rewards = {"zoonami:blue_berry 2"},
	rewards_chance = 3,
	spawn_min_level = 15,
	spawn_max_level = 30,
	spawn_on = {group.water},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.5, -0.8, -0.5, 0.5, 0.3, 0.5},
	spawn_walk_velocity = 1.0,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.kackaburr = {
	name = "Kackaburr",
	asset_name = "kackaburr",
	type = "Plant",
	color = "#30D35CFF",
	tier = 3,
	energy_cap = 9,
	health = 10,
	health_per_level = 2.3,
	attack = 18,
	attack_per_level = 3,
	defense = 16,
	defense_per_level = 2.2,
	agility = 17,
	agility_per_level = 2.4,
	exp_per_level = 125,
	level_up_moves = {[1] = "vine_wrap", [5] = "poison_sting", [16] = "spikes", [28] = "thissle_missle"},
	taught_moves = {"toxin"},
	rewards = {"zoonami:green_berry 3", "zoonami:sunflower 2"},
	rewards_chance = 4,
	spawn_min_level = 20,
	spawn_max_level = 35,
	spawn_on = {group.grass},
	spawn_chance = "rare",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.9, y = 1.9},
	spawn_collisionbox = {-0.6, -0.85, -0.6, 0.6, 0.5, 0.6},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.kazooka = {
	name = "Kazooka",
	asset_name = "kazooka",
	type = "Warrior",
	color = "#BA803EFF",
	tier = 3,
	energy_cap = 11,
	health = 10,
	health_per_level = 1.9,
	attack = 14,
	attack_per_level = 2.6,
	defense = 9,
	defense_per_level = 2.1,
	agility = 15,
	agility_per_level = 2.7,
	exp_per_level = 111,
	level_up_moves = {[1] = "punch", [12] = "dropkick", [19] = "iron_fist", [27] = "bulldoze"},
	taught_moves = {"guard", "rage", "rest"},
	rewards = {"zoonami:red_berry 3"},
	rewards_chance = 4,
	spawn_min_level = 35,
	spawn_max_level = 55,
	spawn_on = {group.sand},
	spawn_by = {group.tall_grass},
	spawn_chance = "rare",
	spawn_time = {"night"},
	spawn_light = {"medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.45, -0.9, -0.45, 0.45, 0.7, 0.45},
	spawn_walk_velocity = 0.8,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.maluga = {
	name = "Maluga",
	asset_name = "maluga",
	type = "Aquatic",
	color = "#6592F2FF",
	tier = 3,
	energy_cap = 13,
	health = 5,
	health_per_level = 3.2,
	attack = 14,
	attack_per_level = 2.1,
	defense = 16,
	defense_per_level = 2.8,
	agility = 5,
	agility_per_level = 1.7,
	exp_per_level = 100,
	level_up_moves = {[1] = "aqua_jet", [11] = "cleanse", [14] = "geyser", [17] = "mudslide", [23] = "vortex"},
	taught_moves = {"barrier", "rest"},
	rewards = {"zoonami:blue_berry 3", "zoonami:blue_roof 5"},
	rewards_chance = 4,
	spawn_min_level = 20,
	spawn_max_level = 40,
	spawn_on = {group.water},
	spawn_chance = "rare",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.7, -0.95, -0.7, 0.7, 0.6, 0.7},
	spawn_walk_velocity = 1,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.mechabyte = {
	name = "Mechabyte",
	asset_name = "mechabyte",
	type = "Robotic",
	color = "#808080FF",
	tier = 3,
	energy_cap = 10,
	health = 12,
	health_per_level = 2.6,
	attack = 14,
	attack_per_level = 1.8,
	defense = 16,
	defense_per_level = 3.7,
	agility = 13,
	agility_per_level = 1.9,
	exp_per_level = 111,
	level_up_moves = {[1] = "vice_grip", [4] = "punch", [15] = "slash", [18] = "iron_fist"},
	taught_moves = {"force_field", "restore", "ultrasonic"},
	rewards = {"zoonami:orange_berry 2"},
	rewards_chance = 2,
	spawn_min_level = 25,
	spawn_max_level = 40,
	spawn_on = {group.stone},
	spawn_chance = "rare",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground"},
	spawn_visual_size = {x = 2.2, y = 2.2},
	spawn_collisionbox = {-0.6, -0.95, -0.6, 0.6, 0.6, 0.6},
	spawn_walk_velocity = 0.7,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.merin = {
	name = "Merin",
	asset_name = "merin",
	type = "Rodent",
	color = "#37E5CBFF",
	tier = 4,
	energy_cap = 10,
	health = 10,
	health_per_level = 2.0,
	attack = 17,
	attack_per_level = 2.2,
	defense = 16,
	defense_per_level = 2.5,
	agility = 19,
	agility_per_level = 2.8,
	exp_per_level = 100,
	level_up_moves = {[1] = "gnaw", [7] = "tail_swipe", [12] = "spikes", [15] = "claw", [24] = "razor_fang"},
	taught_moves = {"rest", "void"},
	rewards = {"zoonami:orange_berry 1", "zoonami:tiger_lily 2"},
	rewards_chance = 3,
	spawn_min_level = 1,
	spawn_max_level = 10,
	spawn_on = {group.grass, group.sand},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1, y = 1},
	spawn_collisionbox = {-0.3, -0.45, -0.3, 0.3, 0.1, 0.3},
	spawn_walk_velocity = 1.1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.metalance = {
	name = "Metalance",
	asset_name = "metalance",
	type = "Warrior",
	color = "#ADADADFF",
	tier = 3,
	energy_cap = 10,
	health = 13,
	health_per_level = 2.4,
	attack = 16,
	attack_per_level = 2.5,
	defense = 17,
	defense_per_level = 2.6,
	agility = 10,
	agility_per_level = 2.1,
	exp_per_level = 95,
	level_up_moves = {[1] = "strike", [16] = "dropkick", [24] = "sword_swipe"},
	taught_moves = {"guard", "barrier", "diced"},
	rewards = {"zoonami:orange_berry 3"},
	rewards_chance = 4,
	spawn_min_level = 15,
	spawn_max_level = 30,
	spawn_on = {group.stone},
	spawn_chance = "rare",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.45, -0.67, -0.45, 0.45, 0.4, 0.45},
	spawn_walk_velocity = 1.1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.moltra = {
	name = "Moltra",
	asset_name = "moltra",
	type = "Fire",
	color = "#FDB54EFF",
	tier = 5,
	energy_cap = 9,
	health = 12,
	health_per_level = 2.3,
	attack = 12,
	attack_per_level = 2.2,
	defense = 16,
	defense_per_level = 2.7,
	agility = 13,
	agility_per_level = 2.3,
	exp_per_level = 88,
	level_up_moves = {[1] = "scorch", [4] = "tail_swipe", [9] = "rockburst", [14] = "burnout", [18] = "spikes"},
	taught_moves = {"inferno", "rage"},
	rewards = {"zoonami:red_berry 2"},
	rewards_chance = 2,
	spawn_min_level = 15,
	spawn_max_level = 30,
	spawn_on = {group.sand, group.stone},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"sky"},
	spawn_visual_size = {x = 2.1, y = 2.1},
	spawn_collisionbox = {-0.4, -0.95, -0.4, 0.4, 0.3, 0.4},
	spawn_walk_velocity = 0.8,
	spawn_float = true,
}

monsters.stats.muddle = {
	name = "Muddle",
	asset_name = "muddle",
	type = "Rock",
	color = "#AF9774FF",
	tier = 4,
	energy_cap = 10,
	health = 13,
	health_per_level = 2.6,
	attack = 15,
	attack_per_level = 2.6,
	defense = 14,
	defense_per_level = 2.4,
	agility = 12,
	agility_per_level = 1.8,
	exp_per_level = 103,
	level_up_moves = {[1] = "boulder_roll", [5] = "punch", [12] = "downpour", [14] = "mudslide", [22] = "fissure"},
	taught_moves = {"ground_pound"},
	rewards = {"zoonami:orange_berry 1", "zoonami:red_roof 5"},
	rewards_chance = 4,
	spawn_min_level = 15,
	spawn_max_level = 25,
	spawn_on = {group.stone},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground", "underground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.45, -0.65, -0.45, 0.45, 0.2, 0.45},
	spawn_walk_velocity = 1.1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.nubble = {
	name = "Nubble",
	asset_name = "nubble",
	type = "Rock",
	color = "#FFC69EFF",
	tier = 4,
	energy_cap = 9,
	health = 12,
	health_per_level = 2.5,
	attack = 13,
	attack_per_level = 2.3,
	defense = 16,
	defense_per_level = 2.7,
	agility = 12,
	agility_per_level = 2.0,
	exp_per_level = 98,
	level_up_moves = {[1] = "pellet", [4] = "prickle", [14] = "mudslide", [17] = "slash"},
	taught_moves = {"barrier", "guard", "pinpoint"},
	morph_level = 15,
	morphs_into = "muddle",
	rewards = {"zoonami:orange_berry 2", "zoonami:daisy 4"},
	rewards_chance = 4,
	spawn_min_level = 5,
	spawn_max_level = 20,
	spawn_on = {group.stone},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.25, y = 1.25},
	spawn_collisionbox = {-0.25, -0.45, -0.25, 0.25, 0.2, 0.25},
	spawn_walk_velocity = 1.1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.pumish = {
	name = "Pumish",
	asset_name = "pumish",
	type = "Aquatic",
	color = "#CD3131FF",
	tier = 4,
	energy_cap = 7,
	health = 13,
	health_per_level = 2.2,
	attack = 10,
	attack_per_level = 2.3,
	defense = 15,
	defense_per_level = 2.6,
	agility = 12,
	agility_per_level = 2.5,
	exp_per_level = 103,
	level_up_moves = {[1] = "high_tide", [11] = "afterburn", [15] = "geyser", [19] = "burnout"},
	taught_moves = {"boil", "restore"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 3,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.water},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.75, y = 1.75},
	spawn_collisionbox = {-0.5, -0.75, -0.5, 0.5, 0.2, 0.5},
	spawn_walk_velocity = 0.3,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.rampede = {
	name = "Rampede",
	asset_name = "rampede",
	type = "Reptile",
	color = "#6E649CFF",
	tier = 3,
	energy_cap = 8,
	health = 11,
	health_per_level = 1.9,
	attack = 15,
	attack_per_level = 2.7,
	defense = 12,
	defense_per_level = 2.2,
	agility = 12,
	agility_per_level = 2.3,
	exp_per_level = 117,
	level_up_moves = {[1] = "tail_swipe", [9] = "smog", [17] = "venom_fangs", [24] = "bulldoze"},
	taught_moves = {"force_field", "void"},
	rewards = {"zoonami:green_berry 2"},
	rewards_chance = 3,
	spawn_min_level = 20,
	spawn_max_level = 35,
	spawn_on = {group.stone},
	spawn_chance = "mythical",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground", "bedrock"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.5, -0.9, -0.5, 0.5, 0.5, 0.5},
	spawn_walk_velocity = 1.2,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.rattler = {
	name = "Rattler",
	asset_name = "rattler",
	type = "Reptile",
	color = "#D1B710FF",
	tier = 5,
	energy_cap = 9,
	health = 11,
	health_per_level = 2.3,
	attack = 13,
	attack_per_level = 2.2,
	defense = 12,
	defense_per_level = 2.2,
	agility = 16,
	agility_per_level = 2.8,
	exp_per_level = 105,
	level_up_moves = {[1] = "constrict", [8] = "afterburn", [13] = "life_drain", [22] = "venom_fangs"},
	taught_moves = {"rage", "refresh"},
	morph_level = 25,
	morphs_into = "hypnotile",
	rewards = {"zoonami:green_berry 1", "zoonami:yellow_tile 5"},
	rewards_chance = 3,
	spawn_min_level = 10,
	spawn_max_level = 20,
	spawn_on = {group.sand},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.3, y = 1.3},
	spawn_collisionbox = {-0.3, -0.53, -0.3, 0.3, 0.35, 0.3},
	spawn_walk_velocity = 0.6,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.rockade = {
	name = "Rockade",
	asset_name = "rockade",
	type = "Rock",
	color = "#635137FF",
	tier = 2,
	energy_cap = 12,
	health = 12,
	health_per_level = 1.8,
	attack = 17,
	attack_per_level = 2.9,
	defense = 13,
	defense_per_level = 2.6,
	agility = 12,
	agility_per_level = 1.6,
	exp_per_level = 125,
	level_up_moves = {[1] = "boulder_roll", [13] = "mudslide", [17] = "iron_fist", [25] = "fissure"},
	taught_moves = {"ground_pound", "guard"},
	rewards = {"zoonami:move_book_barrier 2", "zoonami:move_book_ground_pound 2", "zoonami:mystery_move_book 3"},
	rewards_chance = 1,
	spawn_min_level = 55,
	spawn_max_level = 65,
	spawn_on = {group.cracky},
	spawn_chance = "mythical",
	spawn_time = {"day", "night"},
	spawn_light = {"dark"},
	spawn_height = {"bedrock"},
	spawn_visual_size = {x = 3, y = 3},
	spawn_collisionbox = {-0.85, -1.2, -0.85, 0.85, 1.1, 0.85},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
}

monsters.stats.rockadile = {
	name = "Rockadile",
	asset_name = "rockadile",
	type = "Rock",
	color = "#C9A37FFF",
	tier = 3,
	energy_cap = 9,
	health = 13,
	health_per_level = 2.1,
	attack = 17,
	attack_per_level = 2.8,
	defense = 13,
	defense_per_level = 2.1,
	agility = 10,
	agility_per_level = 1.8,
	exp_per_level = 107,
	level_up_moves = {[1] = "roar", [7] = "boulder_roll", [15] = "spikes", [21] = "venom_fangs", [26] = "fissure"},
	taught_moves = {"rage", "rest"},
	rewards = {"zoonami:orange_berry 2", "zoonami:red_tile 5", "zoonami:zinnia 3"},
	rewards_chance = 2,
	spawn_min_level = 40,
	spawn_max_level = 55,
	spawn_on = {group.stone},
	spawn_chance = "rare",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"bedrock"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.7, -0.95, -0.7, 0.7, 0.4, 0.7},
	spawn_walk_velocity = 0.6,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.ruffalo = {
	name = "Ruffalo",
	asset_name = "ruffalo",
	type = "Beast",
	color = "#493526FF",
	tier = 3,
	energy_cap = 6,
	health = 14,
	health_per_level = 2.6,
	attack = 19,
	attack_per_level = 3.3,
	defense = 12,
	defense_per_level = 1.9,
	agility = 11,
	agility_per_level = 1.9,
	exp_per_level = 120,
	level_up_moves = {[1] = "stomp", [15] = "slash", [27] = "bulldoze"},
	taught_moves = {"ground_pound", "rage"},
	rewards = {"zoonami:red_berry 2", "zoonami:orange_rug 2"},
	rewards_chance = 4,
	spawn_min_level = 25,
	spawn_max_level = 40,
	spawn_on = {group.grass},
	spawn_by = {group.tall_grass},
	spawn_chance = "rare",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2.5, y = 2.5},
	spawn_collisionbox = {-0.9, -1.2, -0.9, 0.9, 0.5, 0.9},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.ruptore = {
	name = "Ruptore",
	asset_name = "ruptore",
	type = "Reptile",
	color = "#007A7FFF",
	tier = 3,
	energy_cap = 8,
	health = 13,
	health_per_level = 2.1,
	attack = 16,
	attack_per_level = 3.0,
	defense = 12,
	defense_per_level = 1.9,
	agility = 11,
	agility_per_level = 1.8,
	exp_per_level = 120,
	level_up_moves = {[1] = "stomp", [12] = "spikes", [15] = "slash", [21] = "venom_fangs"},
	taught_moves = {"flash_fire", "rage"},
	rewards = {"zoonami:green_berry 3"},
	rewards_chance = 3,
	spawn_min_level = 25,
	spawn_max_level = 40,
	spawn_on = {group.stone},
	spawn_chance = "mythical",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground", "bedrock"},
	spawn_visual_size = {x = 3, y = 3},
	spawn_collisionbox = {-0.85, -1.35, -0.85, 0.85, 1.1, 0.85},
	spawn_walk_velocity = 0.6,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.scallapod = {
	name = "Scallapod",
	asset_name = "scallapod",
	type = "Aquatic",
	color = "#D8C791FF",
	tier = 4,
	energy_cap = 8,
	health = 15,
	health_per_level = 2.4,
	attack = 10,
	attack_per_level = 2.6,
	defense = 16,
	defense_per_level = 2.8,
	agility = 5,
	agility_per_level = 1.0,
	exp_per_level = 125,
	level_up_moves = {[1] = "pincer", [9] = "bubble_stream", [15] = "geyser", [17] = "life_drain"},
	taught_moves = {"diced", "guard", "slice"},
	rewards = {"zoonami:blue_berry 2"},
	rewards_chance = 3,
	spawn_min_level = 5,
	spawn_max_level = 20,
	spawn_on = {group.sand},
	spawn_by = {group.water},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.1, y = 1.1},
	spawn_collisionbox = {-0.35, -0.45, -0.35, 0.35, 0.1, 0.35},
	spawn_walk_velocity = 0.5,
	spawn_sink = true,
	spawn_burn = true,
}

monsters.stats.shelkern = {
	name = "Shelkern",
	asset_name = "shelkern",
	type = "Warrior",
	color = "#E3AE85FF",
	tier = 4,
	energy_cap = 8,
	health = 11,
	health_per_level = 2.2,
	attack = 16,
	attack_per_level = 2.4,
	defense = 17,
	defense_per_level = 2.9,
	agility = 10,
	agility_per_level = 1.9,
	exp_per_level = 97,
	level_up_moves = {[1] = "strike", [7] = "prickle", [13] = "grass_blade", [26] = "sword_swipe"},
	taught_moves = {"diced", "guard", "thorns"},
	rewards = {"zoonami:red_berry 2", "zoonami:tiger_lily 2", "zoonami:sanded_plank 5"},
	rewards_chance = 2,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.crumbly},
	spawn_by = {group.leaves},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.8, y = 1.8},
	spawn_collisionbox = {-0.4, -0.75, -0.4, 0.4, 0.2, 0.4},
	spawn_walk_velocity = 1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.shellephant = {
	name = "Shellephant",
	asset_name = "shellephant",
	type = "Aquatic",
	color = "#2697B5FF",
	tier = 4,
	energy_cap = 8,
	health = 16,
	health_per_level = 2.6,
	attack = 12,
	attack_per_level = 2.7,
	defense = 15,
	defense_per_level = 2.3,
	agility = 9,
	agility_per_level = 1.4,
	exp_per_level = 120,
	level_up_moves = {[1] = "bubble_stream", [6] = "smog", [9] = "poison", [15] = "slash", [24] = "vortex"},
	taught_moves = {"boil"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 3,
	spawn_min_level = 15,
	spawn_max_level = 25,
	spawn_on = {group.water},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.65, -0.9, -0.65, 0.65, 0.5, 0.65},
	spawn_walk_velocity = 1,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.skoakie = {
	name = "Skoakie",
	asset_name = "skoakie",
	type = "Rodent",
	color = "#EA5B33FF",
	tier = 4,
	energy_cap = 9,
	health = 12,
	health_per_level = 2.2,
	attack = 14,
	attack_per_level = 2.4,
	defense = 15,
	defense_per_level = 2.7,
	agility = 12,
	agility_per_level = 2.3,
	exp_per_level = 104,
	level_up_moves = {[1] = "swipe", [8] = "stomp", [14] = "claw", [19] = "slash", [26] = "razor_fang"},
	taught_moves = {"guard", "rage", "rest"},
	rewards = {"zoonami:orange_berry 2"},
	rewards_chance = 3,
	spawn_min_level = 20,
	spawn_max_level = 30,
	spawn_on = {group.leaves},
	spawn_by = {group.tree},
	spawn_chance = "uncommon",
	spawn_time = {"night"},
	spawn_light = {"medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.5, y = 1.5},
	spawn_collisionbox = {-0.45, -0.70, -0.45, 0.45, 0.2, 0.45},
	spawn_walk_velocity = 1,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.skygen = {
	name = "Skygen",
	asset_name = "skygen",
	type = "Avian",
	color = "#5FC8F7FF",
	tier = 4,
	energy_cap = 10,
	health = 13,
	health_per_level = 1.9,
	attack = 15,
	attack_per_level = 2.1,
	defense = 17,
	defense_per_level = 2.4,
	agility = 18,
	agility_per_level = 3.1,
	exp_per_level = 108,
	level_up_moves = {[1] = "gust", [4] = "tail_swipe", [12] = "fireball", [27] = "twister"},
	taught_moves = {"flash_fire", "rage", "restore"},
	rewards = {"zoonami:blue_berry 1", "zoonami:daisy 2", "zoonami:tiger_lily 2"},
	rewards_chance = 3,
	spawn_min_level = 30,
	spawn_max_level = 50,
	spawn_on = {group.snowy, group.stone},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"sky"},
	spawn_visual_size = {x = 1.1, y = 1.1},
	spawn_collisionbox = {-0.4, -0.85, -0.4, 0.4, 0.3, 0.4},
	spawn_walk_velocity = 1,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.spaero = {
	name = "Spaero",
	asset_name = "spaero",
	type = "Avian",
	color = "#D8A456FF",
	tier = 4,
	energy_cap = 8,
	health = 15,
	health_per_level = 2.6,
	attack = 19,
	attack_per_level = 2.7,
	defense = 12,
	defense_per_level = 2.1,
	agility = 15,
	agility_per_level = 2.9,
	exp_per_level = 110,
	level_up_moves = {[1] = "swoop", [15] = "swipe", [18] = "quill_drill", [26] = "twister"},
	taught_moves = {"guard", "whirlwind"},
	rewards = {"zoonami:blue_berry 1"},
	rewards_chance = 4,
	spawn_min_level = 15,
	spawn_max_level = 25,
	spawn_on = {group.grass},
	spawn_by = {group.tree},
	spawn_chance = "uncommon",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 2, y = 2},
	spawn_collisionbox = {-0.55, -0.85, -0.55, 0.55, 0.4, 0.55},
	spawn_walk_velocity = 1,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.splime = {
	name = "Splime",
	asset_name = "splime",
	type = "Mutant",
	color = "#30C856FF",
	tier = 5,
	energy_cap = 10,
	health = 17,
	health_per_level = 2.9,
	attack = 13,
	attack_per_level = 2.2,
	defense = 12,
	defense_per_level = 1.5,
	agility = 10,
	agility_per_level = 1.8,
	exp_per_level = 97,
	level_up_moves = {[1] = "acid_bath", [5] = "spore_storm", [14] = "shadow_orb", [19] = "life_drain", [25] = "nightmare"},
	taught_moves = {"rest", "restore"},
	rewards = {"zoonami:red_berry 1", "zoonami:green_berry 1", "zoonami:sunflower 2"},
	rewards_chance = 2,
	spawn_min_level = 15,
	spawn_max_level = 35,
	spawn_on = {group.stone},
	spawn_by = {group.water},
	spawn_chance = "common",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.7, y = 1.7},
	spawn_collisionbox = {-0.3, -0.7, -0.3, 0.3, 0.1, 0.3},
	spawn_walk_velocity = 0.5,
	spawn_float = true,
	spawn_burn = true,
}

monsters.stats.spyder = {
	name = "Spyder",
	asset_name = "spyder",
	type = "Robotic",
	color = "#ADADADFF",
	tier = 4,
	energy_cap = 7,
	health = 13,
	health_per_level = 2.2,
	attack = 13,
	attack_per_level = 2.4,
	defense = 14,
	defense_per_level = 2.5,
	agility = 14,
	agility_per_level = 2.7,
	exp_per_level = 100,
	level_up_moves = {[1] = "electrocute", [3] = "poison_sting", [14] = "life_drain", [23] = "laser_beam"},
	taught_moves = {"force_field", "pinpoint", "split"},
	rewards = {"zoonami:red_berry 1", "zoonami:crystal_glass 2"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 20,
	spawn_on = {group.grass, group.sand},
	spawn_chance = "common",
	spawn_time = {"night"},
	spawn_light = {"medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.25, y = 1.25},
	spawn_collisionbox = {-0.3, -0.55, -0.3, 0.3, 0.1, 0.3},
	spawn_walk_velocity = 0.5,
	spawn_drown = true,
	spawn_burn = true,
}

monsters.stats.squizzle = {
	name = "Squizzle",
	asset_name = "squizzle",
	type = "Fire",
	color = "#DF5239FF",
	tier = 4,
	energy_cap = 8,
	health = 13,
	health_per_level = 2.3,
	attack = 17,
	attack_per_level = 2.6,
	defense = 15,
	defense_per_level = 2.4,
	agility = 12,
	agility_per_level = 1.7,
	exp_per_level = 88,
	level_up_moves = {[1] = "afterburn", [9] = "acid_bath", [17] = "burnout", [25] = "man_melter"},
	taught_moves = {"boil", "inferno"},
	rewards = {"zoonami:red_berry 1", "zoonami:zeenite_lump 1"},
	rewards_chance = 4,
	spawn_min_level = 10,
	spawn_max_level = 25,
	spawn_on = {group.lava},
	spawn_chance = "uncommon",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"underground", "bedrock"},
	spawn_visual_size = {x = 1.1, y = 1.1},
	spawn_collisionbox = {-0.3, -0.46, -0.3, 0.3, 0.2, 0.3},
	spawn_walk_velocity = 0.7,
	spawn_drown = true,
	spawn_swim = true,
}

monsters.stats.tysumi = {
	name = "Tysumi",
	asset_name = "tysumi",
	type = "Aquatic",
	color = "#30BE92FF",
	tier = 1,
	energy_cap = 13,
	health = 12,
	health_per_level = 1.9,
	attack = 14,
	attack_per_level = 2.5,
	defense = 17,
	defense_per_level = 3.1,
	agility = 11,
	agility_per_level = 2.2,
	exp_per_level = 120,
	level_up_moves = {[1] = "high_tide", [7] = "rockburst", [9] = "vine_wrap", [15] = "grass_blade", [25] = "twister"},
	taught_moves = {"guard", "whirlwind"},
	rewards = {"zoonami:move_book_swirl 2", "zoonami:move_book_whirlwind 2", "zoonami:mystery_move_book 5"},
	rewards_chance = 1,
	spawn_min_level = 65,
	spawn_max_level = 75,
	spawn_on = {group.water},
	spawn_by = {group.tall_grass},
	spawn_chance = "legendary",
	spawn_time = {"day", "night"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 3.5, y = 3.5},
	spawn_collisionbox = {-1.2, -1.68, -1.2, 1.2, 0.65, 1.2},
	spawn_walk_velocity = 1,
	spawn_swim = true,
	spawn_burn = true,
}

monsters.stats.wigglar = {
	name = "Wigglar",
	asset_name = "wigglar",
	type = "Insect",
	color = "#D9AD47FF",
	tier = 5,
	energy_cap = 9,
	health = 15,
	health_per_level = 2.4,
	attack = 12,
	attack_per_level = 2.2,
	defense = 16,
	defense_per_level = 2.6,
	agility = 13,
	agility_per_level = 2.1,
	exp_per_level = 80,
	level_up_moves = {[1] = "pellet", [3] = "infestation", [13] = "life_drain", [21] = "bug_bite"},
	taught_moves = {"rest", "slice"},
	rewards = {"zoonami:orange_berry 1", "zoonami:sanded_plank 2"},
	rewards_chance = 3,
	spawn_min_level = 5,
	spawn_max_level = 15,
	spawn_on = {group.sand},
	spawn_chance = "common",
	spawn_time = {"day"},
	spawn_light = {"bright", "medium", "dark"},
	spawn_height = {"aboveground"},
	spawn_visual_size = {x = 1.3, y = 1.3},
	spawn_collisionbox = {-0.3, -0.55, -0.3, 0.3, 0.1, 0.3},
	spawn_walk_velocity = 0.7,
	spawn_drown = true,
	spawn_burn = true,
}

return monsters
