local S       = core.get_translator()
local modpath = core.get_modpath("tool_watcher")

tool_watcher  = {}
tool_watcher.gazed_players = {}

-- Get the durability left in percentage:
local function calculate_wear(toolstack)
    local current_wear = toolstack:get_wear()
    return 100 - (current_wear / 65535) * 100
end

-- Display message in the center of the screen:
tool_watcher.talking = function (player, message, t)
    if type(message) == 'string' then
        local hud_id = player:hud_add({
            hud_elem_type = "text",
            position    = {x = 0.5, y = 0.5},
            offset      = {x = 0, y = 0},
            text        = message,              -- There is some weird translation error from tool_watcher.scanning message so for now I'm forgone the S(message).
            alignment   = {x = 0, y = 0},       -- Center-aligned
            scale       = {x = 128, y = 128},
            size        = {x = 2, y = 2},       -- Size of the text, only size.x matters
            number      = 0xFF0000,             -- Red text
        })

        -- Remove the HUD text after t seconds:
        core.after(t, function()
            if player and player:hud_remove(hud_id)then
                player:hud_remove(hud_id)
            end
        end)
    
    elseif type(message) == 'table' then
        local hud_ids = {}
        local start_offset = -#message+0.5
        for i = 1, #message do
            hud_ids[i] = player:hud_add({
                hud_elem_type = "text",
                position    = {x = 0.5, y = 0.5},
                offset      = {x = 0, y = (start_offset+i)*32},
                text        = message[i],
                alignment   = {x = 0, y = 0},
                scale       = {x = 128, y = 128},
                size        = {x = 2, y = 2},
                number      = 0xFF0000,
            })

            -- Remove the HUD text after t seconds:
            core.after(t, function()
                if player and player:hud_remove(hud_ids[i]) then
                    player:hud_remove(hud_ids[i])
                end
            end)
        end
    end
end

-- Scan the wielded item of a given player:
tool_watcher.scanning = function(player)
    local target_stack = player:get_wielded_item()
    local target_tool  = target_stack:get_name()
    if core.registered_tools[target_tool] then
        local perc_left = calculate_wear(target_stack)
        local tool_name = target_stack:get_description()
        if perc_left < 6 then
            tool_watcher.talking(player, string.format("%s durability at %.2f%%", tool_name, perc_left), 3)
        end
        return perc_left
    end
    return false
end

-- Register Player for item check:
tool_watcher.pact = function(player)
    local name = player:get_player_name()
    tool_watcher.gazed_players[name] = {
        name = name,
        last_time_since_scanned = core.get_gametime(),
        scan_intervals = {
            normal  = tonumber(core.settings:get("check_normal")) or 2,
            relax   = tonumber(core.settings:get("check_relax"))  or 5,
            alert   = tonumber(core.settings:get("check_alert"))  or 1,
            current = tonumber(core.settings:get("check_normal")) or 2
        }
    }
end

-- Update connected players list on join:
-- When a player joins, check if their tool is already bound and re-register them:
core.register_on_joinplayer(function(player)
    local inv  = player:get_inventory()
    local size = inv:get_size("main") or 0
    if inv and size > 1 then
        -- Scan the main inventory for the tool watcher item:
        for i = 1, size do
            local stack = inv:get_stack("main", i)
            if stack:get_name() == "tool_watcher:monitoring_item" then
                local metadata = stack:get_meta()
                local is_bound = metadata:get_int("bound")
                if is_bound == 1 then
                    tool_watcher.pact(player)
                    tool_watcher.talking(player, 'The tool watcher pact holds.', 1)
                    break -- No need to keep checking once we find it.
                end
            end
        end
    end
end)

-- Remove player from connected list on leave:
core.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    tool_watcher.gazed_players[name] = nil
end)

-- Periodically check durability of players' tools:
core.register_globalstep(function(dtime)
    local now_time = core.get_gametime()
    for name, attributes in pairs(tool_watcher.gazed_players) do
        local player      = core.get_player_by_name(name)
        local last_update = now_time - attributes.last_time_since_scanned
        if player and last_update > attributes.scan_intervals.current then
            attributes.last_time_since_scanned = now_time
            local inv = player:get_inventory()
            if inv and inv:contains_item("main", "tool_watcher:monitoring_item") then
                local durability = tool_watcher.scanning(player)
                if durability then
                    if     durability < 10 then attributes.scan_intervals.current = attributes.scan_intervals.alert
                    elseif durability < 50 then attributes.scan_intervals.current = attributes.scan_intervals.normal
                    else                        attributes.scan_intervals.current = attributes.scan_intervals.relax
                    end
                end
            end
        end
    end
end)

-- Item to enable the tool watcher functionality:
core.register_craftitem(":tool_watcher:monitoring_item", {
    description = S("Tool Watcher"),
    inventory_image = "tool_watcher_item.png",
    stack_max = 1,
    on_use = function(itemstack, user, pointed_thing)
        if not user or not user:is_player() then return end
        local player_name = user:get_player_name()
        if tool_watcher.gazed_players[player_name] then
            tool_watcher.talking(user, 
                {'The pact is done,', 'just leave me in your inventory', 'so I can watch over your tools.'}, 2)
        else
            tool_watcher.talking(user, 
                {'Try clicking the floor with me,','then leave me in your inventory and','I will watch over your tools.'}, 2)
        end
    end,
    on_place = function(itemstack, user, pointed_thing)
        if not user or not user:is_player() then return end
        local player_name = user:get_player_name()
        if tool_watcher.gazed_players[player_name] then
            tool_watcher.talking(user, {'The pact is already sealed.','Just leave me in your inventory.'}, 2)
            
            local meta = itemstack:get_meta()
            local x    = meta:get_int("bound")
            core.chat_send_all(tostring(x))
        else
            -- Modify the itemstack so it is only needed to place the watcher the first time:
            local meta = itemstack:get_meta()
            meta:set_int("bound", 1)
            user:get_inventory():set_stack("main", user:get_wield_index(), itemstack)
            -- Adding to the globalstep table:
            tool_watcher.pact(user)
            -- Feedback to the player:
            tool_watcher.talking(user, 'The pact is sealed. Keep me in your inventory.', 2)
            -- Make the player take a symbolic punch with 0 damage:
            user:punch(user, 1.0, {full_punch_interval = 1.0, damage_groups = {fleshy = 1}}, nil)
            user:set_hp(user:get_hp() + 1)
        end
    end,
})

-- Recipe to make the item:
dofile( modpath .. '/recipes.lua')
