-- Overrides for the rice crop

-- Most Farming Redo crops do not have a separate seed item, but the rice
-- crop has the opposite problem: there is no separate crop item
-- X Farming has a crop item, so let's add it and adjust everything accordingly
-- Unlike in X Farming, rice is planted on soil, which is similar to dry-rice
-- cultivation

local S = core.get_translator(core.get_current_modname())

-- The seed item exists already, so just override parts of its definition
core.override_item("farming:seed_rice", {
    description = S("Rice Seeds"),
    tiles = {"x_farming_rice_seed_inv.png"},
    inventory_image = "x_farming_rice_seed_inv.png",
    wield_image = "x_farming_rice_seed_inv.png",
    groups = {compostability = 30, seed = 1, snappy = 3, attached_node = 1,
              flammable = 2, growing = 1},
})

-- Make 'farming:rice' the rice crop item instead of the bag of rice
core.override_item("farming:rice", {
    inventory_image = "x_farming_rice.png",
    -- The rice crop item is not intended to be used in recipes, so there is
    -- no 'food_rice' group
    groups = {compostability = 30, flammable = 2},
    on_place = function(itemstack, placer, pointed_thing)
        return nil
    end,
})

-- Reuse the rice item graphic from Farming Redo and make it a bag of rice
core.register_craftitem(":farming:rice_bag", {
    description = S("Bag of Rice"),
    inventory_image = "farming_rice.png",
    groups = {food_rice = 1, flammable = 2},
})

-- Clear the existing seed to rice bag cooking recipe
core.clear_craft({output = "farming:rice"})

core.register_craft({
    type = "cooking",
    output = "farming:rice_bag",
    recipe = "farming:rice",
    cooktime = 3,
})

-- Override the rice crop graphics
for i = 1, 8 do
    local x = tostring(i)
    core.override_item("farming:rice_" .. x, {
        paramtype2 = "none",
        place_param2 = 0,
        visual_scale = 1.5,  -- 3.0 like in X Farming is way too big
        tiles = {"x_farming_rice_" .. x .. ".png"},
    })
end

-- Change the drops to include the crop item and let stage 6 drop a seed item
-- For some reason, stages 5 and 6 drop the cooked rice item which does not
-- make any sense to me, so just remove those
core.override_item("farming:rice_5", {
    drop = {}
})

core.override_item("farming:rice_6", {
    drop = {
        items = {
            {rarity = 1, items = {"farming:seed_rice"}},
        }
    }
})

core.override_item("farming:rice_7", {
    drop = {
        items = {
            {rarity = 1, items = {"farming:seed_rice"}},
            {rarity = 1, items = {"farming:rice"}},
        }
    }
})

core.override_item("farming:rice_8", {
    drop = {
        items = {
            {rarity = 1, items = {"farming:seed_rice"}},
            {rarity = 2, items = {"farming:seed_rice"}},
            {rarity = 1, items = {"farming:rice"}},
            {rarity = 2, items = {"farming:rice"}},
        }
    }
})

core.register_decoration({
    deco_type = "simple",
    place_on = {
        "default:dirt_with_rainforest_litter", "mcl_core:dirt_with_grass",
        "ebiomes:dirt_with_forest_litter",
        "ebiomes:dirt_with_japanese_rainforest_litter",
        "ebiomes:dirt_with_grass_swamp", "ebiomes:peat_with_swamp_moss_yellow"
    },
    sidelen = 16,
    noise_params = {
        offset = 0,
        scale = 0.004,
        spread = {x = 100, y = 100, z = 100},
        seed = 1373,
        octaves = 3,
        persist = 0.7
    },
    -- Notes on the reasoning:
    -- * In yams, rice spawns in the rainforests and wetlands
    -- * In real-life, rice is a semi-aquatic plant and it is usually grown in
    -- paddy fields, which influenced the spawn conditions used in yams
    -- * Rice might not grow as much in the more temperate rainforests in
    -- real-life, but it does not look out of place in yams, so spawn rice in
    -- those biomes as well to give those environments another crop that players
    -- can find there
    -- * The swamp biome does not really look like a swamp in my opinion, but
    -- let rice spawn there anyway because it does not have many other reasons
    -- for players to visit it
    biomes = {
        "rainforest", "temperate_rainforest", "japanese_rainforest",
        "swamp", "bog"
    },
    y_min = 1,
    y_max = 64,
    decoration = "farming:rice_7"
})
