--initialize helper functions
local function valid_string(string_to_validate)
	return not (type(string_to_validate) ~= "string" or string_to_validate:find("[^%w_]") or string_to_validate == "")
end

local function split_cooldown_name(cooldown_name)
	local split_name = string.split(cooldown_name,":",true,2,false)
	local prefix = split_name[1]
	local suffix = split_name[2]
	if valid_string(prefix) and valid_string(suffix) then
		return split_name
	end
	return nil
end

local function get_cooldown(cooldown_name)
	if type(cooldown_name) ~= "string" then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." must be a string.")
	end
	local split_name = split_cooldown_name(cooldown_name)
	if split_name == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is not a valid cooldown_name.")
		return nil
	end
	if adjustable_cooldowns.cooldowns[split_name[1]] == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is part of an unloaded mod or a mod with no adjustable cooldowns.")
		return nil
	end
	if adjustable_cooldowns.cooldowns[split_name[1]][split_name[2]] == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is not defined.")
		return nil
	end
	return adjustable_cooldowns.cooldowns[split_name[1]][split_name[2]]
end

local function get_player_cooldown(cooldown_name,playername)
	if type(cooldown_name) ~= "string" then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." must be a string.")
	end
	local split_name = split_cooldown_name(cooldown_name)
	if split_name == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is not a valid cooldown_name.")
		return nil
	end
	if adjustable_cooldowns.cooldowns[split_name[1]] == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is part of an unloaded mod or a mod with no adjustable cooldowns.")
		return nil
	end
	if adjustable_cooldowns.cooldowns[split_name[1]][split_name[2]] == nil then
		minetest.log("error","cooldown_name: ".. tostring(cooldown_name) .." is not defined.")
		return nil
	end
	if type(playername) ~= "string" or adjustable_cooldowns.playerlist[playername] == nil then
		minetest.log("error","playername: "..tostring(playername).." is not defined. Cooldown call: "..cooldown_name)
	end
	return adjustable_cooldowns.playerlist[playername].cooldowns[split_cooldown_name[1]][split_cooldown_name[2]]
end

local function initalize_new_cooldown(cooldownname)
	local split_name = split_cooldown_name(cooldownname)
	local cooldown = get_cooldown(cooldownname)
	for playname,cooldown_info in pairs(adjustable_cooldowns.playerlist) do
		if cooldown_info.cooldowns[split_name[1]] == nil then
			cooldown_info.cooldowns[split_name[1]] = {}
		end
		if cooldown_info.cooldowns[split_name[1]][split_name[2]] == nil then
			cooldown_info.cooldowns[split_name[1]][split_name[2]] = {}
			cooldown_info.cooldowns[split_name[1]][split_name[2]].cooldown_rate = 1.0
			cooldown_info.cooldowns[split_name[1]][split_name[2]].current_cooldown = cooldown.respawn_cooldown
			cooldown_info.cooldowns[split_name[1]][split_name[2]].is_disabled = cooldown.disabled_on_default
		end
	end
end

local function initalize_new_group(groupname)
	for playname,cooldown_info in pairs(adjustable_cooldowns.playerlist) do
		if cooldown_info.groups[groupname] == nil then
			cooldown_info.groups[groupname] = 1.0
		end
	end
end

local function valid_cooldown_args(playername,cooldown_name)
	if type(playername) ~= "string" then
		minetest.log("error","Tried to call a function without a valid playername ("..tostring(playername)..", "..tostring(cooldown_name)..").")
		return false
	end
	if get_cooldown(cooldown_name) == nil then
		minetest.log("error","Tried to call a function without a valid cooldown_name ("..playername..", "..tostring(cooldown_name)..").")
		return false
	end
	return true
end


--define the functions
function adjustable_cooldowns.functions.init_cooldown(name,base_cooldown,on_finished_cooldown,respawn_cooldown,universal_modification,disabled_on_default)
	
	-- Enforce modname:cooldown_name naming convention
	local current_modname = minetest.get_current_modname()
	local expected_prefix = current_modname .. ":"
	if name:sub(1, #expected_prefix) ~= expected_prefix then
		error("Name " .. name .. " does not follow naming conventions: " ..
				"\"" .. expected_prefix .. "\" required")
	end
	-- Enforce that the name only contains letters, numbers and underscores.
	local subname = name:sub(#expected_prefix+1)
	if subname:find("[^%w_]") then
		error("Name " .. name .. " does not follow naming conventions: " ..
			"contains unallowed characters")
	end
	if adjustable_cooldowns.cooldowns[current_modname] == nil then
		adjustable_cooldowns.cooldowns[current_modname] = {}
	end
	--Enforce only one cooldown with the same cooldown name
	if adjustable_cooldowns.cooldowns[current_modname][subname] ~= nil then
		error("Name: ".. name .. " is already in use, do not initalize two or more cooldowns with the same cooldown name.")
	end
	--Enforce base cooldown is a number greater than 0
	if type(base_cooldown) ~= "number" or base_cooldown <= 0 then
		error("Name: " .. name .. "has an invalid base cooldown. The base cooldown must be a number and greater than 0.")
	end
	--Enforce on_finished_cooldown is a function
	if type(on_finished_cooldown) ~= "function" then
		error("Name: ".. name .. "has an invalid on_finished_cooldown. on_finished_cooldown must be a function.")
	end
	--initalize default values if not given a valid arugument
	if type(respawn_cooldown) ~= "number" then
		respawn_cooldown = base_cooldown
	end
	if respawn_cooldown < 0 then
		minetest.log("warning", "respawn_cooldown must be greater than or equal to zero. Offending cooldown name: "..name)
		respawn_cooldown = 0
	end
	if respawn_cooldown > base_cooldown then
		minetest.log("warning","respawn_cooldown must be less than or equal to base_cooldown. Offending cooldown name: "..name)
		respawn_cooldown = base_cooldown
	end
	if type(universal_modification) ~= "boolean" then
		universal_modification = true
	end
	if type(disabled_on_default) ~= "boolean" then
		disabled_on_default = false
	end
	
	--initalize the cooldown
	adjustable_cooldowns.cooldowns[current_modname][subname] = {}
	adjustable_cooldowns.cooldowns[current_modname][subname].base_cooldown = base_cooldown
	adjustable_cooldowns.cooldowns[current_modname][subname].on_finished_cooldown = on_finished_cooldown
	adjustable_cooldowns.cooldowns[current_modname][subname].respawn_cooldown = respawn_cooldown
	adjustable_cooldowns.cooldowns[current_modname][subname].universal_modification = universal_modification
	adjustable_cooldowns.cooldowns[current_modname][subname].disabled_on_default = disabled_on_default
	adjustable_cooldowns.cooldowns[current_modname][subname].groups = {}

	initalize_new_cooldown(name)
end

function adjustable_cooldowns.functions.init_group(groupname)
	if groupname:find("[^%w_]") ~= nil then
		error("groupname " .. groupname .. " does not follow naming conventions: contains unallowed characters: ("..string.sub(groupname,groupname:find("[^%w_]"),groupname:find("[^%w_]"))..")")
	end
	if adjustable_cooldowns.groups[groupname] ~= nil then
		minetest.log("warning","Tried to initialize an already initalized group ("..groupname..").")
	end
	adjustable_cooldowns.groups[groupname] = {} --will be a list of supergroups (groups it is a subgroup of)
	initalize_new_group(groupname)
end

function adjustable_cooldowns.functions.add_group(name_to_add,group)
	if type(name_to_add) ~= "string" or type(group) ~= "string" then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.add_group without a valid group ("..tostring(name_to_add)..", "..tostring(group)..").")
		return false
	end
	do
		local first_group = (name_to_add:find("[^%w_]") == nil)
		local second_group = (group:find("[^%w_]") == nil)
		if not (first_group or second_group) then
			minetest.log("error", "Tried to call adjustable_cooldowns.functions.add_group without a valid group ("..name_to_add..", "..group..").")
			return false
		end
		if first_group and adjustable_cooldowns.groups[name_to_add] == nil then
			minetest.log("warning","Tried to call an uninitalized group ("..name_to_add..").")
			adjustable_cooldowns.functions.init_group(name_to_add)
		end
		if second_group and adjustable_cooldowns.groups[group] == nil then
			minetest.log("warning","Tried to call an uninitalized group ("..group..").")
			adjustable_cooldowns.functions.init_group(group)
		end
		if first_group and not second_group then
			minetest.log("warning","Please use name_to_add to have the cooldownname and group to have the group name. ("..name_to_add..", "..group..")")
			local temp = name_to_add
			name_to_add = group
			group = temp
		elseif first_group and second_group then
			for index, supergroup in pairs(adjustable_cooldowns.groups[group]) do
				if supergroup == name_to_add then
					return true
				end
			end
			table.insert(adjustable_cooldowns.groups[name_to_add],group)
			return true
		end
	end
	if name_to_add:find("[^%w_:]") or name_to_add:find("[^%w_]",name_to_add:find(":")+1) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.add_group without a valid group or cooldownname ("..name_to_add..").")
		return false
	end
	local cooldown = get_cooldown(name_to_add)
	if cooldown == nil then
		return false
	end
	table.insert(cooldown.groups,group)
	return true
end

function adjustable_cooldowns.functions.multiply_global_cooldown(playername,cooldown_rate)
	if type(cooldown_rate) ~= "number" or cooldown_rate <= 0 then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_global_cooldown without a valid cooldown_rate ("..tostring(cooldown_rate)..", "..tostring(playername)..").")
		return false
	end
	if type(playername) ~= "string" then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_global_cooldown without a valid playername ("..tostring(cooldown_rate)..", "..tostring(playername)..").")
		return false
	end
	local player_cooldown_info = adjustable_cooldowns.playerlist[playername]
	if player_cooldown_info == nil then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_global_cooldown without a valid playername ("..tostring(cooldown_rate)..", "..playername..").")
		return false
	end
	local cooldown_inverse = 1.0 / cooldown_rate
	player_cooldown_info.cooldown_rate = player_cooldown_info.cooldown_rate * cooldown_inverse
	return true
end

function adjustable_cooldowns.functions.multiply_individual_cooldown(playername,value_name,cooldown_rate)
	if type(value_name) ~= "string" then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_individual_cooldown without a valid value_name ("..tostring(playername)..", "..tostring(value_name)..", "..tostring(cooldown_rate)..").")
		return false
	end
	if type(playername) ~= "string" then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_individual_cooldown without a valid playername ("..tostring(playername)..", "..value_name..", "..tostring(cooldown_rate)..").")
		return false
	end
	if type(cooldown_rate) ~= "number" or cooldown_rate <= 0 then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_individual_cooldown without a valid cooldown_rate ("..playername..", "..value_name..", "..tostring(cooldown_rate)..").")
		return false
	end
	local player_info = adjustable_cooldowns.playerlist[playername]
	if player_info == nil then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_individual_cooldown without a valid playername ("..playername..", "..value_name..", "..tostring(cooldown_rate)..").")
		return false
	end
	local inverse_rate = 1/cooldown_rate
	if value_name:find("[^%w_]") ~= nil then
		if adjustable_cooldowns.groups[value_name] == nil then
			minetest.log("error","Tried to call an uninitalized group ("..value_name..") in adjustable_cooldowns.functions.multiply_individual_cooldown("..playername..", "..value_name..", "..tostring(cooldown_rate)..").")
			return false
		end
		adjustable_cooldowns.groups[value_name] = adjustable_cooldowns.groups[value_name] * inverse_rate
		return true
	end
	local split_name = split_cooldown_name(value_name)
	if split_name == nil or get_cooldown(value_name) == nil then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.multiply_individual_cooldown without a valid value_name ("..playername..", "..value_name..", "..tostring(cooldown_rate)..").")
		return false
	end
	player_info.cooldowns[split_name[1]][player_info[2]].cooldown_rate = player_info.cooldowns[split_name[1]][player_info[2]].cooldown_rate * inverse_rate
	return true
end

function adjustable_cooldowns.functions.reset_cooldown(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.reset_cooldown without a valid argument(s)")
		return false
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	local cooldown = adjustable_cooldowns.cooldowns[split_name[1]][split_name[2]]
	player_info.cooldowns[split_name[1]][split_name[2]].current_cooldown = cooldown.base_cooldown
	return true
end

function adjustable_cooldowns.functions.activate_cooldown(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.activate_cooldown without a valid argument(s)")
		return false
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	player_info.cooldowns[split_name[1]][split_name[2]].current_cooldown = 0
	return true
end

function adjustable_cooldowns.functions.disable_cooldown(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.disable_cooldown without a valid argument(s)")
		return false
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	player_info.cooldowns[split_name[1]][split_name[2]].is_disabled = true
	return true
end

function adjustable_cooldowns.functions.enable_cooldown(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.enable_cooldown without a valid argument(s)")
		return false
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	player_info.cooldowns[split_name[1]][split_name[2]].is_disabled = false
	return true
end

function adjustable_cooldowns.functions.is_disabled(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.is_disabled without a valid argument(s)")
		return true
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	return player_info.cooldowns[split_name[1]][split_name[2]].is_disabled
end

function adjustable_cooldowns.functions.remaining_cooldown(playername,cooldown_name)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.remaining_cooldown without a valid argument(s)")
		return nil
	end
	local split_name = split_cooldown_name(cooldown_name)
	local player_info = adjustable_cooldowns.playerlist[playername]
	return player_info.cooldowns[split_name[1]][split_name[2]].current_cooldown
end

function adjustable_cooldowns.functions.add_cooldown(playername,cooldown_name,seconds_to_add)
	if not valid_cooldown_args(playername,cooldown_name) then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.remaining_cooldown without a valid argument(s)")
		return false
	end
	if type(seconds_to_add) ~= "number" then
		minetest.log("error","Tried to call adjustable_cooldowns.functions.remaining_cooldown without a valid seconds_to_add")
		return false
	end
	local split_name = split_cooldown_name(cooldown_name)
	local max_cooldown = get_cooldown(cooldown_name).base_cooldown
	local player_info = adjustable_cooldowns.playerlist[playername]
	player_info.cooldowns[split_name[1]][split_name[2]].current_cooldown = math.min(player_info.cooldowns[split_name[1]][split_name[2]].current_cooldown + seconds_to_add, max_cooldown)
	return true
end