local S = core.get_translator("squill")
local gui = flow.widgets
local dump_value = squill.dump_value

local get_db_names = squill.prepare_statement("squill", [[
    SELECT DISTINCT db_name FROM schema
    -- Always sort "squill" database last
    ORDER BY db_name = 'test' DESC, db_name = 'squill', db_name
]], squill.RETURN_SINGLE_COLUMN)

local colors = {
    string = "lightgreen",
    number = "orange",
    boolean = "lightblue",
    ["nil"] = "grey",
}

local function make_table(ctx)
    local t = ctx.output_table
    if #t == 0 then
        return gui.Label{
            label = t.affected_rows and
                S("@1 row(s) affected", t.affected_rows) or S("No results"),
            expand = true, align_v = "top"
        }
    end

    local cols = ctx.column_order
    if #cols == 0 then
        cols[1] = "?" -- Prevent client crash in case of a bug
    end

    local tablecolumns = {}
    local cells = {}
    for _, col in ipairs(cols) do
        cells[#cells + 1] = "cyan"
        cells[#cells + 1] = col
        tablecolumns[#tablecolumns + 1] = {type = "color", opts = {}}
        local opts = {}
        for _, row in ipairs(t) do
            if row[col] ~= nil then
                if type(row[col]) == "number" then
                    opts.align = "right"
                end
                break
            end
        end
        tablecolumns[#tablecolumns + 1] = {type = "text", opts = opts}
    end

    for _, row in ipairs(t) do
        for _, col in ipairs(cols) do
            local value = row[col]
            cells[#cells + 1] = assert(colors[type(value)])
            if type(value) ~= "string" then
                value = dump_value(value)
            end
            cells[#cells + 1] = value
        end
    end

    return gui.Stack{
        expand = true,
        gui.TableColumns{tablecolumns = tablecolumns},
        gui.Table{w = 10, h = 2, name = "table", cells = cells},
    }
end

local admin_console = flow.make_gui(function(player, ctx)
    local items = get_db_names()
    if items[1] ~= "test" then
        table.insert(items, 1, "test")
    end
    if ctx.form.db and table.indexof(items, ctx.form.db) < 0 then
        table.insert(items, 1, ctx.form.db)
    end

    local win_info = core.get_player_window_information and
        core.get_player_window_information(player:get_player_name())
    local max_formspec_size = win_info and win_info.max_formspec_size
    return gui.VBox{
        min_w = max_formspec_size and max_formspec_size.x - 0.6 or 20,
        min_h = max_formspec_size and max_formspec_size.y - 0.6,
        window_padding = max_formspec_size and {x = 0, y = 0} or nil,

        gui.HBox{
            gui.Label{label = S("Squill debug console")},
            gui.Spacer{},
            gui.Label{label = S("Database:")},
            ctx.edit_db_name and gui.Field{
                name = "db", min_w = 3,
            } or gui.Dropdown{
                name = "db", min_w = 3,
                items = items,
                index_event = false,
            },
            gui.Button{
                w = 1,
                label = ctx.edit_db_name and S("OK") or S("New"),
                on_event = function()
                    ctx.edit_db_name = not ctx.edit_db_name
                    return true
                end,
            },
        },
        gui.VBox{
            min_h = 5,
            ctx.form.db == "squill" and gui.Label{
                w = max_formspec_size and 10 or 0,
                label = S("Warning: This is an internal database and " ..
                    "is read-only to avoid accidental corruption.")
            } or gui.Nil{},
            gui.Textarea{
                name = "code",
                label = S("SQL code:"),
                w = 10, h = 3, expand = true,
                style = {font = "mono"},
            },
        },
        gui.HBox{
            gui.Button{
                label = S("Run"), expand = true,
                on_event = function()
                    ctx.output = nil
                    ctx.output_table = nil
                    ctx.column_order = nil

                    local ok, err = xpcall(function()
                        local all_rows, all_cols =
                            squill.exec(ctx.form.db, ctx.form.code)

                        local rows = {}
                        -- Combine all statements together
                        for _, row in ipairs(all_rows) do
                            if row.affected_rows then
                                rows.affected_rows =
                                    (rows.affected_rows or 0) + 1
                            end
                            table.insert_all(rows, row)
                        end

                        -- Get a list of all columns
                        local order = {}
                        local seen = {}
                        for _, returned_columns in ipairs(all_cols) do
                            for _, col in ipairs(returned_columns) do
                                if not seen[col] then
                                    seen[col] = true
                                    order[#order + 1] = col
                                end
                            end
                        end

                        ctx.output_table = rows
                        ctx.column_order = order
                    end, debug.traceback)
                    if not ok then
                        ctx.output = "Error:\n" .. tostring(err)
                    end

                    -- Always do a rollback to avoid crashing
                    ok = pcall(squill.exec, ctx.form.db, "ROLLBACK")
                    if ok then
                        ctx.output = (
                            "The current transaction has been rolled back " ..
                            "as it was not completed." ..
                            (ctx.output and "\n\n" .. ctx.output or "")
                        )
                    end

                    ctx.form.table = 0
                    return true
                end,
            },
            gui.Button{
                label = S("Dump database"),
                on_event = function()
                    local rope = {
                        write = function(self, s) self[#self + 1] = s end
                    }
                    squill.dump(ctx.form.db, rope)
                    ctx.output = table.concat(rope)
                    ctx.output_table = nil
                    return true
                end,
            },
        },
        gui.VBox{
            spacing = 0, min_h = 3, expand = true,
            gui.Label{label = S("Output:")},
            (ctx.output or not ctx.output_table) and gui.Textarea{
                w = 10, h = 0.5, default = (ctx.output or S("No output")),
                expand = not ctx.output_table
            } or gui.Nil{},
            ctx.output_table and make_table(ctx) or gui.Nil{},
        },
    }
end)

core.register_chatcommand("squill", {
    privs = {server = true},
    func = function(name)
        local player = core.get_player_by_name(name)
        if not player then return end
        admin_console:show(player)
    end,
})
