# Squill

A subset of SQL for Luanti (formerly known as Minetest). This document assumes
that you are already familiar with SQL.

## Goals

 - No additional installation steps (mod security, external libraries, etc)
 - Reasonable performance for small-ish databases
 - The ability to use both Squill (for singleplayer and smaller servers) and
   SQLite (for large servers which go out of their way to set it up) without
   too much special code.
 - No hard limit on the amount of data stored (unlike `core.serialize()`),
   although storing a lot of data will be slow.
 - Faster startup time than running `core.parse_json()` on a giant file.

## Non-goals

 - Implement every SQL feature
 - Scalability - if you have very large tables, expect high memory usage and
   low performance.

## SQL support

Squill implements a small subset of SQL. I have tried to follow SQLite's syntax
where practical.

The following value types are supported:

 - `real`: Numbers
 - `integer`, `int`: Integers. Attempting to store floating-point values will
   cause an error.
 - `text`: Short strings. Not suitable for storing binary data.
 - `bool`, `boolean`: Booleans
 - `blob`: Blobs (strings) that can store binary data. Suitable for large
   amounts of data, but probably slow if used in a `WHERE` condition.

All tables are roughly the same as SQLite's strict tables - inserting the wrong
type into a column (if it can't be converted through type coercion) will result
in an error.

Notably, the following features are not implemented:

 - `IN`
 - Recursive queries
 - Indexes
 - Foreign keys across multiple columns
 - Left, right, and full joins
 - `ON DELETE`/`ON UPDATE`
 - `DEFAULT`
 - `UNION`
 - Storing multiple types in one column (like SQLite) - you can use the `blob`
   column type and [bluon], `core.write_json`, or `core.serialize` instead.
 - `SELECT 1 test WHERE test = 1` (PostgreSQL doesn't seem to support this
   either, but it works in SQLite)
 - Lots more

[bluon]: https://github.com/appgurueu/modlib/blob/master/doc/bluon.md#simple-example

### Transactions

Transactions are supported through the `BEGIN TRANSACTION`, `COMMIT`, and
`ROLLBACK` SQL commands. Only one transaction can happen at once in each
database.

Make sure you remember to `COMMIT` or `ROLLBACK`. To prevent accidental data
loss, Squill will crash the server if you forget.

A rollback is automatically done if there is a runtime error in a statement
(such as a constraint violation) to avoid leaving the database in an
inconsistent state. Note that parsing errors do not automatically roll back
transactions.

Statements that modify the schema can't be run in transactions.

## GUI

You must set `squill.enable_gui = true` in minetest.conf to enable the GUI.

You can try running queries on databases with the `/squill` command if you have
[flow](https://content.minetest.net/packages/luk3yx/flow) installed.

## Database names

Database names must either be in the form `modname:dbname` or `modname` to
avoid conflicts. Squill has its own internal database called `squill` that
should not be used (its format may change in the future).

## Database API wrappers

If you have a mod that's already using a real SQL database and want to add a
fallback to Squill if your mod can't access the insecure environment, or if you
just like one of these APIs better, you can use one of the below API wrappers.

### [lsqlite3](http://lua.sqlite.org)

To make it easier to use both Squill and lsqlite3 with the same code, only the
file name is used out of the database path if the argument looks like a path.
For example, `sqlite3.open("/path/to/mymod.sqlite")` will open the database
`"mymod"`.

Any boolean values that get returned are converted to integers to match
SQLite's behaviour.

Example code that tries to use lsqlite3 and falls back to Squill[^1]:

```lua
local ie = core.request_insecure_environment()
local sqlite
if ie then
    -- The mod is in secure.trusted_mods, use lsqlite3
    sqlite = ie.require("lsqlite3")

    -- Don't leak the sqlite3 global variable
    sqlite3 = nil
elseif core.global_exists("squill") then
    -- The mod is not in secure.trusted_mods, fall back to using Squill
    sqlite = squill.compat.lsqlite3
else
    error("Please add this mod to secure.trusted_mods or install Squill.")
end

-- (Mod code here)
local db = sqlite.open(core.get_worldpath() .. "/mymod.sqlite")

local function check(retval)
    if retval ~= sqlite.OK then
        error(db:errmsg(), 2)
    end
end

check(db:exec([[
    CREATE TABLE IF NOT EXISTS my_table (
        a INTEGER PRIMARY KEY,
        b TEXT,
        c TEXT
    ) STRICT;

    INSERT INTO my_table (b, c) VALUES ('Hello world!', 'test');
]]))

local stmt = db:prepare("SELECT a, b FROM my_table where c = ?")
if not stmt then
    error(db:errmsg())
end
check(stmt:bind_values("test"))

for a, b in stmt:urows() do
    print(a, b)
end

check(stmt:finalize())

db:close()
```

Some more advanced functions from lsqlite3 aren't implemented, such as hooks.

Note that this API wrapper won't work with anything that depends on SQLite's
ability to store values of multiple types in columns.


[^1]: You will need to add `squill` to `optional_depends` in your mod.conf file
      (or maybe just to `depends` if you want to make ContentDB installs easier
      and don't mind requiring it even if it isn't being used). The code is
      adapted from the lsqlite3 example on the [Luanti Modding Book].

[Luanti Modding Book]: https://rubenwardy.com/minetest_modding_book/en/map/storage.html

### [pgmoon](https://github.com/leafo/pgmoon)

Example (loading pgmoon itself is not covered here):

```lua
local pgmoon = squill.compat.pgmoon

-- Unsupported options such as "host", "port", and "username" are ignored
local pg = pgmoon.connect({database = "mymod"})
assert(pg:connect())

local rows = assert(pg:query("SELECT a, b FROM my_table WHERE c = $1", "test"))
print(dump(rows))

assert(pg:disconnect())
```

Note that you will probably need Squill-specific code for `CREATE TABLE` to use
Squill's types.

## Squill API

The below API functions take a database name as a parameter.

### `squill.exec(db_name, sql)`

You can use `squill.exec` to run one or more SQL statements.

```lua
-- You can use squill.exec for debugging and to create tables
squill.exec("mymod:mydb", [[
    CREATE TABLE IF NOT EXISTS my_table (
        key text,
        other_key text
    );
    INSERT INTO my_table (key, other_key) VALUES ('hello', 'world');
]])
```

You can pass in parameters with `?` or `$1`, `$2`, etc:

```lua
squill.exec("mymod:mydb", "SELECT $1 * $2", 5, 2)
```

Single statements will be cached for future calls, and removed from the cache
if they're not used for a while. Note that only single statements are cached
at the moment, so you should try and avoid using multiple statements with
`squill.exec` in performance-sensitive code.

### `squill.prepare_statement(db_name, sql[, return_mode])`

`squill.prepare_statement` compiles the SQL statement and returns a function
that you can call to run the statement.

Creating or altering the database schema (e.g. `CREATE TABLE`) will invalidate
any existing prepared statements, make sure any schema changes are done at load
time.

Example:

```lua
local get_keys_except = squill.prepare_statement("mymod:mydb", [[
    SELECT key, other_key FROM my_table WHERE key <> ?
]])

for _, row in ipairs(get_keys_except("hello")) do
    print(row.key, row.other_key)
end
```

Prepared statements are cached in case `squill.prepare_statement` or
`squill.exec` is called with exactly the same string.

There's a second return value which indicates what columns the SQL statement
will return, and is something like `{"col1", "col2"}` or
`{affected_rows = true}`. Usually this value should just get ignored, as done
in the example. You must not modify the second return value as it is cached.

`return_mode` adjusts what the function will return, and can be one of:

 - `squill.RETURN_ALL_ROWS` (default)
    - Returns all rows.
    - For example, `SELECT * FROM my_table ORDER BY key` would return
      `{{key = "key1", other_key = "data"}, {key = "key2", other_key = "more_data"}}`.
    - `UPDATE` and `DELETE` statements will return `{affected_rows = rows}`.

 - `squill.RETURN_FIRST_ROW`
    - Returns the first row, for example `{key = "key1", other_key = "data"}`.
    - Returns nil if no rows are returned by the query.

 - `squill.RETURN_SINGLE_COLUMN`
    - Returns the first column, for example
      `SELECT key FROM my_table ORDER BY key` would return `{"key1", "key2"}`.
    - Errors if the statement does not return exactly one column.

 - `squill.RETURN_SINGLE_VALUE`
    - Returns a single value from the first row, for example
      `SELECT key FROM my_table ORDER BY key` would return `"key1"`.
    - Returns `nil` if no rows are returned.
    - Errors if the statement does not return exactly one column.

## Making SQL backups of a database

You can use `squill.dump(db_name, file_object)` to export a backup of a
database. This backup can be imported on a new server with `squill.exec` (or
maybe it could be used to migrate to SQLite). `file_object` can be anything
that supports `:write()`, for example a [modlib.table.rope] object.

You should probably either do this or back up your mod storage database
regularly in case some bug corrupts data.

[modlib.table.rope]: https://github.com/appgurueu/modlib/blob/f6de802d6f50cab6eee1f0d760a7f2f737d81a42/table.lua#L84-L95

## Performance tips

 - Use `squill.prepare_statement` (or the equivalent in the database API
   wrapper you are using) and compile every statement you need at load time.
    - If you don't prepare statements ahead of time, Squill will try and do
      some caching, but queries not run at least once per minute will be
      re-parsed each time.
 - Use LuaJIT.
 - Use query parameters instead of escaping where possible, even if you're not
   preparing statements ahead of time.
 - SELECTs/JOINs on non-null values in singular `UNIQUE`/`PRIMARY KEY` columns
   are fast since they can just do a O(1) lookup if a cached index is in memory
    - Note: This currently only works if `column = value` or `value = column`
      is the only expression in the where/join clause and `column` is in the
      table specified by the join clause.
 - Transactions can speed up bulk inserts and updates, as Squill will only
   write everything to mod storage after `COMMIT`, instead of after each
   statement.
 - Avoid `SELECT *` and only request the columns that you actually need.
 - Use `blob` for large data so that it doesn't get loaded into RAM all the
   time.
 - You can enable the `squill.aggressive_caching` setting to make Squill keep
   caches in memory for longer. I'm not sure how much this will improve
   performance by.
